/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */


package com.klikli_dev.modonomicon.api.datagen;

import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.api.datagen.book.BookCategoryModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookCommandModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookEntryModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookModel;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_7225;
import net.minecraft.class_7403;
import net.minecraft.class_7784;

/**
 * The main provider class for book datagen. Implement a subprovider and hand it over to this provider to generate the files!
 */
public class BookProvider implements class_2405 {

    protected final String modId;
    protected final CompletableFuture<class_7225.class_7874> registries;

    protected final class_7784 packOutput;
    //This is a bit of a relic, one provider is only supposed to generate one book.
    protected final Map<class_2960, BookModel> bookModels;
    protected final List<BookSubProvider> subProviders;


    public BookProvider(class_7784 packOutput, CompletableFuture<class_7225.class_7874> registries, String modId,
                        List<BookSubProvider> subProviders) {
        this.packOutput = packOutput;
        this.registries = registries;
        this.modId = modId;
        this.subProviders = subProviders;
        this.bookModels = new Object2ObjectOpenHashMap<>();
    }

    public String modId() {
        return this.modId;
    }

    protected Path getPath(Path dataFolder, BookModel bookModel) {
        class_2960 id = bookModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(id.method_12832() + "/book.json");
    }

    protected Path getPath(Path dataFolder, BookCategoryModel bookCategoryModel) {
        class_2960 id = bookCategoryModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookCategoryModel.getBook().getId().method_12832())
                .resolve("categories")
                .resolve(id.method_12832() + ".json");
    }

    protected Path getPath(Path dataFolder, BookCommandModel bookCommandModel) {
        class_2960 id = bookCommandModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookCommandModel.getBook().getId().method_12832())
                .resolve("commands")
                .resolve(id.method_12832() + ".json");
    }

    protected Path getPath(Path dataFolder, BookEntryModel bookEntryModel) {
        class_2960 id = bookEntryModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookEntryModel.getCategory().getBook().getId().method_12832())
                .resolve("entries")
                .resolve(id.method_12832() + ".json");
    }

    @Override
    public @NotNull CompletableFuture<?> method_10319(@NotNull class_7403 cache) {
        return this.registries.thenCompose(registries -> {
            List<CompletableFuture<?>> futures = new ArrayList<>();

            Path dataFolder = this.packOutput.method_45972(class_7784.class_7490.field_39367);

            this.subProviders.forEach(subProvider -> subProvider.generate(this.bookModels::put, registries));

            for (var bookModel : this.bookModels.values()) {
                Path bookPath = this.getPath(dataFolder, bookModel);

                if(!bookModel.dontGenerateJson()){ //a model from AddToBookSubProvider
                    futures.add(class_2405.method_10320(cache, bookModel.toJson(registries), bookPath));
                }


                for (var bookCategoryModel : bookModel.getCategories()) {
                    Path bookCategoryPath = this.getPath(dataFolder, bookCategoryModel);

                    if(!bookCategoryModel.dontGenerateJson()){ //a model from AddToCategorySubProvider
                        futures.add(class_2405.method_10320(cache, bookCategoryModel.toJson(registries), bookCategoryPath));
                    }

                    for (var bookEntryModel : bookCategoryModel.getEntries()) {
                        Path bookEntryPath = this.getPath(dataFolder, bookEntryModel);
                        futures.add(class_2405.method_10320(cache, bookEntryModel.toJson(registries), bookEntryPath));
                    }
                }

                for (var bookCommandModel : bookModel.getCommands()) {
                    Path bookCommandPath = this.getPath(dataFolder, bookCommandModel);
                    futures.add(class_2405.method_10320(cache, bookCommandModel.toJson(registries), bookCommandPath));
                }
            }

            return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
        });
    }

    @Override
    public @NotNull String method_10321() {
        return "Books: " + this.modId();
    }

    protected BookModel add(BookModel bookModel) {
        if (this.bookModels.containsKey(bookModel.getId()))
            throw new IllegalStateException("Duplicate book " + bookModel.getId());
        this.bookModels.put(bookModel.getId(), bookModel);
        return bookModel;
    }
}
