/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book;

import F;
import I;
import Z;
import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.Data.Category;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.conditions.BookNoneCondition;
import com.klikli_dev.modonomicon.book.entries.BookEntry;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.util.BookGsonHelper;
import it.unimi.dsi.fastutil.objects.Object2ObjectMaps;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7225;
import net.minecraft.class_9129;

public class BookCategory {

    protected class_2960 id;
    protected Book book;
    protected String name;
    protected BookIcon icon;
    protected BookTextHolder description;
    /**
     * The display mode - node based (thaumonomicon style) or index based (lexica botania / patchouli style)
     */
    protected BookDisplayMode displayMode;
    protected int sortNumber;
    protected class_2960 background;
    protected int backgroundWidth;
    protected int backgroundHeight;
    protected int maxScrollX;
    protected int maxScrollY;
    /**
     * Allows to modify how "zoomed in" the background texture is rendered.
     * A lower value means the texture is zoomed OUT more -> it is sharper / less blurry.
     */
    protected float backgroundTextureZoomMultiplier;
    protected List<BookCategoryBackgroundParallaxLayer> backgroundParallaxLayers;
    protected class_2960 entryTextures;
    protected Map<class_2960, BookEntry> entries;
    protected BookCondition condition;
    protected boolean showCategoryButton;
    /**
     * The entry to open when this category is opened.
     * If null, no entry will be opened.
     */
    protected class_2960 entryToOpen;
    /**
     * If true, the entryToOpen will only be opened the first time the category is opened.
     * If false, the entryToOpen will be opened every time the category is opened.
     */
    protected boolean openEntryToOpenOnlyOnce;

    public BookCategory(class_2960 id, String name, BookTextHolder description, int sortNumber, BookCondition condition, boolean showCategoryButton, BookIcon icon, BookDisplayMode displayMode, class_2960 background, int backgroundWidth, int backgroundHeight, int maxScrollX, int maxScrollY, float backgroundTextureZoomMultiplier, List<BookCategoryBackgroundParallaxLayer> backgroundParallaxLayers, class_2960 entryTextures, class_2960 entryToOpen, boolean openEntryOnlyOnce) {
        this.id = id;
        this.name = name;
        this.description = description;
        this.sortNumber = sortNumber;
        this.condition = condition;
        this.showCategoryButton = showCategoryButton;
        this.icon = icon;
        this.displayMode = displayMode;
        this.background = background;
        this.backgroundWidth = backgroundWidth;
        this.backgroundHeight = backgroundHeight;
        this.maxScrollX = maxScrollX;
        this.maxScrollY = maxScrollY;
        this.backgroundTextureZoomMultiplier = backgroundTextureZoomMultiplier;
        this.backgroundParallaxLayers = backgroundParallaxLayers;
        this.entryTextures = entryTextures;
        this.entries = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
        this.entryToOpen = entryToOpen;
        this.openEntryToOpenOnlyOnce = openEntryOnlyOnce;
    }

    public static BookCategory fromJson(class_2960 id, JsonObject json, class_7225.class_7874 provider) {
        var name = class_3518.method_15265(json, "name");
        var description = BookGsonHelper.getAsBookTextHolder(json, "description", BookTextHolder.EMPTY, provider);
        var sortNumber = class_3518.method_15282(json, "sort_number", -1);
        var icon = BookIcon.fromJson(json.get("icon"), provider);
        var displayMode = BookDisplayMode.byName(class_3518.method_15253(json, "display_mode", BookDisplayMode.NODE.method_15434()));
        var background = class_2960.method_60654(class_3518.method_15253(json, "background", Category.DEFAULT_BACKGROUND));
        var backgroundWidth = class_3518.method_15282(json, "background_width", Category.DEFAULT_BACKGROUND_WIDTH);
        var backgroundHeight = class_3518.method_15282(json, "background_height", Category.DEFAULT_BACKGROUND_HEIGHT);
        var defaultMaxScrollX = class_3518.method_15282(json, "max_scroll_x", Category.DEFAULT_MAX_SCROLL_X);
        var defaultMaxScrollY = class_3518.method_15282(json, "max_scroll_y", Category.DEFAULT_MAX_SCROLL_Y);

        var backgroundTextureZoomMultiplier = class_3518.method_15277(json, "background_texture_zoom_multiplier", Category.DEFAULT_BACKGROUND_TEXTURE_ZOOM_MULTIPLIER);
        var entryTextures = class_2960.method_60654(class_3518.method_15253(json, "entry_textures", Category.DEFAULT_ENTRY_TEXTURES));
        var showCategoryButton = class_3518.method_15258(json, "show_category_button", true);

        BookCondition condition = new BookNoneCondition(); //default to unlocked
        if (json.has("condition")) {
            condition = BookCondition.fromJson(id, json.getAsJsonObject("condition"), provider);
        }

        List<BookCategoryBackgroundParallaxLayer> backgroundParallaxLayers = List.of();
        if (json.has("background_parallax_layers"))
            backgroundParallaxLayers = BookCategoryBackgroundParallaxLayer.fromJson(json.getAsJsonArray("background_parallax_layers"));

        class_2960 entryToOpen = null;
        if (json.has("entry_to_open")) {
            var entryToOpenPath = class_3518.method_15265(json, "entry_to_open");
            entryToOpen = entryToOpenPath.contains(":") ?
                    class_2960.method_60654(entryToOpenPath) :
                    class_2960.method_60655(id.method_12836(), entryToOpenPath);
        }
        boolean openEntryOnlyOnce = class_3518.method_15258(json, "open_entry_to_open_only_once", true);

        return new BookCategory(id, name, description, sortNumber, condition, showCategoryButton, icon, displayMode, background, backgroundWidth, backgroundHeight,
                defaultMaxScrollX, defaultMaxScrollY, backgroundTextureZoomMultiplier, backgroundParallaxLayers, entryTextures, entryToOpen, openEntryOnlyOnce);
    }

    public static BookCategory fromNetwork(class_2960 id, class_9129 buffer) {
        var name = buffer.method_19772();
        var description = BookTextHolder.fromNetwork(buffer);
        var sortNumber = buffer.readInt();
        var icon = BookIcon.fromNetwork(buffer);
        var displayMode = BookDisplayMode.byId(buffer.readByte());
        var background = buffer.method_10810();
        var backgroundWidth = buffer.method_10816();
        var backgroundHeight = buffer.method_10816();
        var defaultMaxScrollX = buffer.method_10816();
        var defaultMaxScrollY = buffer.method_10816();
        var backgroundTextureZoomMultiplier = buffer.readFloat();
        var backgroundParallaxLayers = buffer.method_34066(BookCategoryBackgroundParallaxLayer::fromNetwork);
        var entryTextures = buffer.method_10810();
        var condition = BookCondition.fromNetwork(buffer);
        var showCategoryButton = buffer.readBoolean();
        var entryToOpen = buffer.method_43827(class_2540::method_10810);
        var openEntryOnlyOnce = buffer.readBoolean();
        return new BookCategory(id, name, description, sortNumber, condition, showCategoryButton, icon, displayMode, background, backgroundWidth, backgroundHeight,
                defaultMaxScrollX, defaultMaxScrollY, backgroundTextureZoomMultiplier, backgroundParallaxLayers, entryTextures, entryToOpen, openEntryOnlyOnce);
    }

    public void toNetwork(class_9129 buffer) {
        buffer.method_10814(this.name);
        this.description.toNetwork(buffer);
        buffer.method_53002(this.sortNumber);
        this.icon.toNetwork(buffer);
        buffer.method_52997(this.displayMode.ordinal());
        buffer.method_10812(this.background);
        buffer.method_10804(this.backgroundWidth);
        buffer.method_10804(this.backgroundHeight);
        buffer.method_10804(this.maxScrollX);
        buffer.method_10804(this.maxScrollY);
        buffer.method_52941(this.backgroundTextureZoomMultiplier);
        buffer.method_34062(this.backgroundParallaxLayers, (buf, layer) -> layer.toNetwork(buf));
        buffer.method_10812(this.entryTextures);
        BookCondition.toNetwork(this.condition, buffer);
        buffer.method_52964(this.showCategoryButton);
        buffer.method_43826(this.entryToOpen, class_2540::method_10812);
        buffer.method_52964(this.openEntryToOpenOnlyOnce);
    }

    /**
     * call after loading the book jsons to finalize.
     */
    public void build(class_1937 level, Book book) {
        this.book = book;

        for (var entry : this.entries.values()) {
            BookErrorManager.get().getContextHelper().entryId = entry.getId();
            entry.build(level, this);
            BookErrorManager.get().getContextHelper().entryId = null;
        }

        if (this.entryToOpen != null) {
            var entry = this.entries.get(this.entryToOpen);
            if (entry == null) { //entry must exist in category!
                BookErrorManager.get().error(MessageFormat.format("EntryToOpen \"{0}\" in Category \"{1}\" does not exist.", this.entryToOpen, this.getId()));
            }
        }
    }

    /**
     * Called after build() (after loading the book jsons) to render markdown and store any errors
     */
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        if (!this.description.hasComponent()) {
            this.description = new RenderedBookTextHolder(this.description, textRenderer.render(this.description.getString()));
        }

        for (var entry : this.entries.values()) {
            BookErrorManager.get().getContextHelper().entryId = entry.getId();
            try {
                entry.prerenderMarkdown(textRenderer);
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to render markdown in book '" + this.book.getId() + "' for entry '" + entry.getId() + "'", e);
            }

            BookErrorManager.get().getContextHelper().entryId = null;
        }
    }

    public class_2960 getId() {
        return this.id;
    }

    public Book getBook() {
        return this.book;
    }

    public String getName() {
        return this.name;
    }

    public BookTextHolder getDescription() {
        return this.description;
    }

    public int getSortNumber() {
        return this.sortNumber;
    }

    public BookIcon getIcon() {
        return this.icon;
    }

    public BookDisplayMode getDisplayMode() {
        return this.displayMode;
    }

    public class_2960 getBackground() {
        return this.background;
    }

    public int getBackgroundWidth() {
        return this.backgroundWidth;
    }

    public int getBackgroundHeight() {
        return this.backgroundHeight;
    }

    public int getMaxScrollX() {
        return this.maxScrollX;
    }

    public int getMaxScrollY() {
        return this.maxScrollY;
    }

    public float getBackgroundTextureZoomMultiplier() {
        return this.backgroundTextureZoomMultiplier;
    }

    public List<BookCategoryBackgroundParallaxLayer> getBackgroundParallaxLayers() {
        return this.backgroundParallaxLayers;
    }

    public class_2960 getEntryTextures() {
        return this.entryTextures;
    }

    public Map<class_2960, BookEntry> getEntries() {
        return this.entries;
    }

    public void addEntry(BookEntry entry) {
        this.entries.putIfAbsent(entry.getId(), entry);
    }

    public BookEntry getEntry(class_2960 id) {
        return this.entries.get(id);
    }

    public BookCondition getCondition() {
        return this.condition;
    }


    public boolean openEntryToOpenOnlyOnce() {
        return this.openEntryToOpenOnlyOnce;
    }

    public class_2960 getEntryToOpen() {
        return this.entryToOpen;
    }

    public boolean showCategoryButton() {
        return this.showCategoryButton;
    }
}
