/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.data;

import ;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.Data;
import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.book.BookCategory;
import com.klikli_dev.modonomicon.book.BookCommand;
import com.klikli_dev.modonomicon.book.BookTextHolder;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.entries.BookContentEntry;
import com.klikli_dev.modonomicon.book.entries.BookEntry;
import com.klikli_dev.modonomicon.book.entries.CategoryLinkBookEntry;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.networking.Message;
import com.klikli_dev.modonomicon.networking.SyncBookDataMessage;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.klikli_dev.modonomicon.platform.Services;
import it.unimi.dsi.fastutil.objects.Object2FloatOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectMaps;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3222;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4309;
import net.minecraft.class_7225;
import net.minecraft.class_8779;


public class BookDataManager extends class_4309 {
    public static final String FOLDER = Data.MODONOMICON_DATA_PATH;
    public static final Gson GSON = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().create();

    private static final BookDataManager instance = new BookDataManager();

    private final Map<class_2960, Book> books = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
    private boolean loaded;
    private boolean booksBuilt;
    private class_7225.class_7874 registries;

    private BookDataManager() {
        super(GSON, FOLDER);
    }

    public static BookDataManager get() {
        return instance;
    }

    public void registries(class_7225.class_7874 registries) {
        this.registries = registries;
    }

    public boolean isLoaded() {
        return this.loaded;
    }

    public Map<class_2960, Book> getBooks() {
        return this.books;
    }

    public Book getBook(class_2960 id) {
        return this.books.get(id);
    }

    public Message getSyncMessage() {
        //we hand over a copy of the map, because otherwise in SP scenarios if we clear this.books to prepare for receiving the message, we also clear the books in the message
        return new SyncBookDataMessage(this.books);
    }

    public boolean areBooksBuilt() {
        return this.booksBuilt;
    }

    public void onDatapackSyncPacket(SyncBookDataMessage message) {
        this.preLoad();
        this.books.putAll(message.books);
        this.onLoadingComplete();
    }

    public void onDatapackSync(class_3222 player) {

        this.tryBuildBooks(player.method_37908()); //lazily build books when first client connects

        //If integrated server and host (= SP or lan host), don't send as we already have it
        if(player.field_13987.field_45013.method_10756())
            return;

        Message syncMessage = this.getSyncMessage();

        Services.NETWORK.sendToSplit(player, syncMessage);
    }

    public void onRecipesUpdated(class_1937 level) {
        Client.get().resetUseFallbackFont();
        this.tryBuildBooks(level);
        this.prerenderMarkdown(level.method_30349());
    }

    public void preLoad() {
        this.booksBuilt = false;
        this.loaded = false;
        this.books.clear();
        BookErrorManager.get().reset();
    }

    public void buildBooks(class_1937 level) {
        for (var book : this.books.values()) {
            BookErrorManager.get().getContextHelper().reset();
            BookErrorManager.get().setCurrentBookId(book.getId());
            try {
                book.build(level);
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to build book '" + book.getId() + "'", e);
            }
            BookErrorManager.get().setCurrentBookId(null);
        }
    }

    public void prerenderMarkdown(class_7225.class_7874 provider) {
        Modonomicon.LOG.info("Pre-rendering markdown ...");
        for (var book : this.books.values()) {

            BookErrorManager.get().getContextHelper().reset();
            BookErrorManager.get().setCurrentBookId(book.getId());

            //TODO: allow modders to configure this renderer
            var textRenderer = new BookTextRenderer(book, provider);

            if (!BookErrorManager.get().hasErrors(book.getId())) {
                try {
                    book.prerenderMarkdown(textRenderer);
                } catch (Exception e) {
                    BookErrorManager.get().error("Failed to render markdown for book '" + book.getId() + "'", e);
                }
            } else {
                BookErrorManager.get().error("Cannot render markdown for book '" + book.getId() + " because of errors during book build'");
            }

            BookErrorManager.get().setCurrentBookId(null);
        }
        Modonomicon.LOG.info("Finished pre-rendering markdown.");
    }

    /**
     * On server, called on datapack sync (because we need the data before we send the datapack sync packet) On client,
     * called on recipes updated, because recipes are available to the client only after datapack sync is complete
     */
    public boolean tryBuildBooks(class_1937 level) {
        if (this.booksBuilt) {
            return false;
        }

        if(!level.method_8608()){
            this.resolveMacros(); //macros are only resolved serverside, the resolved macros are then stored in the book.
        }

        Modonomicon.LOG.info("Building books ...");
        this.buildBooks(level);
        this.booksBuilt = true;
        Modonomicon.LOG.info("Books built.");
        return true;
    }

    public void resolveMacros(){
        this.getBooks().forEach((id, book) -> {
            var macroLoaders = LoaderRegistry.getDynamicTextMacroLoaders(id);
            macroLoaders.forEach(loader -> loader.load().forEach(book::addMacro));
        });
    }

    protected void onLoadingComplete() {
        this.loaded = true;
    }

    private Book loadBook(class_2960 key, JsonObject value, class_7225.class_7874 provider) {
        return Book.fromJson(key, value, provider);
    }

    private BookCategory loadCategory(class_2960 key, JsonObject value, class_7225.class_7874 provider) {
        return BookCategory.fromJson(key, value, provider);
    }

    private BookEntry loadEntry(class_2960 id, JsonObject value, boolean autoAddReadConditions, class_7225.class_7874 provider) {
        if (value.has("type")) {
            class_2960 typeId = class_2960.method_12829(value.get("type").getAsString());
            return LoaderRegistry.getEntryJsonLoader(typeId).fromJson(id, value, autoAddReadConditions, provider);
        }

        // This part here is for backwards compatibility and simplicity
        // If an entry does not have a type specified, ContentEntry is assumed
        // unless it has a property called "category_to_open" (CategoryLinkEntry)
        if (value.has("category_to_open")) {
            return CategoryLinkBookEntry.fromJson(id, value, autoAddReadConditions, provider);
        }
        return BookContentEntry.fromJson(id, value, autoAddReadConditions, provider);
    }

    private BookCommand loadCommand(class_2960 key, JsonObject value) {
        return BookCommand.fromJson(key, value);
    }

    /**
     * Loads only the condition on the given category, entry or page and runs testOnLoad.
     *
     * @param key        the resource location of the content
     * @param bookObject the json object representing the content
     * @return false if the condition is not met and the content should not be loaded.
     */
    private boolean testConditionOnLoad(class_2960 key, JsonObject bookObject, class_7225.class_7874 provider) {
        if (!bookObject.has("condition")) {
            return true; //no condition -> always load
        }

        return BookCondition.fromJson(key, bookObject.getAsJsonObject("condition"), provider).testOnLoad();
    }


    private void categorizeContent(Map<class_2960, JsonElement> content,
                                   HashMap<class_2960, JsonObject> bookJsons,
                                   HashMap<class_2960, JsonObject> categoryJsons,
                                   HashMap<class_2960, JsonObject> entryJsons,
                                   HashMap<class_2960, JsonObject> commandJsons
    ) {
        for (var entry : content.entrySet()) {
            var pathParts = entry.getKey().method_12832().split("/");

            var bookId = class_2960.method_60655(entry.getKey().method_12836(), pathParts[0]);
            switch (pathParts[1]) {
                case "book" -> {
                    bookJsons.put(entry.getKey(), entry.getValue().getAsJsonObject());
                }
                case "entries" -> {
                    entryJsons.put(entry.getKey(), entry.getValue().getAsJsonObject());
                }
                case "categories" -> {
                    categoryJsons.put(entry.getKey(), entry.getValue().getAsJsonObject());
                }
                case "commands" -> {
                    commandJsons.put(entry.getKey(), entry.getValue().getAsJsonObject());
                }
                default -> {
                    Modonomicon.LOG.warn("Found unknown content for book '{}': '{}'. " +
                            "Should be one of: [File: book.json, Directory: entries/, Directory: categories/, Directory: commands/]", bookId, entry.getKey());
                    BookErrorManager.get().error(bookId, "Found unknown content for book '" + bookId + "': '" + entry.getKey() + "'. " +
                            "Should be one of: [File: book.json, Directory: entries/, Directory: categories/, Directory: commands/]");
                }
            }
        }
    }

    @Override
    protected void apply(Map<class_2960, JsonElement> content, class_3300 pResourceManager, class_3695 pProfiler) {
        this.preLoad();

        //TODO: handle datapack overrides, see TagLoader#load line 69 (refers to Tag.Builder#addFromJson)

        //first, load all json entries
        var bookJsons = new HashMap<class_2960, JsonObject>();
        var categoryJsons = new HashMap<class_2960, JsonObject>();
        var entryJsons = new HashMap<class_2960, JsonObject>();
        var commandJsons = new HashMap<class_2960, JsonObject>();
        this.categorizeContent(content, bookJsons, categoryJsons, entryJsons, commandJsons);

        //load books
        for (var entry : bookJsons.entrySet()) {
            try {
                var pathParts = entry.getKey().method_12832().split("/");
                var bookId = class_2960.method_60655(entry.getKey().method_12836(), pathParts[0]);
                BookErrorManager.get().setCurrentBookId(bookId);
                BookErrorManager.get().setContext("Loading Book JSON");
                var book = this.loadBook(bookId, entry.getValue(), this.registries);
                this.books.put(book.getId(), book);
                BookErrorManager.get().reset();
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to load book '" + entry.getKey() + "'", e);
                BookErrorManager.get().reset();
            }
        }

        //load categories
        for (var entry : categoryJsons.entrySet()) {
            try {
                //load categories and link to book
                var pathParts = entry.getKey().method_12832().split("/");
                var bookId = class_2960.method_60655(entry.getKey().method_12836(), pathParts[0]);
                BookErrorManager.get().setCurrentBookId(bookId);

                //category id skips the book id and the category directory
                var categoryId = class_2960.method_60655(entry.getKey().method_12836(), Arrays.stream(pathParts).skip(2).collect(Collectors.joining("/")));

                BookErrorManager.get().getContextHelper().categoryId = categoryId;
                //test if we should load the category at all
                if (!this.testConditionOnLoad(categoryId, entry.getValue(), this.registries)) {
                    continue;
                }

                var category = this.loadCategory(categoryId, entry.getValue(), this.registries);

                //link category and book
                var book = this.books.get(bookId);
                book.addCategory(category);

                BookErrorManager.get().reset();
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to load category '" + entry.getKey() + "'", e);
                BookErrorManager.get().reset();
            }
        }

        //load entries
        for (var entry : entryJsons.entrySet()) {
            try {
                //load entries and link to category
                var pathParts = entry.getKey().method_12832().split("/");
                var bookId = class_2960.method_60655(entry.getKey().method_12836(), pathParts[0]);
                BookErrorManager.get().setCurrentBookId(bookId);

                //entry id skips the book id and the entries directory, but keeps category so it is unique
                var entryId = class_2960.method_60655(entry.getKey().method_12836(), Arrays.stream(pathParts).skip(2).collect(Collectors.joining("/")));

                BookErrorManager.get().getContextHelper().entryId = entryId;
                //test if we should load the category at all
                if (!this.testConditionOnLoad(entryId, entry.getValue(), this.registries)) {
                    continue;
                }

                var bookEntry = this.loadEntry(entryId, entry.getValue(), this.books.get(bookId).autoAddReadConditions(), this.registries);

                //link entry and category
                var book = this.books.get(bookId);
                var category = book.getCategory(bookEntry.getCategoryId());
                category.addEntry(bookEntry);

                BookErrorManager.get().reset();
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to load entry '" + entry.getKey() + "'", e);
                BookErrorManager.get().reset();
            }
        }

        //load commands
        for (var entry : commandJsons.entrySet()) {
            try {
                //load commands and link to book
                var pathParts = entry.getKey().method_12832().split("/");
                var bookId = class_2960.method_60655(entry.getKey().method_12836(), pathParts[0]);
                BookErrorManager.get().setCurrentBookId(bookId);

                BookErrorManager.get().setContext("Loading Command JSON");

                //commands id skips the book id and the commands directory
                var commandId = class_2960.method_60655(entry.getKey().method_12836(), Arrays.stream(pathParts).skip(2).collect(Collectors.joining("/")));

                BookErrorManager.get().setContext("Loading Command JSON: " + commandId);
                var command = this.loadCommand(commandId, entry.getValue());

                //link command and book
                var book = this.books.get(bookId);
                book.addCommand(command);
                BookErrorManager.get().reset();
            } catch (Exception e) {
                BookErrorManager.get().error("Failed to load command '" + entry.getKey() + "'", e);
                BookErrorManager.get().reset();
            }
        }

        BookErrorManager.get().reset();

        this.onLoadingComplete();
    }

    public static class Client extends class_4309 {

        private static final Client instance = new Client();

        private static final class_2960 fallbackFont = class_2960.method_60655("minecraft", "default");
        /**
         * Our local advancement cache, because we cannot just store random advancement in ClientAdvancements -> they get rejected
         */
        private final Map<class_2960, class_8779> advancements = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
        private final Object2FloatOpenHashMap<BookTextHolder.ScaleCacheKey> bookTextHolderScaleCache = new Object2FloatOpenHashMap<>();
        private boolean isFallbackLocale;
        private boolean isFontInitialized;

        public Client() {
            super(GSON, FOLDER);
            this.bookTextHolderScaleCache.defaultReturnValue(-1f);
        }

        public static Client get() {
            return instance;
        }

        public void resetUseFallbackFont() {
            this.isFontInitialized = false;
        }

        public boolean useFallbackFont() {
            if (!this.isFontInitialized) {
                this.isFontInitialized = true;

                var locale = class_310.method_1551().method_1526().method_4669();
                this.isFallbackLocale = ClientServices.CLIENT_CONFIG.fontFallbackLocales().stream().anyMatch(l -> l.equals(locale));
            }

            return this.isFallbackLocale;
        }

        public class_2960 safeFont(class_2960 requested) {
            return this.useFallbackFont() ? fallbackFont : requested;
        }

        public class_8779 getAdvancement(class_2960 id) {
            return this.advancements.get(id);
        }

        public void putScale(BookTextHolder holder, int width, int height, float scale) {
            this.bookTextHolderScaleCache.put(new BookTextHolder.ScaleCacheKey(holder, width, height), scale);
        }

        /**
         * Returns -1 if no scale is found
         */
        public float getScale(BookTextHolder holder, int width, int height) {
            return this.bookTextHolderScaleCache.getFloat(new BookTextHolder.ScaleCacheKey(holder, width, height));
        }

        public void addAdvancement(class_8779 advancement) {
            this.advancements.put(advancement.comp_1919(), advancement);
        }

        @Override
        protected void apply(Map<class_2960, JsonElement> object, class_3300 resourceManager, class_3695 profiler) {
            //reset on reload
            this.resetUseFallbackFont();
            this.advancements.clear();
            this.bookTextHolderScaleCache.clear();
        }
    }
}
