/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book;

import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.Data;
import com.klikli_dev.modonomicon.book.entries.BookEntry;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.client.gui.book.BookAddress;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.data.BookEntryJsonLoader;
import com.klikli_dev.modonomicon.util.BookGsonHelper;
import it.unimi.dsi.fastutil.objects.Object2ObjectMaps;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7225;
import net.minecraft.class_9129;

public class Book {
    protected class_2960 id;
    protected String name;
    protected BookTextHolder description;
    protected String tooltip;
    protected String creativeTab;


    protected class_2960 model;
    protected class_2960 bookOverviewTexture;
    protected class_2960 frameTexture;
    protected BookFrameOverlay topFrameOverlay;
    protected BookFrameOverlay bottomFrameOverlay;
    protected BookFrameOverlay leftFrameOverlay;
    protected BookFrameOverlay rightFrameOverlay;
    protected class_2960 bookContentTexture;

    protected class_2960 craftingTexture;
    protected class_2960 turnPageSound;
    protected Map<class_2960, BookCategory> categories;
    protected Map<class_2960, BookEntry> entries;
    protected Map<class_2960, BookCommand> commands;


    protected int defaultTitleColor;
    protected float categoryButtonIconScale;
    protected boolean autoAddReadConditions;
    protected boolean generateBookItem;
    @Nullable
    protected class_2960 customBookItem;

    protected class_2960 font;

    /**
     * The display mode - node based (thaumonomicon style) or index based (lexica botania / patchouli style)
     * If the book is in index mode then all categories will also be shown in index mode. If the book is in node mode, then individual categories can be in index mode.
     * If in index mode, the frame textures will be ignored, instead bookContentTexture will be used.
     */
    protected BookDisplayMode displayMode;

    /**
     * When rendering book text holders, add this offset to the x position (basically, create a left margin).
     * Will be automatically subtracted from the width to avoid overflow.
     */
    protected int bookTextOffsetX;

    /**
     * When rendering book text holders, add this offset to the y position (basically, create a top margin).
     */
    protected int bookTextOffsetY;

    /**
     * When rendering book text holders, add this offset to the width (allows to create a right margin)
     * To make the line end move to the left (as it would for a margin setting in eg css), use a negative value.
     */
    protected int bookTextOffsetWidth;

    /**
     * When rendering book text holders, add this offset to the height (allows to create a bottom margin)
     * To make the bottom end of the text move up (as it would for a margin setting in eg css), use a negative value.
     */
    protected int bookTextOffsetHeight;

    protected int categoryButtonXOffset;
    protected int categoryButtonYOffset;
    protected int searchButtonXOffset;
    protected int searchButtonYOffset;
    protected int readAllButtonYOffset;

    protected class_2960 leafletEntry;

    protected PageDisplayMode pageDisplayMode = PageDisplayMode.DOUBLE_PAGE;
    protected class_2960 singlePageTexture = class_2960.method_60654(Data.Book.DEFAULT_SINGLE_PAGE_TEXTURE);

    /**
     * If true, invalid links do not show an error screen when opening the book.
     * Instead, the book and pages will open, but the link will not work.
     */
    protected boolean allowOpenBooksWithInvalidLinks;

    /**
     * A map of macros. This is filled automatically based on LoaderRegistry#dynamicTextMacroLoaders, not loaded from JSON.
     */
    protected final Map<String, String> textMacros = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());

    public Book(class_2960 id, String name, BookTextHolder description, String tooltip, class_2960 model, BookDisplayMode displayMode, boolean generateBookItem,
                @Nullable class_2960 customBookItem, String creativeTab, class_2960 font, class_2960 bookOverviewTexture, class_2960 frameTexture,
                BookFrameOverlay topFrameOverlay, BookFrameOverlay bottomFrameOverlay, BookFrameOverlay leftFrameOverlay, BookFrameOverlay rightFrameOverlay,
                class_2960 bookContentTexture, class_2960 craftingTexture, class_2960 turnPageSound,
                int defaultTitleColor, float categoryButtonIconScale, boolean autoAddReadConditions, int bookTextOffsetX, int bookTextOffsetY, int bookTextOffsetWidth, int bookTextOffsetHeight,
                int categoryButtonXOffset, int categoryButtonYOffset, int searchButtonXOffset, int searchButtonYOffset, int readAllButtonYOffset, class_2960 leafletEntry,
                PageDisplayMode pageDisplayMode, class_2960 singlePageTexture, boolean allowOpenBooksWithInvalidLinks) {
        this.id = id;
        this.name = name;
        this.description = description;
        this.tooltip = tooltip;
        this.model = model;
        this.displayMode = displayMode;
        this.generateBookItem = generateBookItem;
        this.customBookItem = customBookItem;
        this.creativeTab = creativeTab;
        this.bookOverviewTexture = bookOverviewTexture;
        this.font = font;
        this.frameTexture = frameTexture;
        this.topFrameOverlay = topFrameOverlay;
        this.bottomFrameOverlay = bottomFrameOverlay;
        this.leftFrameOverlay = leftFrameOverlay;
        this.rightFrameOverlay = rightFrameOverlay;
        this.bookContentTexture = bookContentTexture;
        this.craftingTexture = craftingTexture;
        this.turnPageSound = turnPageSound;
        this.defaultTitleColor = defaultTitleColor;
        this.categoryButtonIconScale = categoryButtonIconScale;
        this.autoAddReadConditions = autoAddReadConditions;
        this.categories = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
        this.entries = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
        this.commands = Object2ObjectMaps.synchronize(new Object2ObjectOpenHashMap<>());
        this.bookTextOffsetX = bookTextOffsetX;
        this.bookTextOffsetY = bookTextOffsetY;
        this.bookTextOffsetWidth = bookTextOffsetWidth;
        this.bookTextOffsetHeight = bookTextOffsetHeight;

        this.categoryButtonXOffset = categoryButtonXOffset;
        this.categoryButtonYOffset = categoryButtonYOffset;
        this.searchButtonXOffset = searchButtonXOffset;
        this.searchButtonYOffset = searchButtonYOffset;
        this.readAllButtonYOffset = readAllButtonYOffset;

        this.leafletEntry = leafletEntry;

        this.pageDisplayMode = pageDisplayMode;
        this.singlePageTexture = singlePageTexture;

        this.allowOpenBooksWithInvalidLinks = allowOpenBooksWithInvalidLinks;
    }

    public static Book fromJson(class_2960 id, JsonObject json, class_7225.class_7874 provider) {
        var name = class_3518.method_15265(json, "name");
        var description = BookGsonHelper.getAsBookTextHolder(json, "description", BookTextHolder.EMPTY, provider);
        var tooltip = class_3518.method_15253(json, "tooltip", "");
        var model = class_2960.method_60654(class_3518.method_15253(json, "model", Data.Book.DEFAULT_MODEL));
        var generateBookItem = class_3518.method_15258(json, "generate_book_item", true);
        var displayMode = BookDisplayMode.byName(class_3518.method_15253(json, "display_mode", BookDisplayMode.NODE.method_15434()));
        var customBookItem = json.has("custom_book_item") ?
                class_2960.method_60654(class_3518.method_15265(json, "custom_book_item")) :
                null;
        var creativeTab = class_3518.method_15253(json, "creative_tab", "misc");
        var bookOverviewTexture = class_2960.method_60654(class_3518.method_15253(json, "book_overview_texture", Data.Book.DEFAULT_OVERVIEW_TEXTURE));
        var frameTexture = class_2960.method_60654(class_3518.method_15253(json, "frame_texture", Data.Book.DEFAULT_FRAME_TEXTURE));

        var topFrameOverlay = json.has("top_frame_overlay") ?
                BookFrameOverlay.fromJson(json.get("top_frame_overlay").getAsJsonObject()) :
                Data.Book.DEFAULT_TOP_FRAME_OVERLAY;

        var bottomFrameOverlay = json.has("bottom_frame_overlay") ?
                BookFrameOverlay.fromJson(json.get("bottom_frame_overlay").getAsJsonObject()) :
                Data.Book.DEFAULT_BOTTOM_FRAME_OVERLAY;

        var leftFrameOverlay = json.has("left_frame_overlay") ?
                BookFrameOverlay.fromJson(json.get("left_frame_overlay").getAsJsonObject()) :
                Data.Book.DEFAULT_LEFT_FRAME_OVERLAY;

        var rightFrameOverlay = json.has("right_frame_overlay") ?
                BookFrameOverlay.fromJson(json.get("right_frame_overlay").getAsJsonObject()) :
                Data.Book.DEFAULT_RIGHT_FRAME_OVERLAY;

        var font = class_2960.method_60654(class_3518.method_15253(json, "font", Data.Book.DEFAULT_FONT));

        var bookContentTexture = class_2960.method_60654(class_3518.method_15253(json, "book_content_texture", Data.Book.DEFAULT_CONTENT_TEXTURE));
        var craftingTexture = class_2960.method_60654(class_3518.method_15253(json, "crafting_texture", Data.Book.DEFAULT_CRAFTING_TEXTURE));
        var turnPageSound = class_2960.method_60654(class_3518.method_15253(json, "turn_page_sound", Data.Book.DEFAULT_PAGE_TURN_SOUND));
        var defaultTitleColor = class_3518.method_15282(json, "default_title_color", 0x00000);
        var categoryButtonIconScale = class_3518.method_15277(json, "category_button_icon_scale", 1.0f);
        var autoAddReadConditions = class_3518.method_15258(json, "auto_add_read_conditions", false);

        var bookTextOffsetX = class_3518.method_15282(json, "book_text_offset_x", 0);
        var bookTextOffsetY = class_3518.method_15282(json, "book_text_offset_y", 0);
        var bookTextOffsetWidth = class_3518.method_15282(json, "book_text_offset_width", 0);
        var bookTextOffsetHeight = class_3518.method_15282(json, "book_text_offset_height", 0);

        var categoryButtonXOffset = class_3518.method_15282(json, "category_button_x_offset", 0);
        var categoryButtonYOffset = class_3518.method_15282(json, "category_button_y_offset", 0);
        var searchButtonXOffset = class_3518.method_15282(json, "search_button_x_offset", 0);
        var searchButtonYOffset = class_3518.method_15282(json, "search_button_y_offset", 0);
        var readAllButtonYOffset = class_3518.method_15282(json, "read_all_button_y_offset", 0);

        class_2960 leafletEntry = null;
        if (json.has("leaflet_entry")) {
            var leafletEntryPath = class_3518.method_15265(json, "leaflet_entry");
            //leaflet entries can be without a namespace, in which case we use the book namespace.
            leafletEntry = leafletEntryPath.contains(":") ?
                    class_2960.method_60654(leafletEntryPath) :
                    class_2960.method_60655(id.method_12836(), leafletEntryPath);
        }

        var pageDisplayMode = PageDisplayMode.byName(class_3518.method_15253(json, "page_display_mode", PageDisplayMode.DOUBLE_PAGE.method_15434()));
        var singlePageTexture = class_2960.method_60654(class_3518.method_15253(json, "single_page_texture", Data.Book.DEFAULT_SINGLE_PAGE_TEXTURE));

        var allowOpenBooksWithInvalidLinks = class_3518.method_15258(json, "allow_open_book_with_invalid_links", false);

        return new Book(id, name, description, tooltip, model, displayMode, generateBookItem, customBookItem, creativeTab, font, bookOverviewTexture,
                frameTexture, topFrameOverlay, bottomFrameOverlay, leftFrameOverlay, rightFrameOverlay,
                bookContentTexture, craftingTexture, turnPageSound, defaultTitleColor, categoryButtonIconScale, autoAddReadConditions, bookTextOffsetX, bookTextOffsetY, bookTextOffsetWidth, bookTextOffsetHeight,
                categoryButtonXOffset, categoryButtonYOffset, searchButtonXOffset, searchButtonYOffset, readAllButtonYOffset, leafletEntry, pageDisplayMode, singlePageTexture, allowOpenBooksWithInvalidLinks);
    }


    @SuppressWarnings("deprecation")
    public static Book fromNetwork(class_2960 id, class_9129 buffer) {
        var name = buffer.method_19772();
        var description = BookTextHolder.fromNetwork(buffer);
        var tooltip = buffer.method_19772();
        var model = buffer.method_10810();
        var displayMode = BookDisplayMode.byId(buffer.readByte());

        var generateBookItem = buffer.readBoolean();
        var customBookItem = buffer.method_43827(class_2540::method_10810);
        var creativeTab = buffer.method_19772();

        var font = buffer.method_10810();

        var bookOverviewTexture = buffer.method_10810();

        var frameTexture = buffer.method_10810();

        var topFrameOverlay = BookFrameOverlay.fromNetwork(buffer);
        var bottomFrameOverlay = BookFrameOverlay.fromNetwork(buffer);
        var leftFrameOverlay = BookFrameOverlay.fromNetwork(buffer);
        var rightFrameOverlay = BookFrameOverlay.fromNetwork(buffer);

        var bookContentTexture = buffer.method_10810();
        var craftingTexture = buffer.method_10810();
        var turnPageSound = buffer.method_10810();
        var defaultTitleColor = buffer.readInt();
        var categoryButtonIconScale = buffer.readFloat();
        var autoAddReadConditions = buffer.readBoolean();
        var bookTextOffsetX = (int) buffer.readShort();
        var bookTextOffsetY = (int) buffer.readShort();
        var bookTextOffsetWidth = (int) buffer.readShort();
        var bookTextOffsetHeight = (int) buffer.readShort();

        var categoryButtonXOffset = (int) buffer.readShort();
        var categoryButtonYOffset = (int) buffer.readShort();
        var searchButtonXOffset = (int) buffer.readShort();
        var searchButtonYOffset = (int) buffer.readShort();
        var readAllButtonYOffset = (int) buffer.readShort();

        var leafletEntry = buffer.method_43827(class_2540::method_10810);

        var pageDisplayMode = PageDisplayMode.byId(buffer.readByte());
        var singlePageTexture = buffer.method_10810();

        var allowOpenBooksWithInvalidLinks = buffer.readBoolean();

        var textMacros = buffer.method_34067((b) -> b.method_19772(), (b) -> b.method_19772()); //necessary because using lambda causes ambiguous reference in Neo with their IFriendlyByteBufExtension#readMap
        var book = new Book(id, name, description, tooltip, model, displayMode, generateBookItem, customBookItem, creativeTab, font, bookOverviewTexture,
                frameTexture, topFrameOverlay, bottomFrameOverlay, leftFrameOverlay, rightFrameOverlay,
                bookContentTexture, craftingTexture, turnPageSound, defaultTitleColor, categoryButtonIconScale, autoAddReadConditions, bookTextOffsetX, bookTextOffsetY, bookTextOffsetWidth, bookTextOffsetHeight,
                categoryButtonXOffset, categoryButtonYOffset, searchButtonXOffset, searchButtonYOffset, readAllButtonYOffset, leafletEntry, pageDisplayMode, singlePageTexture, allowOpenBooksWithInvalidLinks);

        book.textMacros().putAll(textMacros);

        return book;
    }

    /**
     * call after loading the book jsons to finalize.
     */
    public void build(class_1937 level) {
        //first "backlink" all our entries directly into the book
        for (var category : this.categories.values()) {
            for (var entry : category.getEntries().values()) {
                this.addEntry(entry);
            }
        }

        //then build categories, which will in turn build entries (which need the above backlinks to resolve parents)
        for (var category : this.categories.values()) {
            BookErrorManager.get().getContextHelper().categoryId = category.getId();
            category.build(level, this);
            BookErrorManager.get().getContextHelper().categoryId = null;
        }

        for (var command : this.commands.values()) {
            command.build(this);
        }
    }

    /**
     * Called after build() (after loading the book jsons) to render markdown and store any errors
     */
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        if (!this.description.hasComponent()) {
            this.description = new RenderedBookTextHolder(this.description, textRenderer.render(this.description.getString()));
        }

        for (var category : this.categories.values()) {
            BookErrorManager.get().getContextHelper().categoryId = category.getId();
            category.prerenderMarkdown(textRenderer);
            BookErrorManager.get().getContextHelper().categoryId = null;
        }
    }

    public void addMacro(String key, String value) {
        textMacros.put(key, value);
    }

    public void toNetwork(class_9129 buffer) {
        buffer.method_10814(this.name);
        this.description.toNetwork(buffer);
        buffer.method_10814(this.tooltip);
        buffer.method_10812(this.model);
        buffer.method_52997(this.displayMode.ordinal());

        buffer.method_52964(this.generateBookItem);

        buffer.method_43826(this.customBookItem, class_2540::method_10812);

        buffer.method_10814(this.creativeTab);

        buffer.method_10812(this.font);

        buffer.method_10812(this.bookOverviewTexture);
        buffer.method_10812(this.frameTexture);

        this.topFrameOverlay.toNetwork(buffer);
        this.bottomFrameOverlay.toNetwork(buffer);
        this.leftFrameOverlay.toNetwork(buffer);
        this.rightFrameOverlay.toNetwork(buffer);

        buffer.method_10812(this.bookContentTexture);
        buffer.method_10812(this.craftingTexture);
        buffer.method_10812(this.turnPageSound);
        buffer.method_53002(this.defaultTitleColor);
        buffer.method_52941(this.categoryButtonIconScale);
        buffer.method_52964(this.autoAddReadConditions);

        buffer.method_52998(this.bookTextOffsetX);
        buffer.method_52998(this.bookTextOffsetY);
        buffer.method_52998(this.bookTextOffsetWidth);
        buffer.method_52998(this.bookTextOffsetHeight);

        buffer.method_52998(this.categoryButtonXOffset);
        buffer.method_52998(this.categoryButtonYOffset);
        buffer.method_52998(this.searchButtonXOffset);
        buffer.method_52998(this.searchButtonYOffset);
        buffer.method_52998(this.readAllButtonYOffset);

        buffer.method_43826(this.leafletEntry, class_2540::method_10812);

        buffer.method_52997(this.pageDisplayMode.ordinal());
        buffer.method_10812(this.singlePageTexture);

        buffer.method_52964(this.allowOpenBooksWithInvalidLinks);
        buffer.method_34063(this.textMacros, (b, v) -> b.method_10814(v), (b, v) -> b.method_10814(v));  //necessary because using lambda causes ambiguous reference in Neo with their IFriendlyByteBufExtension#writeMap
    }

    public boolean autoAddReadConditions() {
        return this.autoAddReadConditions;
    }

    public class_2960 getTurnPageSound() {
        return this.turnPageSound;
    }

    public int getDefaultTitleColor() {
        return this.defaultTitleColor;
    }

    public float getCategoryButtonIconScale() {
        return this.categoryButtonIconScale;
    }

    public class_2960 getId() {
        return this.id;
    }

    public Map<String, String> textMacros() {
        return textMacros;
    }

    public void addCategory(BookCategory category) {
        this.categories.putIfAbsent(category.id, category);
    }

    public BookCategory getCategory(class_2960 id) {
        return this.categories.get(id);
    }

    public Map<class_2960, BookCategory> getCategories() {
        return this.categories;
    }

    public List<BookCategory> getCategoriesSorted() {
        return this.categories.values().stream().sorted(Comparator.comparingInt(BookCategory::getSortNumber)).toList();
    }

    public void addEntry(BookEntry entry) {
        this.entries.putIfAbsent(entry.getId(), entry);
    }

    public BookEntry getEntry(class_2960 id) {
        return this.entries.get(id);
    }

    public Map<class_2960, BookEntry> getEntries() {
        return this.entries;
    }

    public void addCommand(BookCommand command) {
        this.commands.putIfAbsent(command.id, command);
    }

    public Map<class_2960, BookCommand> getCommands() {
        return this.commands;
    }

    public BookCommand getCommand(class_2960 id) {
        return this.commands.get(id);
    }

    public String getName() {
        return this.name;
    }

    public BookTextHolder getDescription() {
        return this.description;
    }

    public String getTooltip() {
        return this.tooltip;
    }

    public String getCreativeTab() {
        return this.creativeTab;
    }

    public class_2960 getBookOverviewTexture() {
        return this.bookOverviewTexture;
    }

    public class_2960 getFont() {
        return this.font;
    }

    public class_2960 getFrameTexture() {
        return this.frameTexture;
    }

    public BookFrameOverlay getTopFrameOverlay() {
        return this.topFrameOverlay;
    }

    public BookFrameOverlay getBottomFrameOverlay() {
        return this.bottomFrameOverlay;
    }

    public BookFrameOverlay getLeftFrameOverlay() {
        return this.leftFrameOverlay;
    }

    public BookFrameOverlay getRightFrameOverlay() {
        return this.rightFrameOverlay;
    }

    @Nullable
    public class_2960 getCustomBookItem() {
        return this.customBookItem;
    }

    public class_2960 getCraftingTexture() {
        return this.craftingTexture;
    }

    public class_2960 getBookContentTexture() {
        return this.bookContentTexture;
    }

    public class_2960 getModel() {
        return this.model;
    }

    public BookDisplayMode getDisplayMode() {
        if (this.isLeaflet()) {
            return BookDisplayMode.INDEX;
        }
        return this.displayMode;
    }

    public boolean generateBookItem() {
        return this.generateBookItem;
    }

    public int getBookTextOffsetX() {
        return this.bookTextOffsetX;
    }

    public int getBookTextOffsetY() {
        return this.bookTextOffsetY;
    }

    public int getBookTextOffsetWidth() {
        return this.bookTextOffsetWidth;
    }

    public int getBookTextOffsetHeight() {
        return this.bookTextOffsetHeight;
    }

    public int getCategoryButtonXOffset() {
        return this.categoryButtonXOffset;
    }

    public int getCategoryButtonYOffset() {
        return this.categoryButtonYOffset;
    }

    public int getSearchButtonXOffset() {
        return this.searchButtonXOffset;
    }

    public int getSearchButtonYOffset() {
        return this.searchButtonYOffset;
    }

    public int getReadAllButtonYOffset() {
        return this.readAllButtonYOffset;
    }

    public class_2960 getLeafletEntry() {
        return this.leafletEntry;
    }

    public boolean isLeaflet() {
        return this.leafletEntry != null;
    }

    public BookAddress getLeafletAddress() {
        var leafletEntry = this.getEntry(this.leafletEntry);
        return BookAddress.ignoreSaved(leafletEntry);
    }

    public PageDisplayMode getPageDisplayMode() {
        return this.pageDisplayMode;
    }

    public class_2960 getSinglePageTexture() {
        return this.singlePageTexture;
    }

    public boolean allowOpenBooksWithInvalidLinks() {
        return this.allowOpenBooksWithInvalidLinks;
    }
}
