/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book;

import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_3518;
import org.jetbrains.annotations.Nullable;

public class BookCommand {
    protected class_2960 id;
    protected Book book;

    protected String command;
    protected int permissionLevel;
    /**
     * -1 is unlimited.
     */
    protected int maxUses;

    /**
     * If set, this message will be displayed if the command fails.
     * If not set, the default failure message will be displayed.
     * Should be a translation key.
     */
    @Nullable
    protected String failureMessage;

    /**
     * If set, this message will be displayed when the command succeeds.
     * If not set, no success message will be displayed.
     * Should be a translation key.
     */
    @Nullable
    protected String successMessage;

    public BookCommand(class_2960 id, String command, int permissionLevel, int maxUses, @Nullable String failureMessage, @Nullable String successMessage) {
        this.id = id;
        this.command = command;
        this.permissionLevel = permissionLevel;
        this.maxUses = maxUses;
        this.failureMessage = failureMessage;
        this.successMessage = successMessage;
    }

    public static BookCommand fromJson(class_2960 id, JsonObject json) {
        var command = class_3518.method_15265(json, "command");
        var permissionLevel = class_3518.method_15282(json, "permission_level", ModonomiconConstants.Data.Command.DEFAULT_PERMISSION_LEVEL);
        var maxUses = class_3518.method_15282(json, "max_uses", ModonomiconConstants.Data.Command.DEFAULT_MAX_USES);
        var failureMessage = class_3518.method_15253(json, "failure_message", null);
        var successMessage = class_3518.method_15253(json, "success_message", null);

        return new BookCommand(id, command, permissionLevel, maxUses, failureMessage, successMessage);
    }

    public static BookCommand fromNetwork(class_2960 id, class_2540 buffer) {
        var command = buffer.method_19772();
        var permissionLevel = (int) buffer.readByte();
        var maxUses = buffer.method_10816();
        var failureMessage = buffer.method_43827(class_2540::method_19772);
        var successMessage = buffer.method_43827(class_2540::method_19772);
        return new BookCommand(id, command, permissionLevel, maxUses, failureMessage, successMessage);
    }

    /**
     * call after loading the book jsons to finalize.
     */
    public void build(Book book) {
        this.book = book;
    }

    public void toNetwork(class_2540 buffer) {
        buffer.method_10814(this.command);
        buffer.method_52997(this.permissionLevel);
        buffer.method_10804(this.maxUses);
        buffer.method_43826(this.failureMessage, class_2540::method_10814);
        buffer.method_43826(this.successMessage, class_2540::method_10814);
    }

    public class_2960 getId() {
        return this.id;
    }

    public Book getBook() {
        return this.book;
    }

    public String getCommand() {
        return this.command;
    }

    public int getPermissionLevel() {
        return this.permissionLevel;
    }

    public int getMaxUses() {
        return this.maxUses;
    }

    public @Nullable String getFailureMessage() {
        return this.failureMessage;
    }

    public @Nullable String getSuccessMessage() {
        return this.successMessage;
    }

    public void execute(class_3222 player) {
        if (!BookUnlockStateManager.get().canRunFor(player, this)) {
            var failureMessage = this.failureMessage == null ? ModonomiconConstants.I18n.Command.DEFAULT_FAILURE_MESSAGE : this.failureMessage;

            player.method_64398(class_2561.method_43471(failureMessage).method_27692(class_124.field_1061));
            return;
        } else {
            var commandSourceStack = new class_2168(player.method_64401(), player.method_19538(), player.method_5802(), player.method_51469(), this.permissionLevel, player.method_5477().getString(), player.method_5476(), player.field_13995, player);

            BookUnlockStateManager.get().setRunFor(player, this);

            try {
                player.field_13995.method_3734().method_44252(commandSourceStack, this.command);

                if (this.successMessage != null) {
                    player.method_64398(class_2561.method_43471(this.successMessage).method_27692(class_124.field_1060));
                }
            } catch (Exception e) {
                Modonomicon.LOG.error("Running command [" + this.id.toString() + "] failed: ", e);
            }
        }

        //Even if the command fails we sync the capability.
        //This allows us to "Pretend" success clientside and disable the command source (button/link/etc) so the player cannot spam-click it.
        //spam-clicking would not allow abuse anyway, but would lead to error messages sent back to the player.
        BookUnlockStateManager.get().syncFor(player);
    }
}
