/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.telemetry;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Executors;
import net.minecraft.SharedConstants;
import org.texboobcat.questory.Questory;
import org.texboobcat.questory.telemetry.HeatmapData;
import org.texboobcat.questory.telemetry.SupabaseConfig;
import org.texboobcat.questory.telemetry.TelemetryEvent;

public class SupabaseClient {
    private static final SupabaseClient instance = new SupabaseClient();
    private static final Gson GSON = new Gson();
    private static final int CONSENT_TIMEOUT_SECONDS = 5;
    private static final int UPLOAD_TIMEOUT_SECONDS = 30;
    private static final int HEATMAP_TIMEOUT_SECONDS = 15;
    private static final int MAX_RETRIES = 3;
    private static final long INITIAL_BACKOFF_MS = 1000L;
    private static final long MAX_BACKOFF_MS = 10000L;
    private static final String CONSENT_CHECK_ENDPOINT = "/rest/v1/rpc/check_consent";
    private static final String CONSENT_REGISTER_ENDPOINT = "/rest/v1/rpc/register_consent";
    private static final String TELEMETRY_ENDPOINT = "/rest/v1/quest_telemetry";
    private final HttpClient httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).version(HttpClient.Version.HTTP_2).executor(Executors.newFixedThreadPool(4)).build();

    private SupabaseClient() {
    }

    public static SupabaseClient getInstance() {
        return instance;
    }

    public boolean isConfigured() {
        return SupabaseConfig.isConfigured();
    }

    public boolean isHealthy() {
        if (!this.isConfigured()) {
            return false;
        }
        try {
            String url = SupabaseConfig.getSupabaseUrl();
            String key = SupabaseConfig.getSupabaseAnonKey();
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url + "/rest/v1/")).header("apikey", key).GET().timeout(Duration.ofSeconds(2L)).build();
            HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
            return response.statusCode() == 200 || response.statusCode() == 404;
        }
        catch (Exception e) {
            return false;
        }
    }

    public ConsentStatus checkConsent(String hwid) throws IOException, InterruptedException {
        if (!this.isConfigured()) {
            Questory.LOGGER.warn("Supabase not configured, cannot check consent");
            return new ConsentStatus(false, 0);
        }
        return this.retryWithBackoff(() -> this.checkConsentInternal(hwid), 3);
    }

    private ConsentStatus checkConsentInternal(String hwid) throws IOException, InterruptedException {
        JsonArray resultArray;
        String url = SupabaseConfig.getSupabaseUrl();
        String key = SupabaseConfig.getSupabaseAnonKey();
        String endpoint = url + CONSENT_CHECK_ENDPOINT;
        JsonObject body = new JsonObject();
        body.addProperty("p_hwid", hwid);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(endpoint)).header("Content-Type", "application/json").header("apikey", key).header("Authorization", "Bearer " + key).POST(HttpRequest.BodyPublishers.ofString(GSON.toJson((JsonElement)body))).timeout(Duration.ofSeconds(5L)).build();
        if (Questory.LOGGER.isDebugEnabled()) {
            Questory.LOGGER.debug("POST {}: {}", (Object)endpoint, (Object)GSON.toJson((JsonElement)body));
        }
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        if (Questory.LOGGER.isDebugEnabled()) {
            Questory.LOGGER.debug("Response {}: {}", (Object)response.statusCode(), (Object)response.body());
        }
        if (response.statusCode() == 200 && (resultArray = JsonParser.parseString((String)response.body()).getAsJsonArray()).size() > 0) {
            JsonObject result = resultArray.get(0).getAsJsonObject();
            boolean consented = result.has("consented") && result.get("consented").getAsBoolean();
            int version = result.has("consent_version") ? result.get("consent_version").getAsInt() : 0;
            return new ConsentStatus(consented, version);
        }
        return new ConsentStatus(false, 0);
    }

    public void registerConsent(String hwid, boolean consented) throws IOException, InterruptedException {
        if (!this.isConfigured()) {
            Questory.LOGGER.warn("Supabase not configured, cannot register consent");
            return;
        }
        this.retryWithBackoff(() -> {
            this.registerConsentInternal(hwid, consented);
            return null;
        }, 3);
    }

    private void registerConsentInternal(String hwid, boolean consented) throws IOException, InterruptedException {
        String url = SupabaseConfig.getSupabaseUrl();
        String key = SupabaseConfig.getSupabaseAnonKey();
        String endpoint = url + CONSENT_REGISTER_ENDPOINT;
        JsonObject body = new JsonObject();
        body.addProperty("p_hwid", hwid);
        body.addProperty("p_consented", Boolean.valueOf(consented));
        body.addProperty("p_consent_version", (Number)1);
        try {
            String userAgent = "Minecraft/" + SharedConstants.m_183709_().m_132493_() + " Questory/" + this.getModVersion();
            body.addProperty("p_user_agent", userAgent);
        }
        catch (Exception userAgent) {
            // empty catch block
        }
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(endpoint)).header("Content-Type", "application/json").header("apikey", key).header("Authorization", "Bearer " + key).POST(HttpRequest.BodyPublishers.ofString(GSON.toJson((JsonElement)body))).timeout(Duration.ofSeconds(5L)).build();
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200 && response.statusCode() != 204) {
            throw new IOException("Failed to register consent: " + response.statusCode() + " - " + response.body());
        }
        Questory.LOGGER.info("Consent registered: {} for HWID: {}", (Object)consented, (Object)(hwid.substring(0, 8) + "..."));
    }

    public void uploadEvents(List<TelemetryEvent> events) throws IOException, InterruptedException {
        if (!this.isConfigured()) {
            Questory.LOGGER.warn("Supabase not configured, cannot upload events");
            return;
        }
        if (events.isEmpty()) {
            return;
        }
        this.retryWithBackoff(() -> {
            this.uploadEventsInternal(events);
            return null;
        }, 3);
    }

    private void uploadEventsInternal(List<TelemetryEvent> events) throws IOException, InterruptedException {
        String url = SupabaseConfig.getSupabaseUrl();
        String key = SupabaseConfig.getSupabaseAnonKey();
        JsonArray jsonArray = new JsonArray();
        for (TelemetryEvent event : events) {
            jsonArray.add((JsonElement)event.toJson());
        }
        String endpoint = url + TELEMETRY_ENDPOINT;
        String body = GSON.toJson((JsonElement)jsonArray);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(endpoint)).header("Content-Type", "application/json").header("apikey", key).header("Authorization", "Bearer " + key).header("Prefer", "return=minimal").POST(HttpRequest.BodyPublishers.ofString(body)).timeout(Duration.ofSeconds(30L)).build();
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 201 && response.statusCode() != 200) {
            throw new IOException("Failed to upload events: " + response.statusCode() + " - " + response.body());
        }
        Questory.LOGGER.debug("Successfully uploaded {} events to centralized database", (Object)events.size());
    }

    public void deletePlayerData(String hwid) throws IOException, InterruptedException {
        if (!this.isConfigured()) {
            Questory.LOGGER.warn("Supabase not configured, cannot delete player data");
            return;
        }
        this.retryWithBackoff(() -> {
            this.deletePlayerDataInternal(hwid);
            return null;
        }, 3);
    }

    private void deletePlayerDataInternal(String hwid) throws IOException, InterruptedException {
        String url = SupabaseConfig.getSupabaseUrl();
        String key = SupabaseConfig.getSupabaseAnonKey();
        UUID playerUuid = UUID.nameUUIDFromBytes(hwid.getBytes());
        String deleteDataEndpoint = url + "/rest/v1/rpc/delete_player_data";
        JsonObject dataBody = new JsonObject();
        dataBody.addProperty("p_uuid", playerUuid.toString());
        HttpRequest dataRequest = HttpRequest.newBuilder().uri(URI.create(deleteDataEndpoint)).header("Content-Type", "application/json").header("apikey", key).header("Authorization", "Bearer " + key).POST(HttpRequest.BodyPublishers.ofString(GSON.toJson((JsonElement)dataBody))).timeout(Duration.ofSeconds(5L)).build();
        this.httpClient.send(dataRequest, HttpResponse.BodyHandlers.ofString());
        String deleteConsentEndpoint = url + "/rest/v1/consent_registry?hwid=eq." + this.urlEncode(hwid);
        HttpRequest consentRequest = HttpRequest.newBuilder().uri(URI.create(deleteConsentEndpoint)).header("apikey", key).header("Authorization", "Bearer " + key).DELETE().timeout(Duration.ofSeconds(5L)).build();
        HttpResponse<String> consentResponse = this.httpClient.send(consentRequest, HttpResponse.BodyHandlers.ofString());
        if (consentResponse.statusCode() != 200 && consentResponse.statusCode() != 204) {
            throw new IOException("Failed to delete consent record: " + consentResponse.statusCode());
        }
        Questory.LOGGER.info("Deleted all data for HWID: {}", (Object)(hwid.substring(0, 8) + "..."));
    }

    public HeatmapData fetchHeatmapData(String modpackId, String modpackVersion) throws IOException, InterruptedException {
        if (!this.isConfigured()) {
            Questory.LOGGER.warn("Supabase not configured, cannot fetch heatmap data");
            return new HeatmapData();
        }
        String url = SupabaseConfig.getSupabaseUrl();
        String key = SupabaseConfig.getSupabaseAnonKey();
        String endpoint = String.format("%s/rest/v1/rpc/get_quest_heatmap", url);
        JsonObject body = new JsonObject();
        body.addProperty("p_modpack_id", modpackId);
        body.addProperty("p_modpack_version", modpackVersion);
        Questory.LOGGER.info("Fetching heatmap data for modpack: {} v{}", (Object)modpackId, (Object)modpackVersion);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(endpoint)).header("Content-Type", "application/json").header("apikey", key).header("Authorization", "Bearer " + key).POST(HttpRequest.BodyPublishers.ofString(GSON.toJson((JsonElement)body))).timeout(Duration.ofSeconds(15L)).build();
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        Questory.LOGGER.info("Heatmap query response: {} - {}", (Object)response.statusCode(), (Object)response.body());
        if (response.statusCode() != 200) {
            throw new IOException("Failed to fetch heatmap data: " + response.statusCode() + " - " + response.body());
        }
        HeatmapData heatmapData = new HeatmapData();
        try {
            JsonArray jsonArray = JsonParser.parseString((String)response.body()).getAsJsonArray();
            Questory.LOGGER.info("Heatmap data array size: {}", (Object)jsonArray.size());
            for (int i = 0; i < jsonArray.size(); ++i) {
                JsonObject row = jsonArray.get(i).getAsJsonObject();
                String questId = row.get("quest_id").getAsString();
                int completionCount = row.has("completion_count") ? row.get("completion_count").getAsInt() : 0;
                int uniquePlayers = row.has("unique_players") ? row.get("unique_players").getAsInt() : 0;
                double avgTime = row.has("avg_completion_time_seconds") ? row.get("avg_completion_time_seconds").getAsDouble() : 0.0;
                heatmapData.addQuestData(questId, completionCount, uniquePlayers, avgTime);
                Questory.LOGGER.debug("Added heatmap data for quest: {} (count: {}, players: {}, avg: {}s)", (Object)questId, (Object)completionCount, (Object)uniquePlayers, (Object)avgTime);
            }
        }
        catch (Exception e) {
            Questory.LOGGER.error("Failed to parse heatmap data", (Throwable)e);
            throw new IOException("Failed to parse heatmap response", e);
        }
        Questory.LOGGER.info("Fetched heatmap data for {} quests", (Object)heatmapData.getQuestCount());
        return heatmapData;
    }

    private <T> T retryWithBackoff(SupplierWithException<T> operation, int maxRetries) throws IOException, InterruptedException {
        Exception lastException = null;
        for (int attempt = 0; attempt < maxRetries; ++attempt) {
            try {
                return operation.get();
            }
            catch (Exception e) {
                lastException = e;
                if (attempt >= maxRetries - 1) continue;
                long backoff = Math.min(1000L * (1L << attempt), 10000L);
                Questory.LOGGER.warn("Attempt {} failed, retrying in {}ms: {}", (Object)(attempt + 1), (Object)backoff, (Object)e.getMessage());
                Thread.sleep(backoff);
                continue;
            }
        }
        if (lastException instanceof IOException) {
            throw (IOException)lastException;
        }
        if (lastException instanceof InterruptedException) {
            throw (InterruptedException)lastException;
        }
        throw new IOException("Operation failed after " + maxRetries + " retries", lastException);
    }

    private String urlEncode(String value) {
        try {
            return URLEncoder.encode(value, "UTF-8");
        }
        catch (Exception e) {
            return value;
        }
    }

    private String getModVersion() {
        try {
            return "1.0.0";
        }
        catch (Exception e) {
            return "unknown";
        }
    }

    public static class ConsentStatus {
        private final boolean consented;
        private final int consentVersion;

        public ConsentStatus(boolean consented, int consentVersion) {
            this.consented = consented;
            this.consentVersion = consentVersion;
        }

        public boolean isConsented() {
            return this.consented;
        }

        public int getConsentVersion() {
            return this.consentVersion;
        }

        public boolean needsUpdate(int currentVersion) {
            return this.consentVersion < currentVersion;
        }
    }

    @FunctionalInterface
    private static interface SupplierWithException<T> {
        public T get() throws Exception;
    }
}

