/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.storage.backend.sqlite;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.texboobcat.questory.quest.QuestProgress;
import org.texboobcat.questory.storage.backend.IPlayerProgressBackend;

public class SqlitePlayerProgressBackend
implements IPlayerProgressBackend {
    private static final Gson GSON = new GsonBuilder().create();
    private final Path databasePath;
    private Connection connection;

    public SqlitePlayerProgressBackend(Path dataPath, String databaseFileName) {
        this.databasePath = dataPath.resolve(databaseFileName);
    }

    @Override
    public boolean connect() {
        try {
            String url = "jdbc:sqlite:" + String.valueOf(this.databasePath.toAbsolutePath());
            this.connection = DriverManager.getConnection(url);
            try (Statement stmt = this.connection.createStatement();){
                stmt.execute("PRAGMA journal_mode=WAL");
                stmt.execute("PRAGMA synchronous=NORMAL");
                stmt.execute("PRAGMA cache_size = -64000");
                stmt.execute("PRAGMA temp_store = MEMORY");
            }
            this.createTables();
            System.out.println("[Questory] SQLite player progress backend connected: " + String.valueOf(this.databasePath));
            return true;
        }
        catch (SQLException e) {
            System.err.println("[Questory] Failed to connect to SQLite database: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    private void createTables() throws SQLException {
        String createPlayerMetadata = "    CREATE TABLE IF NOT EXISTS player_metadata (\n        player_id TEXT PRIMARY KEY,\n        completed_quests TEXT,\n        claimed_rewards TEXT,\n        daily_history TEXT,\n        last_daily_id TEXT,\n        last_updated INTEGER DEFAULT (strftime('%s', 'now'))\n    )\n";
        String createPlayerProgress = "    CREATE TABLE IF NOT EXISTS player_progress (\n        player_id TEXT NOT NULL,\n        quest_id TEXT NOT NULL,\n        requirement_key TEXT NOT NULL,\n        progress INTEGER DEFAULT 0,\n        last_updated INTEGER DEFAULT (strftime('%s', 'now')),\n        PRIMARY KEY (player_id, quest_id, requirement_key)\n    )\n";
        String createIndexes = "    CREATE INDEX IF NOT EXISTS idx_player_quest ON player_progress(player_id, quest_id)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createPlayerMetadata);
            stmt.execute(createPlayerProgress);
            stmt.execute(createIndexes);
        }
    }

    @Override
    public void disconnect() {
        if (this.connection != null) {
            try {
                this.connection.close();
                System.out.println("[Questory] SQLite player progress backend disconnected");
            }
            catch (SQLException e) {
                System.err.println("[Questory] Error closing SQLite connection: " + e.getMessage());
            }
        }
    }

    @Override
    public boolean isConnected() {
        try {
            return this.connection != null && !this.connection.isClosed();
        }
        catch (SQLException e) {
            return false;
        }
    }

    @Override
    public boolean healthCheck() {
        boolean bl;
        block9: {
            if (!this.isConnected()) {
                return false;
            }
            Statement stmt = this.connection.createStatement();
            try {
                stmt.execute("SELECT 1");
                bl = true;
                if (stmt == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (stmt != null) {
                        try {
                            stmt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (SQLException e) {
                    return false;
                }
            }
            stmt.close();
        }
        return bl;
    }

    @Override
    public String getBackendType() {
        return "SQLite";
    }

    @Override
    public QuestProgress loadProgress(UUID playerId) {
        try {
            QuestProgress progress = new QuestProgress(playerId);
            String metadataQuery = "SELECT * FROM player_metadata WHERE player_id = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(metadataQuery);){
                String completedJson;
                stmt.setString(1, playerId.toString());
                ResultSet rs = stmt.executeQuery();
                if (rs.next() && (completedJson = rs.getString("completed_quests")) != null && !completedJson.isEmpty()) {
                    JsonObject json = new JsonObject();
                    json.addProperty("playerId", playerId.toString());
                    json.add("completed", JsonParser.parseString((String)completedJson));
                    json.add("claimedRewards", JsonParser.parseString((String)rs.getString("claimed_rewards")));
                    json.add("dailyHistory", JsonParser.parseString((String)rs.getString("daily_history")));
                    json.addProperty("lastCompletedDailyId", rs.getString("last_daily_id"));
                    json.add("progress", (JsonElement)this.loadProgressData(playerId));
                    json.add("completionTimes", (JsonElement)new JsonObject());
                    progress = QuestProgress.fromJson(json);
                }
            }
            return progress;
        }
        catch (SQLException e) {
            System.err.println("[Questory] Failed to load progress for player " + String.valueOf(playerId) + ": " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    private JsonObject loadProgressData(UUID playerId) throws SQLException {
        JsonObject progressObj = new JsonObject();
        String query = "SELECT quest_id, requirement_key, progress FROM player_progress WHERE player_id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                String questId = rs.getString("quest_id");
                String reqKey = rs.getString("requirement_key");
                int progress = rs.getInt("progress");
                if (!progressObj.has(questId)) {
                    progressObj.add(questId, (JsonElement)new JsonObject());
                }
                progressObj.getAsJsonObject(questId).addProperty(reqKey, (Number)progress);
            }
        }
        return progressObj;
    }

    @Override
    public boolean saveProgress(QuestProgress progress) {
        try {
            this.connection.setAutoCommit(false);
            String metadataUpsert = "    INSERT OR REPLACE INTO player_metadata\n    (player_id, completed_quests, claimed_rewards, daily_history, last_daily_id, last_updated)\n    VALUES (?, ?, ?, ?, ?, strftime('%s', 'now'))\n";
            JsonObject json = progress.toJson();
            try (PreparedStatement stmt = this.connection.prepareStatement(metadataUpsert);){
                stmt.setString(1, progress.getPlayerId().toString());
                stmt.setString(2, json.has("completed") ? json.get("completed").toString() : "[]");
                stmt.setString(3, json.has("claimedRewards") ? json.get("claimedRewards").toString() : "[]");
                stmt.setString(4, json.has("dailyHistory") ? json.get("dailyHistory").toString() : "{}");
                stmt.setString(5, json.has("lastCompletedDailyId") ? json.get("lastCompletedDailyId").getAsString() : "");
                stmt.executeUpdate();
            }
            if (json.has("progress")) {
                JsonObject progressObj = json.getAsJsonObject("progress");
                String progressUpsert = "    INSERT INTO player_progress (player_id, quest_id, requirement_key, progress, last_updated)\n    VALUES (?, ?, ?, ?, strftime('%s', 'now'))\n    ON CONFLICT(player_id, quest_id, requirement_key)\n    DO UPDATE SET progress = excluded.progress, last_updated = strftime('%s', 'now')\n";
                try (PreparedStatement stmt = this.connection.prepareStatement(progressUpsert);){
                    for (String questId : progressObj.keySet()) {
                        JsonObject questProgress = progressObj.getAsJsonObject(questId);
                        for (String reqKey : questProgress.keySet()) {
                            stmt.setString(1, progress.getPlayerId().toString());
                            stmt.setString(2, questId);
                            stmt.setString(3, reqKey);
                            stmt.setInt(4, questProgress.get(reqKey).getAsInt());
                            stmt.addBatch();
                        }
                    }
                    stmt.executeBatch();
                }
            }
            this.connection.commit();
            this.connection.setAutoCommit(true);
            return true;
        }
        catch (SQLException e) {
            System.err.println("[Questory] Failed to save progress for player " + String.valueOf(progress.getPlayerId()) + ": " + e.getMessage());
            e.printStackTrace();
            try {
                this.connection.rollback();
                this.connection.setAutoCommit(true);
            }
            catch (SQLException ex) {
                ex.printStackTrace();
            }
            return false;
        }
    }

    @Override
    public boolean deleteProgress(UUID playerId) {
        try {
            String deleteMetadata = "DELETE FROM player_metadata WHERE player_id = ?";
            String deleteProgress = "DELETE FROM player_progress WHERE player_id = ?";
            try (PreparedStatement stmt1 = this.connection.prepareStatement(deleteMetadata);
                 PreparedStatement stmt2 = this.connection.prepareStatement(deleteProgress);){
                stmt1.setString(1, playerId.toString());
                stmt2.setString(1, playerId.toString());
                stmt1.executeUpdate();
                stmt2.executeUpdate();
            }
            return true;
        }
        catch (SQLException e) {
            System.err.println("[Questory] Failed to delete progress for player " + String.valueOf(playerId) + ": " + e.getMessage());
            return false;
        }
    }

    @Override
    public List<UUID> getAllPlayerIds() {
        ArrayList<UUID> playerIds = new ArrayList<UUID>();
        try {
            String query = "SELECT DISTINCT player_id FROM player_metadata";
            try (Statement stmt = this.connection.createStatement();
                 ResultSet rs = stmt.executeQuery(query);){
                while (rs.next()) {
                    try {
                        playerIds.add(UUID.fromString(rs.getString("player_id")));
                    }
                    catch (IllegalArgumentException e) {
                        System.err.println("[Questory] Invalid UUID in database: " + rs.getString("player_id"));
                    }
                }
            }
        }
        catch (SQLException e) {
            System.err.println("[Questory] Failed to get all player IDs: " + e.getMessage());
        }
        return playerIds;
    }
}

