/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.config;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Stream;

public class ConfigBackupManager {
    private static final String BACKUP_DIR_NAME = "questory-backups";
    private static final String BACKUP_EXTENSION = ".backup";
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss");
    private static final int MAX_BACKUPS = 10;
    private final Path backupDir;

    public ConfigBackupManager(Path configPath) {
        this.backupDir = configPath.resolve(BACKUP_DIR_NAME);
        try {
            Files.createDirectories(this.backupDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            System.err.println("[Questory] Failed to create backup directory: " + e.getMessage());
        }
    }

    public Path createBackup(Path configFile, String reason) {
        if (!Files.exists(configFile, new LinkOption[0])) {
            System.out.println("[Questory] No config file to backup at: " + String.valueOf(configFile));
            return null;
        }
        try {
            String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMAT);
            String fileName = configFile.getFileName().toString();
            String backupFileName = fileName + "." + timestamp;
            if (reason != null && !reason.isEmpty()) {
                backupFileName = backupFileName + "." + this.sanitizeReason(reason);
            }
            backupFileName = backupFileName + BACKUP_EXTENSION;
            Path backupFile = this.backupDir.resolve(backupFileName);
            Files.copy(configFile, backupFile, StandardCopyOption.REPLACE_EXISTING);
            System.out.println("[Questory] Created config backup: " + backupFileName);
            this.cleanupOldBackups();
            return backupFile;
        }
        catch (IOException e) {
            System.err.println("[Questory] Failed to create config backup: " + e.getMessage());
            return null;
        }
    }

    public Path createBackup(Path configFile) {
        return this.createBackup(configFile, "manual");
    }

    public boolean restoreBackup(Path backupFile, Path configFile) {
        if (!Files.exists(backupFile, new LinkOption[0])) {
            System.err.println("[Questory] Backup file not found: " + String.valueOf(backupFile));
            return false;
        }
        try {
            if (Files.exists(configFile, new LinkOption[0])) {
                this.createBackup(configFile, "pre-restore");
            }
            Files.copy(backupFile, configFile, StandardCopyOption.REPLACE_EXISTING);
            System.out.println("[Questory] Restored config from backup: " + String.valueOf(backupFile.getFileName()));
            return true;
        }
        catch (IOException e) {
            System.err.println("[Questory] Failed to restore config backup: " + e.getMessage());
            return false;
        }
    }

    public List<BackupInfo> listBackups() {
        ArrayList<BackupInfo> backups = new ArrayList<BackupInfo>();
        try (Stream<Path> files = Files.list(this.backupDir);){
            files.filter(path -> path.toString().endsWith(BACKUP_EXTENSION)).forEach(path -> {
                BackupInfo info = this.parseBackupInfo((Path)path);
                if (info != null) {
                    backups.add(info);
                }
            });
        }
        catch (IOException e) {
            System.err.println("[Questory] Failed to list backups: " + e.getMessage());
        }
        backups.sort(Comparator.comparing(BackupInfo::getTimestamp).reversed());
        return backups;
    }

    private void cleanupOldBackups() {
        List<BackupInfo> backups = this.listBackups();
        if (backups.size() > 10) {
            for (int i = 10; i < backups.size(); ++i) {
                try {
                    Files.deleteIfExists(backups.get(i).getPath());
                    System.out.println("[Questory] Deleted old backup: " + String.valueOf(backups.get(i).getPath().getFileName()));
                    continue;
                }
                catch (IOException e) {
                    System.err.println("[Questory] Failed to delete old backup: " + e.getMessage());
                }
            }
        }
    }

    private BackupInfo parseBackupInfo(Path backupFile) {
        String fileName = backupFile.getFileName().toString();
        try {
            fileName = fileName.substring(0, fileName.length() - BACKUP_EXTENSION.length());
            String[] parts = fileName.split("\\.");
            if (parts.length < 3) {
                return null;
            }
            String timestamp = parts[parts.length - 2] + "_" + parts[parts.length - 1];
            String reason = parts.length > 3 ? parts[parts.length - 3] : "unknown";
            LocalDateTime dateTime = LocalDateTime.parse(timestamp.replace("_", "T"), DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss"));
            return new BackupInfo(backupFile, dateTime, reason);
        }
        catch (Exception e) {
            System.err.println("[Questory] Failed to parse backup info for: " + fileName);
            return null;
        }
    }

    private String sanitizeReason(String reason) {
        return reason.replaceAll("[^a-zA-Z0-9_-]", "_").toLowerCase();
    }

    public static class BackupInfo {
        private final Path path;
        private final LocalDateTime timestamp;
        private final String reason;

        public BackupInfo(Path path, LocalDateTime timestamp, String reason) {
            this.path = path;
            this.timestamp = timestamp;
            this.reason = reason;
        }

        public Path getPath() {
            return this.path;
        }

        public LocalDateTime getTimestamp() {
            return this.timestamp;
        }

        public String getReason() {
            return this.reason;
        }

        public String toString() {
            return this.timestamp.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")) + " - " + this.reason + " (" + String.valueOf(this.path.getFileName()) + ")";
        }
    }
}

