/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.storage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;
import org.texboobcat.questory.quest.ChapterData;
import org.texboobcat.questory.quest.GroupNode;
import org.texboobcat.questory.quest.ItemRequirement;
import org.texboobcat.questory.quest.ItemReward;
import org.texboobcat.questory.quest.Quest;
import org.texboobcat.questory.quest.XpReward;

public class QuestStorage {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private final Path questsPath;
    private final Path chaptersPath;
    private final Path groupTreeFile;

    public QuestStorage(Path dataPath) {
        this.questsPath = dataPath.resolve("quests");
        this.chaptersPath = dataPath.resolve("chapters");
        this.groupTreeFile = dataPath.resolve("group_tree.json");
        try {
            Files.createDirectories(this.questsPath, new FileAttribute[0]);
            Files.createDirectories(this.chaptersPath, new FileAttribute[0]);
        }
        catch (IOException e) {
            System.err.println("Failed to create data directories: " + e.getMessage());
        }
    }

    public Map<String, Quest> loadAllQuests() {
        HashMap<String, Quest> quests = new HashMap<String, Quest>();
        try {
            if (!Files.exists(this.questsPath, new LinkOption[0])) {
                System.out.println("Quests directory does not exist, creating it...");
                Files.createDirectories(this.questsPath, new FileAttribute[0]);
                return quests;
            }
            this.loadQuestsFromDirectory(this.questsPath, quests);
            System.out.println("Loaded " + quests.size() + " quests");
        }
        catch (IOException e) {
            System.err.println("Error loading quests: " + e.getMessage());
            e.printStackTrace();
        }
        return quests;
    }

    private void loadQuestsFromDirectory(Path directory, Map<String, Quest> quests) throws IOException {
        if (!Files.exists(directory, new LinkOption[0]) || !Files.isDirectory(directory, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> paths = Files.list(directory);){
            paths.forEach(path -> {
                try {
                    Quest quest;
                    if (Files.isDirectory(path, new LinkOption[0])) {
                        this.loadQuestsFromDirectory((Path)path, quests);
                    } else if (path.toString().endsWith(".json") && (quest = this.loadQuestFromFile((Path)path)) != null) {
                        quests.put(quest.getId(), quest);
                    }
                }
                catch (IOException e) {
                    System.err.println("Error processing path " + String.valueOf(path) + ": " + e.getMessage());
                }
            });
        }
    }

    public Quest loadQuestFromFile(Path file) {
        try {
            String content = Files.readString(file);
            JsonObject json = JsonParser.parseString((String)content).getAsJsonObject();
            return Quest.fromJson(json);
        }
        catch (Exception e) {
            System.err.println("Failed to load quest from " + String.valueOf(file) + ": " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    public boolean saveQuest(Quest quest) {
        try {
            Path questFile = this.getQuestFile(quest);
            Files.createDirectories(questFile.getParent(), new FileAttribute[0]);
            String json = GSON.toJson((JsonElement)quest.toJson());
            Files.writeString(questFile, (CharSequence)json, new OpenOption[0]);
            return true;
        }
        catch (IOException e) {
            System.err.println("Failed to save quest " + quest.getId() + ": " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    public void saveAllQuests(Collection<Quest> quests) {
        for (Quest quest : quests) {
            this.saveQuest(quest);
        }
    }

    public Map<String, ChapterData> loadAllChapters() {
        HashMap<String, ChapterData> chapters = new HashMap<String, ChapterData>();
        try {
            if (!Files.exists(this.chaptersPath, new LinkOption[0])) {
                Files.createDirectories(this.chaptersPath, new FileAttribute[0]);
                return chapters;
            }
            try (Stream<Path> paths = Files.list(this.chaptersPath);){
                paths.filter(p -> p.toString().endsWith(".json")).forEach(p -> {
                    try {
                        String content = Files.readString(p);
                        JsonObject json = JsonParser.parseString((String)content).getAsJsonObject();
                        ChapterData cd = ChapterData.fromJson(json);
                        chapters.put(cd.getTitle(), cd);
                    }
                    catch (Exception e) {
                        System.err.println("Failed to load chapter data from " + String.valueOf(p) + ": " + e.getMessage());
                    }
                });
            }
        }
        catch (IOException e) {
            System.err.println("Error loading chapters: " + e.getMessage());
        }
        return chapters;
    }

    public boolean saveChapter(ChapterData chapter) {
        try {
            Path file = this.getChapterFile(chapter);
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
            String json = GSON.toJson((JsonElement)chapter.toJson());
            Files.writeString(file, (CharSequence)json, new OpenOption[0]);
            return true;
        }
        catch (IOException e) {
            System.err.println("Failed to save chapter " + chapter.getTitle() + ": " + e.getMessage());
            return false;
        }
    }

    public void saveAllChapters(Collection<ChapterData> chapters) {
        for (ChapterData c : chapters) {
            this.saveChapter(c);
        }
    }

    public boolean saveGroupTree(GroupNode root) {
        try {
            if (root == null) {
                return false;
            }
            Files.createDirectories(this.groupTreeFile.getParent(), new FileAttribute[0]);
            String json = GSON.toJson((JsonElement)root.toJson());
            Files.writeString(this.groupTreeFile, (CharSequence)json, new OpenOption[0]);
            return true;
        }
        catch (IOException e) {
            System.err.println("Failed to save group tree: " + e.getMessage());
            return false;
        }
    }

    public GroupNode loadGroupTree() {
        try {
            if (!Files.exists(this.groupTreeFile, new LinkOption[0])) {
                return null;
            }
            String content = Files.readString(this.groupTreeFile);
            JsonObject json = JsonParser.parseString((String)content).getAsJsonObject();
            return GroupNode.fromJson(json);
        }
        catch (Exception e) {
            System.err.println("Failed to load group tree: " + e.getMessage());
            return null;
        }
    }

    private Path getChapterFile(ChapterData chapter) {
        String name = chapter.getTitle().replaceAll("[^a-zA-Z0-9-_]", "_");
        return this.chaptersPath.resolve(name + ".json");
    }

    public boolean deleteQuest(String questId) {
        try {
            Path questFile = this.questsPath.resolve(questId + ".json");
            if (Files.exists(questFile, new LinkOption[0])) {
                Files.delete(questFile);
                return true;
            }
        }
        catch (IOException e) {
            System.err.println("Failed to delete quest " + questId + ": " + e.getMessage());
        }
        return false;
    }

    private Path getQuestFile(Quest quest) {
        String group = quest.getGroup();
        if (group != null && !group.equals("default")) {
            return this.questsPath.resolve(group).resolve(quest.getId() + ".json");
        }
        return this.questsPath.resolve(quest.getId() + ".json");
    }

    public void createExampleQuest() {
        Quest exampleQuest = new Quest("quest_get_wood", "Getting Started", "Punch a tree to collect wood.");
        exampleQuest.setIcon("minecraft:oak_log");
        ItemRequirement woodReq = new ItemRequirement("minecraft:oak_log", 1);
        exampleQuest.addRequirement(woodReq);
        ItemReward axeReward = new ItemReward("minecraft:stone_axe", 1);
        exampleQuest.addReward(axeReward);
        XpReward xpReward = new XpReward(100);
        exampleQuest.addReward(xpReward);
        this.saveQuest(exampleQuest);
        System.out.println("Created example quest: quest_get_wood");
    }
}

