/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.team;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.texboobcat.questory.api.events.TeamCreatedEvent;
import org.texboobcat.questory.api.events.TeamEventBus;
import org.texboobcat.questory.api.events.TeamMemberJoinedEvent;
import org.texboobcat.questory.config.QuestoryConfig;
import org.texboobcat.questory.quest.QuestProgress;
import org.texboobcat.questory.storage.TeamStorage;
import org.texboobcat.questory.team.BuiltInTeamProvider;
import org.texboobcat.questory.team.ITeamProvider;
import org.texboobcat.questory.team.QuestTeam;

public class TeamManager {
    private static TeamManager instance;
    private final Map<UUID, QuestTeam> teams = new HashMap<UUID, QuestTeam>();
    private final Map<UUID, UUID> playerTeamMap = new HashMap<UUID, UUID>();
    private final Map<UUID, Set<UUID>> pendingInvites = new HashMap<UUID, Set<UUID>>();
    private final List<ITeamProvider> teamProviders = new ArrayList<ITeamProvider>();
    private ITeamProvider activeProvider;
    private final TeamStorage storage;
    private final boolean useExternalProvider;

    private TeamManager(Path dataPath, boolean useExternalProvider) {
        this.storage = new TeamStorage(dataPath);
        this.useExternalProvider = useExternalProvider;
        this.registerProvider(new BuiltInTeamProvider(this));
    }

    public static void initialize(Path dataPath, boolean useExternalProvider) {
        if (instance == null) {
            instance = new TeamManager(dataPath, useExternalProvider);
            instance.loadAll();
        }
    }

    public static TeamManager getInstance() {
        if (instance == null) {
            throw new IllegalStateException("TeamManager not initialized!");
        }
        return instance;
    }

    public void registerProvider(ITeamProvider provider) {
        if (provider.isAvailable()) {
            this.teamProviders.add(provider);
            this.teamProviders.sort(Comparator.comparingInt(ITeamProvider::getPriority).reversed());
            this.updateActiveProvider();
            System.out.println("[Questory] Registered team provider: " + provider.getProviderId() + " (priority: " + provider.getPriority() + ")");
        }
    }

    private void updateActiveProvider() {
        QuestoryConfig cfg = QuestoryConfig.getInstance();
        if (this.useExternalProvider) {
            if (cfg.teamPreferredProvider != null && !cfg.teamPreferredProvider.isEmpty()) {
                this.activeProvider = this.teamProviders.stream().filter(p -> p.getProviderId().equals(cfg.teamPreferredProvider)).findFirst().orElse(null);
                if (this.activeProvider != null) {
                    System.out.println("[Questory] Active team provider: " + this.activeProvider.getProviderId() + " (preferred)");
                    return;
                }
            }
            this.activeProvider = this.teamProviders.stream().filter(p -> !p.getProviderId().equals("questory:builtin")).findFirst().orElse(null);
            if (this.activeProvider == null && cfg.teamFallbackToBuiltIn) {
                this.activeProvider = this.getBuiltInProvider();
            }
        } else {
            this.activeProvider = this.getBuiltInProvider();
        }
        if (this.activeProvider != null) {
            System.out.println("[Questory] Active team provider: " + this.activeProvider.getProviderId());
        } else {
            System.err.println("[Questory] No team provider available!");
        }
    }

    private ITeamProvider getBuiltInProvider() {
        return this.teamProviders.stream().filter(p -> p.getProviderId().equals("questory:builtin")).findFirst().orElseThrow(() -> new IllegalStateException("Built-in provider not found"));
    }

    public ITeamProvider getActiveProvider() {
        return this.activeProvider;
    }

    public void loadAll() {
        this.teams.clear();
        this.playerTeamMap.clear();
        Map<UUID, QuestTeam> loadedTeams = this.storage.loadAllTeams();
        this.teams.putAll(loadedTeams);
        for (QuestTeam team : this.teams.values()) {
            for (UUID member : team.getMembers()) {
                this.playerTeamMap.put(member, team.getTeamId());
            }
        }
        System.out.println("[Questory] Loaded " + this.teams.size() + " teams");
    }

    public void saveAll() {
        this.storage.saveAllTeams(this.teams.values());
    }

    public QuestTeam createTeam(String name, UUID owner) {
        QuestoryConfig cfg = QuestoryConfig.getInstance();
        if (!cfg.teamAllowCreation) {
            System.out.println("[Questory] Team creation is disabled in config");
            return null;
        }
        UUID teamId = UUID.randomUUID();
        QuestTeam team = new QuestTeam(teamId, name, owner);
        this.teams.put(teamId, team);
        this.playerTeamMap.put(owner, teamId);
        this.storage.saveTeam(team);
        System.out.println("[Questory] Created team: " + name + " (owner: " + String.valueOf(owner) + ")");
        try {
            TeamEventBus.post(new TeamCreatedEvent(team));
        }
        catch (Throwable t) {
            System.err.println("[Questory] Error firing TeamCreatedEvent: " + t.getMessage());
        }
        return team;
    }

    public boolean deleteTeam(UUID teamId) {
        QuestTeam team = this.teams.remove(teamId);
        if (team != null) {
            for (UUID member : team.getMembers()) {
                this.playerTeamMap.remove(member);
            }
            this.storage.deleteTeam(teamId);
            System.out.println("[Questory] Deleted team: " + team.getTeamName());
            return true;
        }
        return false;
    }

    public QuestTeam getTeam(UUID teamId) {
        return this.teams.get(teamId);
    }

    public UUID getPlayerTeamId(UUID playerId) {
        return this.activeProvider.getTeamId(playerId);
    }

    public QuestTeam getPlayerTeam(UUID playerId) {
        UUID teamId = this.playerTeamMap.get(playerId);
        return teamId != null ? this.teams.get(teamId) : null;
    }

    public Set<UUID> getTeamMembers(UUID teamId) {
        return this.activeProvider.getTeamMembers(teamId);
    }

    public String getTeamName(UUID teamId) {
        return this.activeProvider.getTeamName(teamId);
    }

    public boolean isInTeam(UUID playerId) {
        return this.activeProvider.getTeamId(playerId) != null;
    }

    public boolean isTeamOwner(UUID playerId, UUID teamId) {
        return this.activeProvider.isTeamOwner(playerId, teamId);
    }

    public QuestTeam buildExternalTeamSnapshot(UUID teamId) {
        if (teamId == null || this.activeProvider == null) {
            return null;
        }
        QuestTeam builtIn = this.teams.get(teamId);
        if (builtIn != null) {
            return builtIn;
        }
        Object name = this.getTeamName(teamId);
        if (name == null || ((String)name).isEmpty()) {
            name = "Team " + teamId.toString().substring(0, 8);
        }
        HashSet<UUID> members = new HashSet<UUID>(this.getTeamMembers(teamId));
        UUID owner = null;
        for (UUID pid : members) {
            try {
                if (!this.isTeamOwner(pid, teamId)) continue;
                owner = pid;
                break;
            }
            catch (Throwable throwable) {
            }
        }
        if (owner == null && !members.isEmpty()) {
            owner = (UUID)members.iterator().next();
        }
        if (owner == null) {
            owner = teamId;
        }
        QuestTeam snapshot = new QuestTeam(teamId, (String)name, owner);
        for (UUID pid : members) {
            if (pid.equals(owner)) continue;
            snapshot.addMember(pid);
        }
        return snapshot;
    }

    public boolean invitePlayer(UUID teamId, UUID playerId) {
        QuestTeam team = this.teams.get(teamId);
        if (team == null || this.playerTeamMap.containsKey(playerId)) {
            return false;
        }
        this.pendingInvites.computeIfAbsent(playerId, k -> new HashSet()).add(teamId);
        System.out.println("[Questory] Player " + String.valueOf(playerId) + " invited to team " + team.getTeamName());
        return true;
    }

    public boolean acceptInvite(UUID playerId, UUID teamId) {
        QuestoryConfig cfg = QuestoryConfig.getInstance();
        Set<UUID> invites = this.pendingInvites.get(playerId);
        if (invites == null || !invites.contains(teamId)) {
            return false;
        }
        QuestTeam team = this.teams.get(teamId);
        if (team == null) {
            return false;
        }
        if (team.getMemberCount() >= cfg.teamMaxSize) {
            System.out.println("[Questory] Team is full (max: " + cfg.teamMaxSize + ")");
            return false;
        }
        this.leaveTeam(playerId);
        team.addMember(playerId);
        this.playerTeamMap.put(playerId, teamId);
        invites.remove(teamId);
        this.storage.saveTeam(team);
        System.out.println("[Questory] Player " + String.valueOf(playerId) + " joined team " + team.getTeamName());
        this.saveAll();
        try {
            TeamEventBus.post(new TeamMemberJoinedEvent(teamId, playerId));
        }
        catch (Throwable t) {
            System.err.println("[Questory] Error firing TeamMemberJoinedEvent: " + t.getMessage());
        }
        return true;
    }

    public boolean declineInvite(UUID playerId, UUID teamId) {
        Set<UUID> invites = this.pendingInvites.get(playerId);
        if (invites != null) {
            return invites.remove(teamId);
        }
        return false;
    }

    public boolean leaveTeam(UUID playerId) {
        QuestTeam team;
        UUID teamId = this.playerTeamMap.remove(playerId);
        if (teamId != null && (team = this.teams.get(teamId)) != null) {
            if (team.isOwner(playerId)) {
                Set<UUID> members = team.getMembers();
                members.remove(playerId);
                if (members.isEmpty()) {
                    this.deleteTeam(teamId);
                } else {
                    UUID newOwner = members.iterator().next();
                    team.setOwner(newOwner);
                    team.removeMember(playerId);
                    this.storage.saveTeam(team);
                }
            } else {
                team.removeMember(playerId);
                this.storage.saveTeam(team);
            }
            System.out.println("[Questory] Player " + String.valueOf(playerId) + " left team " + team.getTeamName());
            return true;
        }
        return false;
    }

    public boolean kickPlayer(UUID teamId, UUID playerId, UUID kickerId) {
        QuestTeam team = this.teams.get(teamId);
        if (team == null || !team.isOwner(kickerId) || playerId.equals(kickerId)) {
            return false;
        }
        if (team.removeMember(playerId)) {
            this.playerTeamMap.remove(playerId);
            this.storage.saveTeam(team);
            System.out.println("[Questory] Player " + String.valueOf(playerId) + " kicked from team " + team.getTeamName());
            return true;
        }
        return false;
    }

    public Set<UUID> getPendingInvites(UUID playerId) {
        Set<UUID> invites = this.pendingInvites.get(playerId);
        return invites != null ? new HashSet<UUID>(invites) : Collections.emptySet();
    }

    public Collection<QuestTeam> getAllTeams() {
        return new ArrayList<QuestTeam>(this.teams.values());
    }

    public QuestProgress getTeamProgress(UUID teamId, String questId) {
        QuestTeam team = this.teams.get(teamId);
        if (team != null) {
            QuestProgress progress = team.getSharedProgress(questId);
            if (progress == null) {
                progress = new QuestProgress(teamId);
                team.setSharedProgress(questId, progress);
            }
            return progress;
        }
        return null;
    }

    public void saveTeamProgress(UUID teamId) {
        QuestTeam team = this.teams.get(teamId);
        if (team != null) {
            this.storage.saveTeam(team);
        }
    }
}

