/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.telemetry;

import com.google.gson.JsonObject;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.texboobcat.questory.Questory;
import org.texboobcat.questory.modpack.ModpackDetector;
import org.texboobcat.questory.modpack.ModpackInfo;
import org.texboobcat.questory.telemetry.HardwareIdGenerator;
import org.texboobcat.questory.telemetry.SupabaseClient;
import org.texboobcat.questory.telemetry.TelemetryEvent;

public class TelemetryManager {
    private static final TelemetryManager instance = new TelemetryManager();
    private static final int MAX_BATCH_SIZE = 50;
    private static final int BATCH_INTERVAL_SECONDS = 60;
    private static final long MIN_EVENT_INTERVAL_MS = 10L;
    private final ConcurrentLinkedQueue<TelemetryEvent> eventQueue = new ConcurrentLinkedQueue();
    private final ExecutorService uploadExecutor = Executors.newFixedThreadPool(2, r -> {
        Thread t = new Thread(r, "QuestoryTelemetryUpload");
        t.setDaemon(true);
        return t;
    });
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread t = new Thread(r, "QuestoryTelemetryScheduler");
        t.setDaemon(true);
        return t;
    });
    private final UUID sessionId;
    private final String hardwareId;
    private final long sessionStartTime;
    private volatile ModpackInfo modpackInfo;
    private volatile SupabaseClient.ConsentStatus cachedConsentStatus = null;
    private volatile long lastConsentCheck = 0L;
    private static final long CONSENT_CHECK_INTERVAL_MS = 60000L;
    private final AtomicBoolean asyncCheckInProgress = new AtomicBoolean(false);
    private volatile boolean forceConsentPrompt = false;
    private final AtomicLong lastEventTime = new AtomicLong(0L);
    private final AtomicLong totalEvents = new AtomicLong();
    private final AtomicLong failedUploads = new AtomicLong();
    private final AtomicLong droppedEvents = new AtomicLong();

    private TelemetryManager() {
        this.sessionId = UUID.randomUUID();
        this.sessionStartTime = System.currentTimeMillis();
        this.hardwareId = HardwareIdGenerator.getHardwareId();
        Questory.LOGGER.info("Generated hardware ID for telemetry: {}", (Object)(this.hardwareId.substring(0, 8) + "..."));
        this.modpackInfo = ModpackDetector.detect();
        Questory.LOGGER.info("Detected modpack: {} v{}", (Object)this.modpackInfo.getName(), (Object)this.modpackInfo.getVersion());
        this.startScheduler();
        this.checkConsentStatusAsync();
        Runtime.getRuntime().addShutdownHook(new Thread(this::shutdown, "QuestoryTelemetryShutdown"));
        this.scheduler.scheduleAtFixedRate(this::logMetrics, 1L, 1L, TimeUnit.HOURS);
    }

    public static TelemetryManager getInstance() {
        return instance;
    }

    public String getHardwareId() {
        return this.hardwareId;
    }

    public ModpackInfo getModpackInfo() {
        return this.modpackInfo;
    }

    public ModpackInfo refreshModpackInfo() {
        ModpackDetector.clearCache();
        ModpackInfo freshInfo = ModpackDetector.detect();
        ModpackInfo oldInfo = this.modpackInfo;
        this.modpackInfo = freshInfo;
        Questory.LOGGER.info("Refreshed modpack info: {} (was: {})", (Object)freshInfo, (Object)oldInfo);
        return freshInfo;
    }

    public boolean isEnabled() {
        if (this.forceConsentPrompt) {
            return false;
        }
        if (this.cachedConsentStatus != null) {
            long now = System.currentTimeMillis();
            if (now - this.lastConsentCheck > 60000L) {
                this.checkConsentStatusAsync();
            }
            return this.cachedConsentStatus.isConsented();
        }
        this.checkConsentStatusAsync();
        return false;
    }

    public SupabaseClient.ConsentStatus getConsentStatus() {
        return this.cachedConsentStatus;
    }

    private void checkConsentStatusAsync() {
        if (!this.asyncCheckInProgress.compareAndSet(false, true)) {
            return;
        }
        this.uploadExecutor.submit(() -> {
            try {
                SupabaseClient.ConsentStatus status;
                this.cachedConsentStatus = status = SupabaseClient.getInstance().checkConsent(this.hardwareId);
                this.lastConsentCheck = System.currentTimeMillis();
                if (status.needsUpdate(1) && status.isConsented()) {
                    Questory.LOGGER.info("Consent version outdated, may need to re-prompt");
                }
            }
            catch (Exception e) {
                Questory.LOGGER.debug("Async consent check failed: {}", (Object)e.getMessage());
                if (this.cachedConsentStatus == null) {
                    // empty if block
                }
            }
            finally {
                this.asyncCheckInProgress.set(false);
            }
        });
    }

    public boolean shouldShowConsentScreen() {
        if (this.forceConsentPrompt) {
            return true;
        }
        if (this.cachedConsentStatus == null) {
            this.checkConsentStatusAsync();
            return false;
        }
        return !this.cachedConsentStatus.isConsented() || this.cachedConsentStatus.needsUpdate(1);
    }

    public void enableTelemetry() {
        this.uploadExecutor.submit(() -> {
            try {
                SupabaseClient.getInstance().registerConsent(this.hardwareId, true);
                this.cachedConsentStatus = new SupabaseClient.ConsentStatus(true, 1);
                this.lastConsentCheck = System.currentTimeMillis();
                this.forceConsentPrompt = false;
                Questory.LOGGER.info("Telemetry enabled by user");
                this.trackEvent(TelemetryEvent.EventType.SESSION_START, null, null);
            }
            catch (Exception e) {
                Questory.LOGGER.error("Failed to register consent opt-in", (Throwable)e);
            }
        });
    }

    public void disableTelemetry() {
        this.eventQueue.clear();
        this.uploadExecutor.submit(() -> {
            try {
                SupabaseClient.getInstance().registerConsent(this.hardwareId, false);
                this.cachedConsentStatus = new SupabaseClient.ConsentStatus(false, 1);
                this.lastConsentCheck = System.currentTimeMillis();
                this.forceConsentPrompt = false;
                Questory.LOGGER.info("Telemetry disabled by user");
            }
            catch (Exception e) {
                Questory.LOGGER.error("Failed to register consent opt-out", (Throwable)e);
            }
        });
    }

    public void resetConsentForTesting() {
        this.eventQueue.clear();
        this.forceConsentPrompt = true;
        this.cachedConsentStatus = new SupabaseClient.ConsentStatus(false, 0);
        this.lastConsentCheck = System.currentTimeMillis();
    }

    public void deleteAllData() {
        this.disableTelemetry();
        Questory.LOGGER.info("Local telemetry data cleared. For server-side deletion, contact mod author with HWID: {}", (Object)(this.hardwareId.substring(0, 8) + "..."));
    }

    public String exportHardwareId() {
        return this.hardwareId;
    }

    public void trackEvent(TelemetryEvent.EventType eventType, String questId, JsonObject metadata) {
        long last;
        if (!this.isEnabled()) {
            return;
        }
        long now = System.currentTimeMillis();
        if (now - (last = this.lastEventTime.get()) < 10L) {
            this.droppedEvents.incrementAndGet();
            return;
        }
        this.lastEventTime.set(now);
        try {
            TelemetryEvent event = TelemetryEvent.builder().playerUuid(UUID.nameUUIDFromBytes(this.hardwareId.getBytes())).sessionId(this.sessionId).eventType(eventType).questId(questId).modpackId(this.modpackInfo.getId()).modpackVersion(this.modpackInfo.getVersion()).timestamp(Instant.now()).metadata(metadata).build();
            this.eventQueue.add(event);
            this.totalEvents.incrementAndGet();
            if (this.eventQueue.size() >= 50) {
                this.uploadBatch();
            }
        }
        catch (Exception e) {
            Questory.LOGGER.error("Failed to track telemetry event", (Throwable)e);
        }
    }

    public void trackQuestCompleted(String questId, long completionTimeMs) {
        JsonObject metadata = new JsonObject();
        metadata.addProperty("completion_time_ms", (Number)completionTimeMs);
        this.trackEvent(TelemetryEvent.EventType.QUEST_COMPLETED, questId, metadata);
    }

    public void trackQuestViewed(String questId) {
        this.trackEvent(TelemetryEvent.EventType.QUEST_VIEWED, questId, null);
    }

    public void trackQuestOpened(String questId) {
        this.trackEvent(TelemetryEvent.EventType.QUEST_OPENED, questId, null);
    }

    private void uploadBatch() {
        if (this.eventQueue.isEmpty()) {
            return;
        }
        CopyOnWriteArrayList<TelemetryEvent> toUpload = new CopyOnWriteArrayList<TelemetryEvent>();
        int count = 0;
        while (count < 50 && !this.eventQueue.isEmpty()) {
            TelemetryEvent event = this.eventQueue.poll();
            if (event == null) continue;
            toUpload.add(event);
            ++count;
        }
        if (toUpload.isEmpty()) {
            return;
        }
        CopyOnWriteArrayList<TelemetryEvent> batch = toUpload;
        this.uploadExecutor.submit(() -> {
            try {
                SupabaseClient.getInstance().uploadEvents(batch);
                Questory.LOGGER.debug("Uploaded {} telemetry events to centralized database", (Object)batch.size());
            }
            catch (Exception e) {
                Questory.LOGGER.error("Failed to upload telemetry batch", (Throwable)e);
                this.failedUploads.incrementAndGet();
                if (this.eventQueue.size() < 100) {
                    this.eventQueue.addAll(batch);
                }
                Questory.LOGGER.warn("Event queue full, discarding {} events", (Object)batch.size());
            }
        });
    }

    private void startScheduler() {
        this.scheduler.scheduleAtFixedRate(() -> {
            try {
                if (this.isEnabled()) {
                    this.uploadBatch();
                }
            }
            catch (Exception e) {
                Questory.LOGGER.error("Scheduler error", (Throwable)e);
            }
        }, 60L, 60L, TimeUnit.SECONDS);
    }

    private void logMetrics() {
        Questory.LOGGER.info("Telemetry Stats: {} total events, {} failed uploads, {} dropped (rate limit), {} queued", (Object)this.totalEvents.get(), (Object)this.failedUploads.get(), (Object)this.droppedEvents.get(), (Object)this.eventQueue.size());
    }

    public void shutdown() {
        try {
            Questory.LOGGER.info("Shutting down telemetry manager...");
            if (this.isEnabled() && !this.eventQueue.isEmpty()) {
                Questory.LOGGER.info("Uploading {} remaining events...", (Object)this.eventQueue.size());
                this.uploadBatch();
                Thread.sleep(2000L);
            }
            this.scheduler.shutdown();
            this.uploadExecutor.shutdown();
            try {
                if (!this.scheduler.awaitTermination(3L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                }
                if (!this.uploadExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.uploadExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.scheduler.shutdownNow();
                this.uploadExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
            Questory.LOGGER.info("Telemetry manager shut down successfully");
        }
        catch (Exception e) {
            Questory.LOGGER.error("Error during telemetry shutdown", (Throwable)e);
        }
    }

    public UUID getSessionId() {
        return this.sessionId;
    }

    public long getSessionDuration() {
        return System.currentTimeMillis() - this.sessionStartTime;
    }
}

