/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.questory.client.gui.editor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

public class AutoSaveManager {
    private static AutoSaveManager instance;
    private final Map<String, SaveCallback> registeredEditors = new ConcurrentHashMap<String, SaveCallback>();
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread thread = new Thread(r, "AutoSave-Thread");
        thread.setDaemon(true);
        return thread;
    });
    private final AutoSaveConfig config = new AutoSaveConfig();
    private final List<AutoSaveEntry> history = Collections.synchronizedList(new ArrayList());
    private ScheduledFuture<?> autoSaveTask;
    private boolean paused = false;

    private AutoSaveManager() {
    }

    public static synchronized AutoSaveManager getInstance() {
        if (instance == null) {
            instance = new AutoSaveManager();
            instance.start();
        }
        return instance;
    }

    private void start() {
        if (this.autoSaveTask != null) {
            this.autoSaveTask.cancel(false);
        }
        if (this.config.enabled && this.config.intervalSeconds > 0) {
            this.autoSaveTask = this.scheduler.scheduleAtFixedRate(this::performAutoSave, this.config.intervalSeconds, this.config.intervalSeconds, TimeUnit.SECONDS);
        }
    }

    public void register(String editorId, SaveCallback callback) {
        if (editorId == null || callback == null) {
            throw new IllegalArgumentException("editorId and callback cannot be null");
        }
        this.registeredEditors.put(editorId, callback);
    }

    public void unregister(String editorId) {
        if (editorId != null) {
            this.registeredEditors.remove(editorId);
        }
    }

    public void triggerSave(String editorId) {
        if (editorId == null) {
            return;
        }
        SaveCallback callback = this.registeredEditors.get(editorId);
        if (callback != null) {
            try {
                callback.onAutoSave();
                this.recordSave(editorId, "manual", true, null);
            }
            catch (Exception e) {
                this.recordSave(editorId, "manual", false, e.getMessage());
            }
        }
    }

    private void performAutoSave() {
        if (this.paused || !this.config.enabled) {
            return;
        }
        for (Map.Entry<String, SaveCallback> entry : this.registeredEditors.entrySet()) {
            String editorId = entry.getKey();
            SaveCallback callback = entry.getValue();
            try {
                callback.onAutoSave();
                this.recordSave(editorId, "auto", true, null);
            }
            catch (Exception e) {
                this.recordSave(editorId, "auto", false, e.getMessage());
                System.err.println("[AutoSave] Failed to auto-save " + editorId + ": " + e.getMessage());
            }
        }
    }

    private void recordSave(String editorId, String entityId, boolean success, String error) {
        AutoSaveEntry entry = new AutoSaveEntry(editorId, entityId, System.currentTimeMillis(), success, error);
        this.history.add(entry);
        while (this.history.size() > 100) {
            this.history.remove(0);
        }
    }

    public void pauseAutoSave() {
        this.paused = true;
    }

    public void resumeAutoSave() {
        this.paused = false;
    }

    public boolean isPaused() {
        return this.paused;
    }

    public AutoSaveConfig getConfig() {
        return this.config;
    }

    public void updateConfig(AutoSaveConfig newConfig) {
        this.config.enabled = newConfig.enabled;
        this.config.intervalSeconds = Math.max(5, newConfig.intervalSeconds);
        this.config.saveOnFocusLoss = newConfig.saveOnFocusLoss;
        this.config.saveOnEntitySwitch = newConfig.saveOnEntitySwitch;
        this.config.maxAutoSaves = newConfig.maxAutoSaves;
        this.config.purgeAfterDays = newConfig.purgeAfterDays;
        this.start();
    }

    public List<AutoSaveEntry> getAutoSaveHistory() {
        return new ArrayList<AutoSaveEntry>(this.history);
    }

    public int getRegisteredCount() {
        return this.registeredEditors.size();
    }

    public boolean isEnabled() {
        return this.config.enabled && !this.paused;
    }

    public String getStatusString() {
        if (!this.config.enabled) {
            return "Disabled";
        }
        if (this.paused) {
            return "Paused";
        }
        long mostRecentSave = 0L;
        for (int i = this.history.size() - 1; i >= 0; --i) {
            AutoSaveEntry entry = this.history.get(i);
            if (!entry.isSuccess()) continue;
            mostRecentSave = entry.getTimestamp();
            break;
        }
        if (mostRecentSave == 0L) {
            return "Active (no saves yet)";
        }
        long ageMs = System.currentTimeMillis() - mostRecentSave;
        long seconds = ageMs / 1000L;
        long minutes = seconds / 60L;
        if (minutes > 0L) {
            return "Last saved: " + minutes + "m ago";
        }
        return "Last saved: " + seconds + "s ago";
    }

    public void shutdown() {
        if (this.autoSaveTask != null) {
            this.autoSaveTask.cancel(false);
        }
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.registeredEditors.clear();
    }

    public String getDebugInfo() {
        return String.format("AutoSaveManager[enabled=%s, paused=%s, interval=%ds, registered=%d, history=%d]", this.config.enabled, this.paused, this.config.intervalSeconds, this.registeredEditors.size(), this.history.size());
    }

    public static class AutoSaveConfig {
        public boolean enabled = true;
        public int intervalSeconds = 30;
        public boolean saveOnFocusLoss = false;
        public boolean saveOnEntitySwitch = true;
        public int maxAutoSaves = 5;
        public int purgeAfterDays = 7;

        public AutoSaveConfig copy() {
            AutoSaveConfig copy = new AutoSaveConfig();
            copy.enabled = this.enabled;
            copy.intervalSeconds = this.intervalSeconds;
            copy.saveOnFocusLoss = this.saveOnFocusLoss;
            copy.saveOnEntitySwitch = this.saveOnEntitySwitch;
            copy.maxAutoSaves = this.maxAutoSaves;
            copy.purgeAfterDays = this.purgeAfterDays;
            return copy;
        }
    }

    public static interface SaveCallback {
        public void onAutoSave();
    }

    public static class AutoSaveEntry {
        private final String editorId;
        private final String entityId;
        private final long timestamp;
        private final boolean success;
        private final String error;

        public AutoSaveEntry(String editorId, String entityId, long timestamp, boolean success, String error) {
            this.editorId = editorId;
            this.entityId = entityId;
            this.timestamp = timestamp;
            this.success = success;
            this.error = error;
        }

        public String getEditorId() {
            return this.editorId;
        }

        public String getEntityId() {
            return this.entityId;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getError() {
            return this.error;
        }

        public String getAgeString() {
            long ageMs = System.currentTimeMillis() - this.timestamp;
            long seconds = ageMs / 1000L;
            long minutes = seconds / 60L;
            long hours = minutes / 60L;
            if (hours > 0L) {
                return hours + "h ago";
            }
            if (minutes > 0L) {
                return minutes + "m ago";
            }
            return seconds + "s ago";
        }
    }
}

