/*
    Perspective
    Contributor(s): dannytaylor
    Github: https://github.com/mclegoman/perspective
    Licence: GNU LGPLv3
*/

package com.mclegoman.perspective.client.entity;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import com.mclegoman.luminance.client.shaders.Shader;
import com.mclegoman.luminance.client.shaders.Shaders;
import com.mclegoman.luminance.common.util.LogType;
import com.mclegoman.perspective.client.data.ClientData;
import com.mclegoman.perspective.client.entity.states.PerspectiveRenderState;
import com.mclegoman.perspective.client.events.PerspectiveEvents;
import com.mclegoman.perspective.client.shaders.ShaderPackEntry;
import com.mclegoman.perspective.client.shaders.ShaderPacks;
import com.mclegoman.perspective.client.shaders.TexturedEntityShader;
import com.mclegoman.perspective.client.translation.Translation;
import com.mclegoman.perspective.client.texture.TextureHelper;
import com.mclegoman.perspective.client.util.ListHelper;
import com.mclegoman.perspective.common.data.Data;
import com.mclegoman.luminance.common.util.IdentifierHelper;
import com.mclegoman.perspective.client.config.PerspectiveConfig;
import com.mclegoman.perspective.common.util.Identifiers;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import net.minecraft.class_10017;
import net.minecraft.class_10042;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7923;

public class TexturedEntity {
	private static final List<class_2960> forbiddenEntities = new ArrayList<>();
	public static List<class_2960> getForbiddenEntities() {
		return forbiddenEntities;
	}
	public static void addForbiddenEntity(class_2960... entityIds) {
		Collections.addAll(forbiddenEntities, entityIds);
	}
	public static void addForbiddenEntity(class_1299<?>... entityTypes) {
		for (class_1299<?> entityType : entityTypes) addForbiddenEntity(class_7923.field_41177.method_10221(entityType));
	}
	public static boolean isForbiddenEntity(class_2960 entityId) {
		return forbiddenEntities.contains(entityId);
	}
	public static class_2960 getEntityTypeId(class_1299<?> entityType) {
		return class_7923.field_41177.method_10221(entityType);
	}
	public static class_1299<?> getEntityType(class_2960 entityTypeId) {
		return class_7923.field_41177.method_63535(entityTypeId);
	}
	private static void addDefaultForbiddenEntities() {
		try {
			// This prevents users from trying to use textured entity features on players. Use appearance instead.
			addForbiddenEntity(class_1299.field_6097);
			// The dragon is simply just not compatible, if it ever becomes compatible, this can be removed.
			addForbiddenEntity(class_1299.field_6116);
			// Fireworks now use the itemStack itself to render.
			addForbiddenEntity(class_1299.field_6133);
			// TNT now use the blockState to render.
			addForbiddenEntity(class_1299.field_6063);
			// Boat/Raft Rendering has changed, ideally we should be able to replace the texture of boats.
			addForbiddenEntity(class_1299.field_54410, class_1299.field_54411, class_1299.field_54416, class_1299.field_54417, class_1299.field_54420, class_1299.field_54421, class_1299.field_54412, class_1299.field_54413, class_1299.field_54408, class_1299.field_54415, class_1299.field_54406, class_1299.field_54407, class_1299.field_54414, class_1299.field_54409, class_1299.field_54422, class_1299.field_54423, class_1299.field_54562, class_1299.field_54563, class_1299.field_54419, class_1299.field_54418);
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to add default forbidden textured entities: {}", error));
		}
	}
	public static void init() {
		try {
			addDefaultForbiddenEntities();
			PerspectiveEvents.ClientResourceReloaders.register(Identifiers.TEXTURED_ENTITY, new TexturedEntityDataReloader());
			PerspectiveEvents.SpectatorHandlers.register(Identifiers.TEXTURED_ENTITY, new TexturedEntityShader());
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to initialize textured entity: {}", error));
		}
	}
	private static class_2960 getOverrideTexture(String prefix, String suffix, JsonArray overrides, class_2960 fallback, class_2960 vanilla) {
		if (!overrides.isEmpty()) {
			for (JsonElement element : overrides) {
				String entityPrefix = class_3518.method_15253((JsonObject) element, "prefix", "");
				String entitySuffix = class_3518.method_15253((JsonObject) element, "suffix", "");
				String entityTexture = class_3518.method_15253((JsonObject) element, "texture", "");
				String entityTextureNamespace = entityTexture.contains(":") ? entityTexture.substring(0, entityTexture.lastIndexOf(":")) : "minecraft";
				String entityTexturePath = entityTexture.contains(":") ? entityTexture.substring(entityTexture.lastIndexOf(":") + 1) : entityTexture;
				if (prefix.equals(entityPrefix) && suffix.equals(entitySuffix)) return entityTexture.equalsIgnoreCase("") ? vanilla : class_2960.method_60655(entityTextureNamespace, entityTexturePath.endsWith(".png") ? entityTexturePath : entityTexturePath + ".png");
			}
		}
		return fallback;
	}
	private static class_2960 getOverrideTexture(JsonArray overrides, class_2960 fallback, class_2960 vanilla) {
		return getOverrideTexture("", "", overrides, fallback, vanilla);
	}
	public static class_2960 getTexture(class_10017 renderState, class_2960 fallback) {
		return getTexture(renderState, "", "", "", fallback);
	}
	public static class_2960 getTexture(class_10017 renderState, String overrideNamespace, class_2960 fallback) {
		return getTexture(renderState, overrideNamespace, "", "", fallback);
	}
	public static class_2960 getTexture(class_10017 renderState, String prefix, String suffix, class_2960 fallback) {
		return getTexture(renderState, "", prefix, suffix, fallback);
	}
	public static class_2960 getTexture(class_10017 renderState, String overrideNamespace, String prefix, String suffix, class_2960 fallback) {
		try {
			if (TexturedEntityDataReloader.isReady) {
				class_2960 entityType = getEntityTypeId(((PerspectiveRenderState)renderState).perspective$getType());
				String namespace = fallback.method_12836();
				if (!overrideNamespace.isEmpty()) namespace = overrideNamespace;
				Optional<TexturedEntityEntry> entityData = getEntity(renderState);
				if (entityData.isPresent()) {
					boolean shouldReplaceTexture = true;
						if (renderState instanceof class_10042) {
							JsonObject entitySpecific = entityData.get().getEntitySpecific();
							if (entitySpecific != null) {
								if (entitySpecific.has("ages")) {
									JsonObject ages = class_3518.method_15281(entitySpecific, "ages", new JsonObject());
									if (((class_10042) renderState).field_53457) {
										if (ages.has("baby")) {
											JsonObject typeRegistry = class_3518.method_15281(ages, "baby", new JsonObject());
											shouldReplaceTexture = class_3518.method_15258(typeRegistry, "enabled", true);
										}
									} else {
										if (ages.has("adult")) {
											JsonObject typeRegistry = class_3518.method_15281(ages, "adult", new JsonObject());
											shouldReplaceTexture = class_3518.method_15258(typeRegistry, "enabled", true);
										}
									}
								}
							}
						}
					if (shouldReplaceTexture) return TextureHelper.getTexture(getOverrideTexture(prefix, suffix, entityData.get().getOverrides(), class_2960.method_60655(namespace, "textures/textured_entity/" + entityType.method_12836() + "/" + entityType.method_12832() + "/" + (prefix + entityData.get().getName().toLowerCase() + suffix) + ".png"), fallback), fallback);
				}
			}
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to set textured entity texture: {}", error));
		}
		return fallback;
	}
	private static List<TexturedEntityEntry> getRegistry(String namespace, String entity_type) {
		List<TexturedEntityEntry> entityRegistry = new ArrayList<>();
		try {
			for (TexturedEntityEntry registry : TexturedEntityDataReloader.getRegistry()) {
				if (registry.getNamespace().equals(namespace) && registry.getType().equals(entity_type)) entityRegistry.add(registry);
			}
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to get textured entity string registry: {}", error));
		}
		return entityRegistry;
	}
	private static List<TexturedEntityEntry> getRandomRegistry(List<TexturedEntityEntry> registry) {
		List<TexturedEntityEntry> entityRegistry = new ArrayList<>();
		try {
			for (TexturedEntityEntry data : registry) if (data.getCanBeRandom() && (data.getEnabled() || data.getName().equalsIgnoreCase("default"))) entityRegistry.add(data);
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to get random textured entity string registry: {}", error));
		}
		return entityRegistry;
	}
	private static Optional<String> getEntityName(class_10017 renderState) {
		if (renderState.field_53337 != null) return Optional.of(renderState.field_53337.getString());
		else return Optional.of("default");
	}
	private static Optional<TexturedEntityEntry> getEntityData(List<TexturedEntityEntry> registry, String entityName) {
		try {
			for (TexturedEntityEntry entityData : registry) {
				if (entityName.equals(entityData.getName())) {
					if (entityData.getEnabled()) return Optional.of(entityData);
				}
			}
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to get textured entity entity data for entity '{}': {}", entityName, error));
		}
		return Optional.empty();
	}
	public static Optional<class_2960> getEntitySpecificModel(class_10017 renderState) {
		if (renderState != null) {
			Optional<TexturedEntityEntry> entityData = getEntity(renderState);
			if (entityData.isPresent()) {
				JsonObject entitySpecific = entityData.get().getEntitySpecific();
				if (entitySpecific != null) {
					if (entitySpecific.has("model")) {
						return Optional.of(class_2960.method_60654(class_3518.method_15265(entitySpecific, "model").toLowerCase()));
					}
				}
			}
		}
		return Optional.of(class_2960.method_60655(Data.getVersion().getID(), "default"));
	}
	public static Optional<TexturedEntityEntry> getEntity(class_1297 entity) {
		return getEntity(entity.method_5797() != null ? entity.method_5797().method_54160() : null, entity.method_5667(), entity.method_5864());
	}
	public static Optional<TexturedEntityEntry> getEntity(class_10017 renderState) {
		return getEntity(((PerspectiveRenderState) renderState).perspective$getStringName(), ((PerspectiveRenderState)renderState).perspective$getUUID(), ((PerspectiveRenderState)renderState).perspective$getType());
	}
	private static Optional<TexturedEntityEntry> getEntity(@Nullable String entityName, UUID uuid, class_1299<?> entityType) {
		try {
			if (entityName == null) entityName = "default";
			class_2960 entityId = getEntityTypeId(entityType);
			if (!isForbiddenEntity(entityId)) {
				List<TexturedEntityEntry> registry = getRegistry(IdentifierHelper.getStringPart(IdentifierHelper.Type.NAMESPACE, IdentifierHelper.stringFromIdentifier(entityId)), IdentifierHelper.getStringPart(IdentifierHelper.Type.KEY, IdentifierHelper.stringFromIdentifier(entityId)));
				List<TexturedEntityEntry> randomRegistry = getRandomRegistry(registry);
				if (TexturedEntityDataReloader.isReady && !registry.isEmpty()) {
					if (PerspectiveConfig.config.texturedNamedEntity.value()) {
						Optional<TexturedEntityEntry> entityData = getEntityData(registry, entityName);
						if (entityData.isPresent()) return entityData;
					}
					if (PerspectiveConfig.config.texturedRandomEntity.value()) {
						TexturedEntityEntry data = (TexturedEntityEntry) ListHelper.getRandom(uuid, randomRegistry);
						if (data.getName().equalsIgnoreCase("default")) {
							Optional<TexturedEntityEntry> entityData = getEntityData(randomRegistry, "default");
							if (entityData.isPresent()) return entityData;
						} else return Optional.of(data);
					}
					if (PerspectiveConfig.config.texturedNamedEntity.value()) {
						// If the entity texture isn't replaced by the previous checks, it doesn't have a valid textured entity,
						// so we return the default textured entity if it exists.
						if (!entityName.equalsIgnoreCase("default")) {
							Optional<TexturedEntityEntry> entityData = getEntityData(registry, "default");
							if (entityData.isPresent()) return entityData;
						}
					}
				}
			}
		} catch (Exception error) {
			Data.getVersion().sendToLog(LogType.ERROR, Translation.getString("Failed to get textured entity entity data: {}", error));
		}
		return Optional.empty();
	}
	public static boolean setTexturedEntity(boolean oldValue, boolean newValue) {
		return oldValue && newValue;
	}
	public static void applyShader(class_1297 entity) {
		setShader(getShaders(entity, getShaderPack(entity)));
	}
	public static void clearShader() {
		setShader(new ArrayList<>());
	}
	public static void setShader(List<Shader.Data> shaders) {
		PerspectiveEvents.ShaderRender.register(getTexturedEntityId(), new ArrayList<>());
		PerspectiveEvents.ShaderRender.modify(getTexturedEntityId(), shaders);
	}
	public static class_2960 getTexturedEntityId(String suffix) {
		return class_2960.method_60655(Data.getVersion().getID(), "textured_entity" + suffix);
	}
	public static class_2960 getTexturedEntityId() {
		return getTexturedEntityId("");
	}
	public static TexturedEntityEntry.SpectatorShader getShaderPack(class_1297 entity) {
		Optional<TexturedEntityEntry> texturedEntityEntry = getEntity(entity);
		return texturedEntityEntry.map(TexturedEntityEntry::getShaderPack).orElse(null);
	}
	public static List<Shader.Data> getShaders(class_1297 entity, TexturedEntityEntry.SpectatorShader spectatorShader) {
		List<Shader.Data> shaders = new ArrayList<>();
		if (entity != null) {
			if (spectatorShader != null) {
				ShaderPackEntry shaderPack = ShaderPacks.getShaderPack(spectatorShader.registry(), spectatorShader.shaderPack());
				if (shaderPack == null && spectatorShader.shaderPack().equals(Identifiers.RANDOM)) {
					// Check if shader isn't valid AND id is perspective:random, then set shaderPack to random based on uuid.
					shaderPack = (ShaderPackEntry) ListHelper.getRandom(entity.method_5667(), ShaderPacks.getRegistry(spectatorShader.registry()).values().stream().toList());
				}
				if (shaderPack != null) {
					int i = 0;
					for (ShaderPackEntry.Shader shader : shaderPack.shaders()) {
						shaders.add(new Shader.Data(getTexturedEntityId(String.valueOf(i++)), new Shader(Shaders.get(shader.registry(), shader.luminance()), () -> Shader.RenderType.WORLD, () -> ClientData.minecraft.field_1719 != null && (ClientData.minecraft.field_1719 == entity))));
					}
				} else Data.getVersion().sendToLog(LogType.WARN, "Could not locate the current shader pack!: " + spectatorShader.registry() + ":" + spectatorShader.shaderPack());
			}
		}
		return shaders;
	}
}