/**
 * Retrieves all the upcoming departures for the specified station.
 *
 * @param {mtr.client.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing schedule information.
 * @param {mtr.data.Station} Station - The station instance from the Minecraft Transit Railway (MTR) mod. (If you get the route through getStation() then the instance is in '_raw' value)
 * @param {Array} filter - An array of platform identifiers used to filter upcoming departures, returning only the departures from the platforms included in the array.
 * If this parameter is an empty array, null, undefined, or not an array, all departures from all platforms will be retrieved.
 * @returns {Object[]|null} An array of schedule entries (JS-friendly objects) representing the next departures ordered from soonest to latest.
 * Returns an empty array if no departures are found, or null if `ClientData` or `Station` are not provided.
 */
function getAllDepartures(ClientData, Station, filter) {
  let filterPlatforms = (Array.isArray(filter) && filter.length > 0) ? filter.map(String) : null;
  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getAllDepartures(): You must add the MTR Client Data");
    return null;
  }

  if (!Station) {
    print("JOANNET NEUE ERROR>> getAllDepartures(): You must add an Station instance");
    return null;
  }

  if (ClientData.ROUTES.size() === 0 || ClientData.PLATFORMS.size() === 0 || ClientData.SCHEDULES_FOR_PLATFORM.size() === 0) {
    print("JOANNET NEUE WARNING>> getAllDepartures(): This world doesn't have any platforms nor routes, ergo there's not schedules. Function will always return an empty array.");
  }

  const departures = [];
  const platformIDsInStation = [];
  for (let platform of ClientData.PLATFORMS) {
    let platformCoords = new Vector3f(platform.getMidPos());
    if (Station.inArea(platformCoords.x(), platformCoords.z())) {
      if (!filterPlatforms || filterPlatforms.includes(String(platform.name))) {
        platformIDsInStation.push(platform.id);
      }
    }
  }

  //print(`JOANNET NEUE DEBUG>> getAllDepartures(): Platforms in Station ${platformIDsInStation} (${platformIDsInStation.length} element/s)`);
  for (let i = 0; i < platformIDsInStation.length; i++) {
    let platformLong = new Packages.java.lang.Long(platformIDsInStation[i]);
    if (ClientData.SCHEDULES_FOR_PLATFORM.containsKey(platformLong)) {
      let departuresSet = ClientData.SCHEDULES_FOR_PLATFORM.get(platformLong);
      for (let departure of departuresSet) {
        departures.push({
          arrivalMillis: Number(departure.arrivalMillis),
          trainCars: Number(departure.trainCars),
          platformId: Number(platformLong),
          routeId: Number(departure.routeId),
          currentStationIndex: Number(departure.currentStationIndex)
        });
      }
    }
  }

  //print(`JOANNET NEUE DEBUG>> getAllDepartures(): Departures ${departures} (${departures.length} element/s)`);

  if (platformIDsInStation.length === 0) {
    print("JOANNET NEUE INFO>> getAllDepartures(): Station \"" + Station.id + "\" has no platforms. No departures to search, returning empty array...");
  }

  return departures.sort((a, b) => a.arrivalMillis - b.arrivalMillis);
}