/**
 * Finds the closest platform to the given model coordinates and the squared distance to it.
 *
 * @param {mtr.data.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing platform information.
 * @param {cn.zbx1425.sowcer.math.Vector3f} modelVector3f - The 3D position vector from the MTR-NTE mod representing the model's coordinates.
 * @returns {{closestPlatform: number|null, distanceRaw: number|null}} An object containing:
 *   - `closestPlatform`: The ID of the closest platform, or `null` if no platforms exist or if `ClientData` or `modelVector3f` are not provided.
 *   - `distanceRaw`: The squared distance from the model to the closest platform, or `null` if unavailable.
 */
function getClosestPlatformAndDistance(ClientData, modelVector3f) {
  const res = {
    closestPlatform: null,
    distanceRaw: null
  };

  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getClosestPlatformAndDistance(): You must add the MTR Client Data");
    return res;
  }

  if (!modelVector3f) {
    print("JOANNET NEUE ERROR>> getClosestPlatformAndDistance(): You must give the model's Vector3f");
    return res;
  }

  if (ClientData.PLATFORMS.size() === 0) {
    print("JOANNET NEUE WARNING>> getClosestPlatformAndDistance(): This world doesn't have any platforms. Function will always return object with null values");
  }

  let minDistSq = Number.MAX_VALUE;
  for (let platform of ClientData.PLATFORMS) {
    let chosenPlatform = new Vector3f(platform.getMidPos());
    let distSq = modelVector3f.distanceSq(chosenPlatform);

    if (distSq < minDistSq) {
      minDistSq = distSq;
      res.closestPlatform = platform.id;
      res.distanceRaw = distSq;
    }
  }

  return res;
}