/**
 * Finds the ID of the closest station to the given model coordinates, optionally filtering by station name.
 *
 * @param {mtr.client.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing station information.
 * @param {cn.zbx1425.sowcer.math.Vector3f} modelVector3f - The 3D position vector from the MTR-NTE mod representing the model's coordinates.
 * @param {string} [includes] - Optional substring to filter station names (case-insensitive). Only stations whose names include this string will be considered.
 * @returns {number|null} The ID of the closest station. Returns `null` if no stations exist, if none match the filter, or if `ClientData` or `modelVector3f` are null.
 */
function getClosestStation(ClientData, modelVector3f, includes) {
  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getClosestStation(): You must add the MTR Client Data");
    return null
  }

  if (!modelVector3f) {
    print("JOANNET NEUE ERROR>> getClosestStation(): You must give the model's Vector3f");
    return null
  }

  if (ClientData.STATIONS.size() === 0) {
    print("JOANNET NEUE WARNING>> getClosestStation(): This world doesn't have any stations. Function will always return null");
  }

  let closestStation = null
  let minDistSq = Number.MAX_VALUE;
  for (let station of ClientData.STATIONS) {
    if (includes && !station.name.toLowerCase().includes(includes.toLowerCase())) { continue; }
    if (station.getCenter()) {
      let chosenStation = new Vector3f(station.getCenter());
      let distSq = modelVector3f.distanceSq(chosenStation);
      if (distSq < minDistSq) {
        minDistSq = distSq;
        closestStation = station.id !== null ? Number(station.id) : null;
      }
    }
  }

  return closestStation;
}