/**
 * Finds the closest station to the given model coordinates and returns its ID along with the squared distance to its center point.
 * Optionally filters stations by name.
 *
 * @param {mtr.client.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing station information.
 * @param {cn.zbx1425.sowcer.math.Vector3f} modelVector3f - The 3D position vector from the MTR-NTE mod representing the model's coordinates.
 * @param {string} [includes] - Optional substring to filter station names (case-insensitive). Only stations whose names include this string will be considered.
 * @returns {{closestStation: number|null, distanceRaw: number|null}} An object containing:
 *   - `closestStation`: The ID of the closest matching station, or `null` if none found.
 *   - `distanceRaw`: The squared distance from the model to the station’s center point, or `null` if unavailable.
 */
function getClosestStationAndDistance(ClientData, modelVector3f, includes) {
  const res = {
    closestStation: null,
    distanceRaw: null
  };

  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getClosestStationAndDistance(): You must add the MTR Client Data");
    return res;
  }

  if (!modelVector3f) {
    print("JOANNET NEUE ERROR>> getClosestStationAndDistance(): You must give the model's Vector3f");
    return res;
  }

  if (ClientData.STATIONS.size() === 0) {
    print("JOANNET NEUE WARNING>> getClosestStationAndDistance(): This world doesn't have any stations. Function will always return object with null values");
  }
  
  let minDistSq = Number.MAX_VALUE;
  for (let station of ClientData.STATIONS) {
    if (includes && !station.name.toLowerCase().includes(includes.toLowerCase())) { continue; }
    if (station.getCenter()) {
      let chosenStation = new Vector3f(station.getCenter());
      let distSq = modelVector3f.distanceSq(chosenStation);
      if (distSq < minDistSq) {
        minDistSq = distSq;
        res.closestStation = station.id !== null ? Number(station.id) : null;
        res.distanceRaw = distSq;
      }
    }
  }

  return res;
}