/**
 * Obtains the route delays by querying the MTR API.
 *
 * This data is not accessible via the MTRClientData, so this function directly
 * searches through the MTR API to retrieve delay information for a specific route.
 *
 * @param {string} routeName - The name of the route to query (CASE SENSITIVE).
 * @param {number} [level=0] - The world number where to locate the delays.
 *                             Default is 0 (overworld).
 * @param {string} [url="localhost"] - The URL of the MTR API. Defaults to "localhost".
 * @param {number} [port=8888] - The port of the MTR API. Defaults to 8888.
 * @param {boolean} [secure=false] - Whether to use HTTPS. Defaults to false.
 * @returns {Array<Object>} An array of delay objects for the given route.
 * Each object includes:
 *   - `destination` {string} - The destination of the route.
 *   - `delay` {number} - Delay in ticks.
 *   - `delayMs` {number} - Delay in milliseconds.
 *   - `delaySec` {number} - Delay in seconds (rounded).
 *   - `delayMin` {number} - Delay in minutes (rounded).
 * @returns {Array<>} Empty array is returned in case of function error or no matches has been found.
 */
function getDelaysFromRoute(routeName, level, url, port, secure) {
  if (!routeName) {
    print("JOANNET NEUE ERROR>> getDelayFromRoute(): routeName must have a valid value.")
    return null;
  }

  let finalLevel;
  if (!isNaN(Number(level))) {
    finalLevel = Number(level);
  } else {
    finalLevel = 0;
  }

  let finalURL;
  if (typeof url === 'string' && url.trim() !== '') {
    finalURL = url;
  } else {
    finalURL = 'localhost';
  }

  let finalPort;
  if (!isNaN(Number(port)) && !(Number(port) >= 0 && Number(port) <= 1023)) {
    finalPort = port;
  } else {
    finalPort = 8888;
  }

  let finalSecure = "http";
  if (typeof secure === 'boolean' && secure === true) {
    finalSecure = "https";
  }

  try {
    // Fetching data...
    const connection = new URL(`${finalSecure}://${finalURL}:${finalPort}/delays`).openConnection();
    connection.setRequestMethod("GET");
    connection.setRequestProperty("Accept", "application/json");
    if (connection.getResponseCode() !== 200) throw new Error("JOANNET NEUE ERROR>> getDelayFromRoute(): HTTP request failed with status code: " + connection.getResponseCode());
    const reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));
    let response = ""; let line;
    while ((line = reader.readLine()) != null) { response += line; }
    reader.close(); const res = JSON.parse(response)[finalLevel];

    //Data's here! Let's interact with it.
    const reportedDelays = [];
    const filtered = res.filter(it => it.name.toString() === String(routeName))
    for (let i = 0; i < filtered.length; i++) {
      reportedDelays.push(
        {
          destination: filtered[i].destination,
          delay: filtered[i].delay, // Ticks
          delayMs: filtered[i].delay * 50, // Milliseconds
          delaySec: Math.round(filtered[i].delay * 0.05), // Seconds
          delayMin: Math.round((filtered[i].delay * 0.05) / 60) // Minutes
        }
      )
    }

    return reportedDelays;
  } catch (error) {
    print("JOANNET NEUE ERROR>> getDelayFromRoute(): Error getting delays: " + error);
    if (error.message.toLowerCase().includes("connection refused: connect")) print("JOANNET NEUE WARN>> getDelayFromRoute(): Ignore previous error if you're not connected in a world.")
    return [];
  }
}