/**
 * Retrieves the upcoming departures for a given platform by its ID.
 *
 * @param {mtr.client.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing schedule information.
 * @param {number} platformId - The unique ID of the platform to query.
 * @returns {Object[]|null} An array of schedule entries (JS-friendly objects) representing the next departures.
 * Returns an empty array if no departures are found, or null if `ClientData` or `platformId` are not provided.
 */
function getDepartures(ClientData, platformId) {
  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getDepartures(): You must add the MTR Client Data");
    return null;
  }

  if (!platformId) {
    print("JOANNET NEUE ERROR>> getDepartures(): You must set a valid platformId to search");
    return null;
  }

  if (ClientData.ROUTES.size() === 0 || ClientData.PLATFORMS.size() === 0 || ClientData.SCHEDULES_FOR_PLATFORM.size() === 0) {
    print("JOANNET NEUE WARNING>> getDepartures(): This world doesn't have any platforms nor routes, ergo there's not schedules. Function will always return an empty array.");
  }

  const departures = [];
  let platformLong = new Packages.java.lang.Long(platformId);
  if (ClientData.SCHEDULES_FOR_PLATFORM.containsKey(platformLong)) {
    let departuresSet = ClientData.SCHEDULES_FOR_PLATFORM.get(platformLong);
    for (let departure of departuresSet) {
      departures.push({
        arrivalMillis: Number(departure.arrivalMillis),
        trainCars: Number(departure.trainCars),
        routeId: Number(departure.routeId),
        currentStationIndex: Number(departure.currentStationIndex)
      });
    }
  }

  return departures.sort((a, b) => a.arrivalMillis - b.arrivalMillis);
}