/**
 * Retrieves all station IDs from a route.
 *
 * @param {mtr.data.ClientData} ClientData - The client data object from the Minecraft Transit Railway (MTR) mod containing station and platform information.
 * @param {mtr.data.Route} Route - The route instance from the Minecraft Transit Railway (MTR) mod. (If you get the route through getRoute() then the instance is in '_raw' value)
 * @param {number} originPlatformID - The platform ID from which to start the list.
 * If null, undefined, invalid (not a integer number), or not found in the list, the function will proceed using all available platforms.
 *
 * @returns {number[]|null} An array of station IDs where the route stps. Returns an empty array if none are found, or null if `ClientData` or `Route` is not provided.
 */
function getStationsFromRoute(ClientData, Route, originPlatformID) {
  let firstPlatformID = Number.isInteger(Number(originPlatformID)) ? Number(originPlatformID) : null;

  if (!ClientData) {
    print("JOANNET NEUE ERROR>> getStationsFromRoute(): You must add the MTR Client Data");
    return null;
  }

  if (!Route) {
    print("JOANNET NEUE ERROR>> getStationsFromRoute(): You must add a route instance");
    return null;
  }

  if (ClientData.STATIONS.size() === 0) {
    print("JOANNET NEUE WARNING>> getStationsFromRoute(): This world doesn't have any stations. Function will always return an empty array.");
  }

  //Get the platforms IDs from the route
  const routePlatformIds = [];
  for (let pID of Route.platformIds) {
    routePlatformIds.push(Number(pID.platformId));
  }

  //Check if we want to start the list from the user's platform ID, if not then let's continue with all
  const finalRoutePlatformIds = firstPlatformID ? routePlatformIds.filter(id => id >= firstPlatformID) : routePlatformIds;

  //Get the platform locations from the platform IDs
  const routePlatformVectors = [];
  for (let platformId of finalRoutePlatformIds) {
    for (let platform of ClientData.PLATFORMS) {
      if (Number(platform.id) === Number(platformId)) {
        routePlatformVectors.push(new Vector3f(platform.getMidPos()));
        break;
      }
    }
  }

  //And then, get the station ID from the platform vectors
  const routeStationIds = [];
  for (let platformVector3f of routePlatformVectors) {
    for (let station of ClientData.STATIONS) {
      if (station.inArea(platformVector3f.x(), platformVector3f.z())) {
        routeStationIds.push(Number(station.id));
        break;
      }
    }
  }

  if (routeStationIds.length === 0) {
    print("JOANNET NEUE INFO>> getStationsFromRoute(): Route \"" + Route.id + "\" has no stations. Returning empty array...");
  }

  return routeStationIds;
}