/**
 * Draws text centered both horizontally and vertically within a specified rectangle.
 * Supports multi-line splitting up to a maximum number of lines and optional automatic font-size adjustment
 * to ensure the text block fits within the rectangle.
 *
 * @param {Graphics2D} ctx - The Graphics2D context to draw on.
 * @param {String} text - The text string to render.
 * @param {int} x - The x-coordinate of the top-left corner of the rectangle.
 * @param {int} y - The y-coordinate of the top-left corner of the rectangle.
 * @param {int} width - The width of the rectangle in which to center the text.
 * @param {int} height - The height of the rectangle in which to center the text.
 * @param {Font} font - The base Font to use for rendering.
 * @param {int} maxLines - The maximum number of lines to split the text into (default 1 if falsy).
 * @param {boolean} autoAdjust - Whether to grow/shrink font to best fit the rectangle.
 * @param {int} padding - Padding (in pixels) around the text block.
 */
function drawCenteredText(ctx, text, x, y, width, height, font, maxLines, autoAdjust, padding) {
  // default parameters
  var linesMax = maxLines || 1;
  var pad = padding || 0;

  // calculate padded drawing area
  var availX = x + pad;
  var availY = y + pad;
  var availW = width - 2 * pad;
  var availH = height - 2 * pad;

  // Private helper: wraps a string into lines that fit within a given width.
  function wrap(text, fm) {
    var words = text.split(' ');
    var lines = [];
    if (fm.stringWidth(text) <= availW) {
      lines.push(text);
    } else {
      var line = words[0];
      var count = 1;
      for (var i = 1; i < words.length; i++) {
        var candidate = line + ' ' + words[i];
        if (fm.stringWidth(candidate) <= availW) {
          line = candidate;
        } else {
          lines.push(line);
          count++;
          if (count < linesMax) {
            line = words[i];
          } else {
            line = words.slice(i).join(' ');
            break;
          }
        }
      }
      lines.push(line);
    }
    return lines;
  }

  // determine best font size
  var baseSize = font.getSize2D();
  var bestSize = baseSize;

  if (autoAdjust) {
    var lastFit = baseSize;
    var maxSize = baseSize * 2;
    for (var size = 4; size <= maxSize; size++) {
      var testFont = font.deriveFont(size);
      ctx.setFont(testFont);
      var fm = ctx.getFontMetrics();
      var lines = wrap(text, fm);
      var totalH = lines.length * fm.getHeight();
      if (totalH <= availH) {
        lastFit = size;
      } else {
        break;
      }
    }
    bestSize = lastFit;
  }

  // apply final font
  var finalFont = font.deriveFont(bestSize);
  ctx.setFont(finalFont);
  var fm = ctx.getFontMetrics();

  // split text into lines
  var lines = wrap(text, fm);

  // compute vertical centering
  var lineHeight = fm.getHeight();
  var totalHeight = lines.length * lineHeight;
  var startY = availY + (availH - totalHeight) / 2 + fm.getAscent();

  // draw lines
  for (var i = 0; i < lines.length; i++) {
    var line = lines[i];
    var lineWidth = fm.stringWidth(line);
    var drawX = availX + (availW - lineWidth) / 2;
    var drawY = startY + i * lineHeight;
    ctx.drawString(line, drawX, drawY);
  }
}
