/**
 * Get the closest station from the model coords, optionally filtering by station name.
 * @param {mtr.client.ClientData} ClientData - ClientData from the Minecraft Transit Railway.
 * @param {cn.zbx1425.sowcer.math.Vector3f} modelVector3f - Vector3f from the MTR-NTE mod.
 * @param {string} [filterWord] - Optional word to filter station names (case-insensitive).
 * @throws {Error} If ClientData and/or modelVector3f are null.
 * @return {{closestStation: mtr.data.Station|null, distanceRaw: number|null}|null} 
 *         The closest station matching the filter (if provided) and its squared distance, or null if none found.
 */
function getClosestStation(ClientData, modelVector3f, filterWord) {
  if (!ClientData) {
    throw new Error("You must add the MTR Client Data");
  }

  if (!modelVector3f) {
    throw new Error("You must give the model's Vector3f");
  }

  if (ClientData.STATIONS.size() === 0) {
    return null
  }

  const res = {
    closestStation: null,
    distanceRaw: null
  };
  
  let minDistSq = Number.MAX_VALUE;
  for (let station of ClientData.STATIONS) {
    if (filterWord && !station.name.toLowerCase().includes(filterWord.toLowerCase())) { continue; }
    if (station.getCenter()) {
      let chosenStation = new Vector3f(station.getCenter());
      let distSq = modelVector3f.distanceSq(chosenStation);
      if (distSq < minDistSq) {
        minDistSq = distSq;
        res.closestStation = station;
        res.distanceRaw = distSq;
      }
    }
  }

  return res;
}