package io.github.ghasto69.happy_airships

import net.minecraft.core.NonNullList
import net.minecraft.resources.ResourceKey
import net.minecraft.server.level.ServerLevel
import net.minecraft.world.Containers
import net.minecraft.world.InteractionHand
import net.minecraft.world.InteractionResult
import net.minecraft.world.damagesource.DamageSource
import net.minecraft.world.entity.EntityType
import net.minecraft.world.entity.HasCustomInventoryScreen
import net.minecraft.world.entity.monster.piglin.PiglinAi
import net.minecraft.world.entity.player.Inventory
import net.minecraft.world.entity.player.Player
import net.minecraft.world.entity.vehicle.AbstractBoat
import net.minecraft.world.entity.vehicle.ContainerEntity
import net.minecraft.world.inventory.AbstractContainerMenu
import net.minecraft.world.inventory.DispenserMenu
import net.minecraft.world.item.Item
import net.minecraft.world.item.ItemStack
import net.minecraft.world.level.Level
import net.minecraft.world.level.gameevent.GameEvent
import net.minecraft.world.level.storage.ValueInput
import net.minecraft.world.level.storage.ValueOutput
import net.minecraft.world.level.storage.loot.LootTable
import java.util.function.Supplier

abstract class AbstractDispenserBoat(
    entityType: EntityType<out AbstractBoat>,
    level: Level,
    supplier: Supplier<Item>,
) : AbstractBoat(entityType, level, supplier), HasCustomInventoryScreen, ContainerEntity {
    companion object {
        private const val CONTAINER_SIZE = 9
    }
    private var itemStacks: NonNullList<ItemStack> = NonNullList.withSize(CONTAINER_SIZE, ItemStack.EMPTY)
    private var lootTable: ResourceKey<LootTable>? = null
    private var lootTableSeed: Long = 0

    override fun getSinglePassengerXOffset() = 0.15F
    override fun getMaxPassengers() = 1

    override fun readAdditionalSaveData(valueInput: ValueInput) {
        super.readAdditionalSaveData(valueInput)
        readChestVehicleSaveData(valueInput)
    }

    override fun addAdditionalSaveData(valueOutput: ValueOutput) {
        super.addAdditionalSaveData(valueOutput)
        addChestVehicleSaveData(valueOutput)
    }

    public override fun destroy(serverLevel: ServerLevel, damageSource: DamageSource) {
        this.destroy(serverLevel, this.dropItem)
        this.chestVehicleDestroyed(damageSource, serverLevel, this)
    }

    override fun remove(removalReason: RemovalReason) {
        if (!this.level().isClientSide && removalReason.shouldDestroy()) {
            Containers.dropContents(this.level(), this, this)
        }

        super.remove(removalReason)
    }

    override fun interact(player: Player, interactionHand: InteractionHand): InteractionResult {
        val interactionResult = super.interact(player, interactionHand)
        if (interactionResult !== InteractionResult.PASS) {
            return interactionResult
        } else if (this.canAddPassenger(player) && !player.isSecondaryUseActive) {
            return InteractionResult.PASS
        } else {
            val interactionResult2 = this.interactWithContainerVehicle(player)
            val level = player.level()
            if (interactionResult2.consumesAction() && level is ServerLevel) {
                this.gameEvent(GameEvent.CONTAINER_OPEN, player)
            }

            return interactionResult2
        }
    }

    override fun openCustomInventoryScreen(player: Player) {
        player.openMenu(this)
        val level = player.level()
        if (level is ServerLevel) {
            this.gameEvent(GameEvent.CONTAINER_OPEN, player)
        }
    }

    override fun getContainerLootTable() = lootTable

    override fun setContainerLootTable(resourceKey: ResourceKey<LootTable>?) {
        lootTable = resourceKey
    }

    override fun getContainerLootTableSeed() = lootTableSeed

    override fun setContainerLootTableSeed(l: Long) {
        lootTableSeed = l
    }

    override fun getItemStacks() = itemStacks

    override fun clearItemStacks() {
        itemStacks = NonNullList.withSize(CONTAINER_SIZE, ItemStack.EMPTY)
    }

    override fun getContainerSize() = CONTAINER_SIZE

    override fun getItem(i: Int): ItemStack = getChestVehicleItem(i)

    override fun removeItem(i: Int, j: Int): ItemStack = removeChestVehicleItem(i, j)

    override fun removeItemNoUpdate(i: Int): ItemStack = removeChestVehicleItemNoUpdate(i)

    override fun setItem(i: Int, itemStack: ItemStack) {
        setChestVehicleItem(i, itemStack)
    }

    override fun setChanged() {}

    override fun stillValid(player: Player) = isChestVehicleStillValid(player)

    override fun clearContent() {
        clearChestVehicleContent()
    }

    override fun createMenu(
        i: Int,
        inventory: Inventory,
        player: Player
    ): AbstractContainerMenu? {
        if (this.lootTable != null && player.isSpectator) {
            return null
        } else {
            unpackChestVehicleLootTable(inventory.player)
            return DispenserMenu(i, inventory, this)
        }
    }
}