package io.github.ghasto69.happy_airships

import com.mojang.blaze3d.vertex.PoseStack
import com.mojang.blaze3d.vertex.VertexMultiConsumer
import com.mojang.math.Axis
import io.github.ghasto69.happy_airships.datagen.ModItemTags
import net.minecraft.client.Minecraft
import net.minecraft.client.model.HappyGhastModel
import net.minecraft.client.renderer.MultiBufferSource
import net.minecraft.client.renderer.RenderType
import net.minecraft.client.renderer.entity.RenderLayerParent
import net.minecraft.client.renderer.entity.layers.RenderLayer
import net.minecraft.client.renderer.entity.state.HappyGhastRenderState
import net.minecraft.client.renderer.texture.OverlayTexture
import net.minecraft.resources.ResourceLocation

class PropellerLayer<M : HappyGhastModel>(renderLayerParent: RenderLayerParent<HappyGhastRenderState, M>) :
    RenderLayer<HappyGhastRenderState, M>(renderLayerParent) {
    companion object {
        val RENDER_TYPE_ID: ResourceLocation = HappyAirships.identifier("textures/entity/happy_ghast/propeller.png")
    }

    override fun render(
        poseStack: PoseStack,
        multiBufferSource: MultiBufferSource,
        light: Int,
        renderState: HappyGhastRenderState,
        f: Float,
        g: Float
    ) {
        if (!renderState.bodyItem.`is`(ModItemTags.HARNESSES_WITH_PROPELLERS)) return

        data class Vertex(val x: Float, val y: Float, val z: Float, val u: Float, val v: Float)

        poseStack.pushPose()

        val glint = renderState.bodyItem.hasFoil()
        val glintVertexConsumer = multiBufferSource.getBuffer(RenderType.entityGlint())

        var vertexConsumer = multiBufferSource.getBuffer(RenderType.entityCutoutNoCull(RENDER_TYPE_ID))
        if (glint) vertexConsumer = VertexMultiConsumer.create(glintVertexConsumer, vertexConsumer)

        poseStack.translate(-2f, -2.5f, 2f + (1f / 8f))

        // --- Centered Rotation ---
        poseStack.translate(2f, 2f, 0f) // Move to center of quad

        poseStack.mulPose(
            Axis.ZP.rotationDegrees(
                (renderState.ageInTicks + Minecraft.getInstance().deltaTracker.gameTimeDeltaTicks) * (if (renderState.isRidden) 1f else 0.1f) * 20 % 360f
            )
        ) // Apply rotation

        poseStack.translate(-2f, -2f, 0f) // Move back to original position

        val vertices = listOf(
            Vertex(0f, 0f, 0f, 0f, 1f),  // Bottom-Right
            Vertex(4f, 0f, 0f, 1f, 1f),  // Top-Right
            Vertex(4f, 4f, 0f, 1f, 0f),  // Top-Left
            Vertex(0f, 4f, 0f, 0f, 0f)   // Bottom-Left
        )

        for (vertex in vertices) {
            vertexConsumer.addVertex(poseStack.last(), vertex.x, vertex.y, vertex.z).setColor(255, 255, 255, 255)
                .setUv(vertex.u, vertex.v).setOverlay(OverlayTexture.NO_OVERLAY).setNormal(0.0f, 0.0f, 1.0f)
                .setLight(light)
        }

        poseStack.popPose()

        var planksConsumer = multiBufferSource.getBuffer(RenderType.entityCutoutNoCull(
                ResourceLocation.withDefaultNamespace("textures/block/dark_oak_planks.png")
        ))
        if(glint) planksConsumer = VertexMultiConsumer.create(glintVertexConsumer, planksConsumer)


        val woodenVertices = listOf(
            Vertex(0f, 0f, 0f, 0f, 4f / 16f),                 // Bottom-Right
            Vertex(1f / 8f, 0f, 0f, 1f / 16f, 4f / 16f),       // Top-Right
            Vertex(1f / 8f, 4f / 8f, 0f, 1f / 16f, 0f),        // Top-Left
            Vertex(0f, 4f / 8f, 0f, 0f, 0f)                    // Bottom-Left
        )

        poseStack.pushPose()
        poseStack.translate(2f / 8f, -0.5f, 2f + (1f / 8f))
        poseStack.mulPose(Axis.ZP.rotationDegrees(90f))
        poseStack.mulPose(Axis.YP.rotationDegrees(90f))

        for (vertex in woodenVertices) {
            planksConsumer.addVertex(poseStack.last(), vertex.x, vertex.y, vertex.z).setColor(255, 255, 255, 255)
                .setUv(vertex.u, vertex.v).setOverlay(OverlayTexture.NO_OVERLAY).setNormal(0.0f, 1.0f, 0.0f)
                .setLight(light)
        }

        poseStack.popPose()

        poseStack.pushPose()
        poseStack.translate(0f, -0.5f - 2f / 8f, 2f + (1f / 8f))
        poseStack.mulPose(Axis.YP.rotationDegrees(90f))

        for (vertex in woodenVertices) {
            planksConsumer.addVertex(poseStack.last(), vertex.x, vertex.y, vertex.z).setColor(255, 255, 255, 255)
                .setUv(vertex.u, vertex.v).setOverlay(OverlayTexture.NO_OVERLAY).setNormal(1.0f, 0.0f, 0.0f)
                .setLight(light)
        }

        poseStack.popPose()
    }
}