/*
SimpleJSON.java
Written in 2024 by cheny0y0
To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
*/

/*
 * REGE modified.
 */

package rege.rege.misc.customsavedirs.cc0fork;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Simple JSON implementation(process, parse, dump) in Java.
 * Works on Java 5 +.
 * Use {@link #SimpleJSON(long)}, {@link #SimpleJSON(double)},
 * {@link #SimpleJSON(java.math.BigDecimal)},
 * {@link #SimpleJSON(java.lang.String)}, {@link #SimpleJSON(boolean)},
 * {@link #SimpleJSON(SimpleJSON[])}, {@link #SimpleJSON(java.lang.Iterable)},
 * {@link #SimpleJSON(java.util.Map.Entry[])},
 * {@link #SimpleJSON(java.util.Map)} to new JSONs.
 * Use {@link #getProperty(Object)}. {@link #setProperty(Object, SimpleJSON)},
 * {@link #deleteProperty(Object)} to modify JSONs.
 * Use {@link #toString()} to dump JSONs.
 * Use {@link #parseJSON(String)} to parse JSON strings and return JSONs.
 * @author cheny0y0
 */
public class SimpleJSON {
    /**
     * The character around the JSON strings.
     */
    public static final char STRING_WRAP = '"';
    /**
     * The character which separates JSON arrays' and objects' items.
     */
    public static final char ITEM_SEP = ',';
    /**
     * The character which separates JSON objects' key-value pairs.
     */
    public static final char KEY_VALUE = ':';
    /**
     * The character which starts JSON arrays.
     */
    public static final char ARRAY_START = '[';
    /**
     * The character which ends JSON arrays.
     */
    public static final char ARRAY_END = ']';
    /**
     * The character which starts JSON objects.
     */
    public static final char OBJECT_START = '{';
    /**
     * The character which ends JSON objects.
     */
    public static final char OBJECT_END = '}';
    /**
     * The "false" singleton's identifier.
     */
    public static final String FALSE_IDENTIFIER = "false";
    /**
     * The "true" singleton's identifier.
     */
    public static final String TRUE_IDENTIFIER = "true";
    /**
     * The "null" singleton's identifier.
     */
    public static final String NULL_IDENTIFIER = "null";
    /**
     * The whitespaces.
     */
    public static final String WHITESPACES = "\t\n\r ";
    /**
     * The whitespaces that are allowed inside JSON strings.
     */
    public static final String STRING_ALLOWED_WHITESPACES = " ";
    /**
     * The minus sign used in number prefixes and after exponent symbols.
     */
    public static final char MINUS_SIGN = '-';
    /**
     * The plus sign used after exponent symbols.
     */
    public static final char PLUS_SIGN = '+';
    /**
     * The figure zero.
     */
    public static final char ZERO = '0';
    /**
     * The decimal point.
     */
    public static final char DECIMAL_POINT = '.';
    /**
     * The exponent symbols.
     */
    public static final String EXPONENT = "Ee";
    /**
     * The digit figures.
     */
    public static final String DIGITS = "0123456789";
    /**
     * The start character of escape in JSON strings.
     */
    public static final char ESCAPE_START = '\\';
    private static final HashMap<@NotNull Character, @NotNull String>
    ESCAPE_RESULTS;
    private static final HashMap<@NotNull Character, @NotNull Byte>
    UNICODE_ESCAPE_VALUES;

    private static boolean containsChar(@NotNull String str, char ch) {
        for (char i : str.toCharArray()) {
            if (i == ch) {
                return true;
            }
        }
        return false;
    }

    /**
     * Parse the string and return a {@link SimpleJSON} if succeeded.
     * @param str The string to be parsed.
     * @return A {@link SimpleJSON} if succeeded.
     * @throws IllegalArgumentException if failed.
     */
    @NotNull
    public static SimpleJSON parseJSON(@NotNull String str)
    throws IllegalArgumentException {
        final KVPair PAIR = parse(str, 0, false);
        final int LEN = str.length();
        int ptr = Integer.parseInt(PAIR.getKey());
        while (ptr < LEN && containsChar(WHITESPACES, str.charAt(ptr))) {
            ptr++;
        }
        if (ptr < LEN) {
            throw new IllegalArgumentException("extra content beyond char " +
                                               ptr);
        }
        return PAIR.getValue();
    }

    @NotNull
    private static KVPair
    parse(@NotNull String str, int pointer, boolean stringOnly)
    throws IllegalArgumentException {
        final int LEN = str.length();
        try {
            while (containsChar(WHITESPACES, str.charAt(pointer))) {
                pointer++;
            }
        } catch (StringIndexOutOfBoundsException e) {
            throw new IllegalArgumentException("no data found");
        }
        char achar = str.charAt(pointer);
        if (achar == OBJECT_START && !stringOnly) {
            SimpleJSON json = new SimpleJSON(new HashMap<String, SimpleJSON>());
            pointer++;
            while (pointer < LEN && containsChar(
                WHITESPACES, str.charAt(pointer)
            )) {
                pointer++;
            }
            try {
                if (str.charAt(pointer) == OBJECT_END) {
                    return new KVPair(Integer.toString(pointer + 1), json);
                }
            } catch (StringIndexOutOfBoundsException e) {
                throw new IllegalArgumentException("'{' was never closed");
            }
            while (pointer < LEN) {
                final KVPair READKEY = parse(str, pointer, true);
                pointer = Integer.parseInt(READKEY.getKey());
                while (pointer < LEN && containsChar(
                    WHITESPACES, str.charAt(pointer)
                )) {
                    pointer++;
                }
                try {
                    if (str.charAt(pointer) != KEY_VALUE) {
                        throw new IllegalArgumentException(
                            "expect ':' after key"
                        );
                    }
                } catch (StringIndexOutOfBoundsException e) {
                    throw new IllegalArgumentException(
                        "expect ':' after key"
                    );
                }
                pointer++;
                if (pointer >= LEN) {
                    throw new IllegalArgumentException("expect value");
                }
                final KVPair READVALUE = parse(str, pointer, false);
                json.setProperty(READKEY.getValue(), READVALUE.getValue());
                pointer = Integer.parseInt(READVALUE.getKey());
                while (pointer < LEN && containsChar(
                    WHITESPACES, str.charAt(pointer)
                )) {
                    pointer++;
                }
                try {
                    achar = str.charAt(pointer);
                } catch (StringIndexOutOfBoundsException e) {
                    throw new IllegalArgumentException("'{' was never closed");
                }
                if (achar == OBJECT_END) {
                    return new KVPair(Integer.toString(pointer + 1), json);
                } else if (achar != ITEM_SEP) {
                    throw new IllegalArgumentException(
                        "expect ',' after value"
                    );
                }
                pointer++;
            }
            throw new IllegalArgumentException("'{' was never closed");
        }
        if (achar == ARRAY_START && !stringOnly) {
            SimpleJSON json = new SimpleJSON(new SimpleJSON[0]);
            int size = 0;
            pointer++;
            while (pointer < LEN && containsChar(
                WHITESPACES, str.charAt(pointer)
            )) {
                pointer++;
            }
            try {
                if (str.charAt(pointer) == ARRAY_END) {
                    return new KVPair(Integer.toString(pointer + 1), json);
                }
            } catch (StringIndexOutOfBoundsException e) {
                throw new IllegalArgumentException("'[' was never closed");
            }
            while (pointer < LEN) {
                final KVPair READVALUE = parse(str, pointer, false);
                json.setProperty(size, READVALUE.getValue());
                size++;
                pointer = Integer.parseInt(READVALUE.getKey());
                while (pointer < LEN && containsChar(
                    WHITESPACES, str.charAt(pointer)
                )) {
                    pointer++;
                }
                try {
                    achar = str.charAt(pointer);
                } catch (StringIndexOutOfBoundsException e) {
                    throw new IllegalArgumentException("'[' was never closed");
                }
                if (achar == ARRAY_END) {
                    return new KVPair(Integer.toString(pointer + 1), json);
                } else if (achar != ITEM_SEP) {
                    throw new IllegalArgumentException(
                        "expect ',' after element"
                    );
                }
                pointer++;
            }
            throw new IllegalArgumentException("'[' was never closed");
        }
        if (achar == NULL_IDENTIFIER.charAt(0) && !stringOnly) {
            try {
                final int LEN2 = NULL_IDENTIFIER.length();
                if (str.substring(pointer, pointer + LEN2)
                    .equals(NULL_IDENTIFIER)) {
                    return new KVPair(Integer.toString(pointer + LEN2), null);
                }
            } catch (StringIndexOutOfBoundsException ignored) {}
        }
        if (achar == FALSE_IDENTIFIER.charAt(0) && !stringOnly) {
            try {
                final int LEN2 = FALSE_IDENTIFIER.length();
                if (str.substring(pointer, pointer + LEN2)
                    .equals(FALSE_IDENTIFIER)) {
                    return new KVPair(Integer.toString(pointer + LEN2),
                                      new SimpleJSON(false));
                }
            } catch (StringIndexOutOfBoundsException ignored) {}
        }
        if (achar == TRUE_IDENTIFIER.charAt(0) && !stringOnly) {
            try {
                final int LEN2 = TRUE_IDENTIFIER.length();
                if (str.substring(pointer, pointer + LEN2)
                    .equals(TRUE_IDENTIFIER)) {
                    return new KVPair(Integer.toString(pointer + LEN2),
                                      new SimpleJSON(true));
                }
            } catch (StringIndexOutOfBoundsException ignored) {}
        }
        if (achar == STRING_WRAP) {
            pointer++;
            final StringBuilder SB = new StringBuilder();
            while (pointer < LEN) {
                achar = str.charAt(pointer);
                if (achar == STRING_WRAP) {
                    return new KVPair(Integer.toString(pointer + 1),
                                      new SimpleJSON(SB.toString()));
                }
                if (achar == ESCAPE_START) {
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                        if (!ESCAPE_RESULTS.containsKey(achar)) {
                            throw new IllegalArgumentException(
                                "invalid escape character"
                            );
                        }
                        final String ESCAPE_RESULT = ESCAPE_RESULTS.get(achar);
                        if (ESCAPE_RESULT.length() == 0) {
                            pointer++;
                            achar = str.charAt(pointer);
                            if (!UNICODE_ESCAPE_VALUES.containsKey(achar)) {
                                throw new IllegalArgumentException(
                                    "invalid u-escape character"
                                );
                            }
                            int cp = UNICODE_ESCAPE_VALUES.get(achar)
                                     .intValue() << 12;
                            pointer++;
                            achar = str.charAt(pointer);
                            if (!UNICODE_ESCAPE_VALUES.containsKey(achar)) {
                                throw new IllegalArgumentException(
                                    "invalid u-escape character"
                                );
                            }
                            cp |=
                            UNICODE_ESCAPE_VALUES.get(achar).intValue() << 8;
                            pointer++;
                            achar = str.charAt(pointer);
                            if (!UNICODE_ESCAPE_VALUES.containsKey(achar)) {
                                throw new IllegalArgumentException(
                                    "invalid u-escape character"
                                );
                            }
                            cp |=
                            UNICODE_ESCAPE_VALUES.get(achar).intValue() << 4;
                            pointer++;
                            achar = str.charAt(pointer);
                            if (!UNICODE_ESCAPE_VALUES.containsKey(achar)) {
                                throw new IllegalArgumentException(
                                    "invalid u-escape character"
                                );
                            }
                            cp |= UNICODE_ESCAPE_VALUES.get(achar).intValue();
                            SB.append((char)cp);
                        } else {
                            SB.append(ESCAPE_RESULT);
                        }
                        pointer++;
                        continue;
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException(
                            "unterminated escape sequence"
                        );
                    }
                }
                if (containsChar(WHITESPACES, achar) &&
                    !containsChar(STRING_ALLOWED_WHITESPACES, achar)) {
                    throw new IllegalArgumentException(
                        "invalid whitespace in string literal"
                    );
                }
                SB.append(achar);
                pointer++;
            }
            throw new IllegalArgumentException("'\"' was never closed");
        }
        if (!stringOnly) {
            final int SP = pointer;
            achar = str.charAt(pointer);
            if (achar == MINUS_SIGN) {
                pointer++;
                try {
                    achar = str.charAt(pointer);
                } catch (StringIndexOutOfBoundsException e) {
                    throw new IllegalArgumentException("expect digit");
                }
            }
            if (achar == ZERO) {
                pointer++;
                try {
                    achar = str.charAt(pointer);
                } catch (StringIndexOutOfBoundsException e) {
                    return new KVPair(
                        Integer.toString(pointer),
                        new SimpleJSON(new BigDecimal(str.substring(SP)))
                    );
                }
                if (achar == DECIMAL_POINT) {
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                        if (!containsChar(DIGITS, achar)) {
                            throw new IllegalArgumentException("expect digit");
                        }
                        pointer++;
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    achar = str.charAt(pointer);
                    if (!containsChar(EXPONENT, achar)) {
                        return new KVPair(
                            Integer.toString(pointer), new SimpleJSON(
                                new BigDecimal(str.substring(SP, pointer))
                            )
                        );
                    }
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar != PLUS_SIGN && achar != MINUS_SIGN &&
                        !containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar == PLUS_SIGN || achar == MINUS_SIGN) {
                        pointer++;
                        try {
                            achar = str.charAt(pointer);
                        } catch (StringIndexOutOfBoundsException e) {
                            throw new IllegalArgumentException("expect digit");
                        }
                    }
                    if (!containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    pointer++;
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    return new KVPair(
                        Integer.toString(pointer), new SimpleJSON(
                            new BigDecimal(str.substring(SP, pointer))
                        )
                    );
                }
                if (containsChar(EXPONENT, achar)) {
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar != PLUS_SIGN && achar != MINUS_SIGN &&
                        !containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar == PLUS_SIGN || achar == MINUS_SIGN) {
                        pointer++;
                        try {
                            achar = str.charAt(pointer);
                        } catch (StringIndexOutOfBoundsException e) {
                            throw new IllegalArgumentException("expect digit");
                        }
                    }
                    if (!containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    pointer++;
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    return new KVPair(
                        Integer.toString(pointer), new SimpleJSON(
                            new BigDecimal(str.substring(SP, pointer))
                        )
                    );
                }
                return new KVPair(
                    Integer.toString(pointer),
                    new SimpleJSON(new BigDecimal(str.substring(SP, pointer)))
                );
            }
            if (containsChar(DIGITS, achar)) {
                pointer++;
                try {
                    while (containsChar(
                        DIGITS, str.charAt(pointer)
                    )) {
                        pointer++;
                    }
                } catch (StringIndexOutOfBoundsException e) {
                    return new KVPair(
                        Integer.toString(pointer),
                        new SimpleJSON(new BigDecimal(str.substring(SP)))
                    );
                }
                achar = str.charAt(pointer);
                if (achar == DECIMAL_POINT) {
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                        if (!containsChar(DIGITS, achar)) {
                            throw new IllegalArgumentException("expect digit");
                        }
                        pointer++;
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    achar = str.charAt(pointer);
                    if (!containsChar(EXPONENT, achar)) {
                        return new KVPair(
                            Integer.toString(pointer), new SimpleJSON(
                                new BigDecimal(str.substring(SP, pointer))
                            )
                        );
                    }
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar != PLUS_SIGN && achar != MINUS_SIGN &&
                        !containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar == PLUS_SIGN || achar == MINUS_SIGN) {
                        pointer++;
                        try {
                            achar = str.charAt(pointer);
                        } catch (StringIndexOutOfBoundsException e) {
                            throw new IllegalArgumentException("expect digit");
                        }
                    }
                    if (!containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    pointer++;
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    return new KVPair(
                        Integer.toString(pointer), new SimpleJSON(
                            new BigDecimal(str.substring(SP, pointer))
                        )
                    );
                }
                if (containsChar(EXPONENT, achar)) {
                    pointer++;
                    try {
                        achar = str.charAt(pointer);
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar != PLUS_SIGN && achar != MINUS_SIGN &&
                        !containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException(
                            "expect digit, '+' or '-'"
                        );
                    }
                    if (achar == PLUS_SIGN || achar == MINUS_SIGN) {
                        pointer++;
                        try {
                            achar = str.charAt(pointer);
                        } catch (StringIndexOutOfBoundsException e) {
                            throw new IllegalArgumentException("expect digit");
                        }
                    }
                    if (!containsChar(DIGITS, achar)) {
                        throw new IllegalArgumentException("expect digit");
                    }
                    pointer++;
                    try {
                        while (containsChar(
                            DIGITS, str.charAt(pointer)
                        )) {
                            pointer++;
                        }
                    } catch (StringIndexOutOfBoundsException e) {
                        return new KVPair(
                            Integer.toString(pointer),
                            new SimpleJSON(new BigDecimal(str.substring(SP)))
                        );
                    }
                    return new KVPair(
                        Integer.toString(pointer), new SimpleJSON(
                            new BigDecimal(str.substring(SP, pointer))
                        )
                    );
                }
                return new KVPair(
                    Integer.toString(pointer),
                    new SimpleJSON(new BigDecimal(str.substring(SP, pointer)))
                );
            }
        }
        throw new IllegalArgumentException(stringOnly ? "expect '\"'" :
                                           "invalid token");
    }

    static {
        ESCAPE_RESULTS = new HashMap<Character, String>();
        ESCAPE_RESULTS.put('b', "\b");
        ESCAPE_RESULTS.put('f', "\f");
        ESCAPE_RESULTS.put('n', "\n");
        ESCAPE_RESULTS.put('r', "\r");
        ESCAPE_RESULTS.put('t', "\t");
        ESCAPE_RESULTS.put('u', "");
        ESCAPE_RESULTS.put('/', "/");
        ESCAPE_RESULTS.put('\\', "\\");
        ESCAPE_RESULTS.put('"', "\"");
        UNICODE_ESCAPE_VALUES = new HashMap<Character, Byte>();
        UNICODE_ESCAPE_VALUES.put('0', (byte)0);
        UNICODE_ESCAPE_VALUES.put('1', (byte)1);
        UNICODE_ESCAPE_VALUES.put('2', (byte)2);
        UNICODE_ESCAPE_VALUES.put('3', (byte)3);
        UNICODE_ESCAPE_VALUES.put('4', (byte)4);
        UNICODE_ESCAPE_VALUES.put('5', (byte)5);
        UNICODE_ESCAPE_VALUES.put('6', (byte)6);
        UNICODE_ESCAPE_VALUES.put('7', (byte)7);
        UNICODE_ESCAPE_VALUES.put('8', (byte)8);
        UNICODE_ESCAPE_VALUES.put('9', (byte)9);
        UNICODE_ESCAPE_VALUES.put('A', (byte)10);
        UNICODE_ESCAPE_VALUES.put('B', (byte)11);
        UNICODE_ESCAPE_VALUES.put('C', (byte)12);
        UNICODE_ESCAPE_VALUES.put('D', (byte)13);
        UNICODE_ESCAPE_VALUES.put('E', (byte)14);
        UNICODE_ESCAPE_VALUES.put('F', (byte)15);
        UNICODE_ESCAPE_VALUES.put('a', (byte)10);
        UNICODE_ESCAPE_VALUES.put('b', (byte)11);
        UNICODE_ESCAPE_VALUES.put('c', (byte)12);
        UNICODE_ESCAPE_VALUES.put('d', (byte)13);
        UNICODE_ESCAPE_VALUES.put('e', (byte)14);
        UNICODE_ESCAPE_VALUES.put('f', (byte)15);
    }

    /**
     * Types of JSON elements.
     */
    public static enum JSONType {
        /**
         * Example: {@code 0}, {@code -3}, {@code 4.5}, {@code 3e13},
         * {@code -0.34e+3}, {@code 444.50e-9}.
         */
        NUMBER,
        /**
         * Example: {@code ""}, {@code "abc123"}, {@code "\n\u0002"}.
         */
        STRING,
        /**
         * Only {@code false} and {@code true}.
         */
        BOOLEAN,
        /**
         * Example: {@code []}, {@code [123]}, {@code ["abc", [], 189, true]}.
         */
        ARRAY,
        /**
         * Example: {@code {}}, {@code {"key": "val"},
         * {@code {"num": 1, "name": "abc", "enabled": false, "extra": [{}]}}}
         */
        OBJECT,
        /**
         * Only {@code null}.
         */
        NULL;
    }

    public static class KVPair implements Entry<String, SimpleJSON> {
        /**
         * The key, in {@link java.lang.String}.
         */
        @NotNull
        public final String key;
        @NotNull
        private SimpleJSON val;

        /**
         * New a key-value pair with specified {@code key} and {@code val}. No
         * side effect.
         * @param key The key.
         * @param val The value.
         */
        @Contract(pure = true)
        public KVPair(@Nullable String key, @Nullable SimpleJSON val) {
            this.key = (key == null) ? "null" : key;
            this.val = (val == null) ? new SimpleJSON((String)null) : val;
        }

        /**
         * Get the key. No side effect.
         * @return The key, in {@link java.lang.String}.
         */
        @Contract(pure = true)
        @NotNull
        public String getKey() {
            return this.key;
        }

        /**
         * Get the value. No side effect.
         * @return The value, in {@link SimpleJSON}.
         */
        @Contract(pure = true)
        @NotNull
        public SimpleJSON getValue() {
            return this.val;
        }

        /**
         * Set the value.
         * @return The previous value, in {@link SimpleJSON}.
         */
        @Contract(pure = false)
        @NotNull
        public SimpleJSON setValue(@Nullable SimpleJSON value) {
            final SimpleJSON RES = this.val;
            this.val = (value == null) ? new SimpleJSON((String)null) : value;
            return RES;
        }
    }

    /**
     * The type of the JSON.
     * @see JSONType
     */
    public final JSONType type;
    private final @Nullable String content;
    private final @Nullable ArrayList<SimpleJSON> ifArray;
    private final @Nullable ArrayList<KVPair> ifObject;

    /**
     * New a JSON with number type, value from {@code num}. No side effect.
     * @param num The value.
     */
    @Contract(pure = true)
    public SimpleJSON(long num) {
        this.type = JSONType.NUMBER;
        this.content = Long.toString(num);
        this.ifArray = null;
        this.ifObject = null;
    }

    /**
     * New a JSON with number type, value from {@code num}.
     * @param num The value.
     * @throws IllegalArgumentException If {@code num} is infinity or NaN.
     */
    public SimpleJSON(double num) throws IllegalArgumentException {
        if (num == Double.POSITIVE_INFINITY ||
            num == Double.NEGATIVE_INFINITY) {
            throw new IllegalArgumentException(
                "Infinity values are not accepted in JSON"
            );
        }
        if (num != num) {
            throw new IllegalArgumentException("NaN is not accepted in JSON");
        }
        this.type = JSONType.NUMBER;
        this.content = Double.toString(num);
        this.ifArray = null;
        this.ifObject = null;
    }

    /**
     * New a JSON with number type, value from {@code num} if {@code num} is
     * not null; otherwise new a JSON with null type. No side effect.
     * @param num The value.
     */
    @Contract(pure = true)
    public SimpleJSON(@Nullable BigDecimal num) {
        this.type = (num == null) ? JSONType.NULL : JSONType.NUMBER;
        this.content = (num == null) ? null : num.toString();
        this.ifArray = null;
        this.ifObject = null;
    }

    /**
     * New a JSON with string type, value from {@code str} if {@code str} is
     * not null; otherwise new a JSON with null type. No side effect.
     * @param str The value.
     */
    @Contract(pure = true)
    public SimpleJSON(@Nullable String str) {
        this.type = (str == null) ? JSONType.NULL : JSONType.STRING;
        this.content = str;
        this.ifArray = null;
        this.ifObject = null;
    }

    /**
     * New a JSON with boolean type, value from {@code bl}. No side effect.
     * @param bl The value.
     */
    @Contract(pure = true)
    public SimpleJSON(boolean bl) {
        this.type = JSONType.BOOLEAN;
        this.content = bl ? "" : null;
        this.ifArray = null;
        this.ifObject = null;
    }

    /**
     * New a JSON with array type. Its elements are from {@code arr}. No side
     * effect.
     * @param arr The elements of the array.
     */
    @Contract(pure = true)
    public SimpleJSON(@Nullable SimpleJSON... arr) {
        if (arr == null) {
            this.type = JSONType.NULL;
            this.content = null;
            this.ifArray = null;
            this.ifObject = null;
        } else {
            this.type = JSONType.ARRAY;
            this.content = null;
            this.ifArray = new ArrayList<SimpleJSON>(arr.length);
            this.ifObject = null;
            for (SimpleJSON i : arr) {
                this.ifArray.add((i==null) ? new SimpleJSON((String)null) : i);
            }
        }
    }

    /**
     * New a JSON with array type. Its elements are from {@code arr}.
     * @param arr The elements of the array.
     */
    public SimpleJSON(@Nullable Iterable<? extends SimpleJSON> arr) {
        if (arr == null) {
            this.type = JSONType.NULL;
            this.content = null;
            this.ifArray = null;
            this.ifObject = null;
        } else {
            this.type = JSONType.ARRAY;
            this.content = null;
            this.ifArray = new ArrayList<SimpleJSON>();
            this.ifObject = null;
            for (SimpleJSON i : arr) {
                this.ifArray.add((i == null) ? new SimpleJSON((String)null) : i);
            }
        }
    }

    /**
     * New a JSON with object type, ordered. Its key-value pairs are from
     * {@code obj}.
     * @param obj The key-value pairs of the object.
     * @throws NullPointerException if there is a null
     * {@link java.util.Map.Entry}&lt;?, ? extends {@link SimpleJSON}&gt;.
     */
    @SuppressWarnings("unchecked")
    public SimpleJSON(@Nullable Entry<?, ? extends SimpleJSON>... obj)
    throws NullPointerException {
        if (obj == null) {
            this.type = JSONType.NULL;
            this.content = null;
            this.ifArray = null;
            this.ifObject = null;
        } else {
            this.type = JSONType.OBJECT;
            this.content = null;
            this.ifArray = null;
            this.ifObject = new ArrayList<KVPair>();
            for (Entry<?, ? extends SimpleJSON> i : obj) {
                final Object KEY = i.getKey();
                final String SKEY = (KEY == null) ? null : KEY.toString();
                final SimpleJSON VAL = i.getValue();
                final int SIZE = this.ifObject.size();
                for (int j = 0; j < SIZE; j++) {
                    if (this.ifObject.get(j).getKey().equals(SKEY)) {
                        this.ifObject.remove(j);
                        break;
                    }
                }
                this.ifObject.add(new KVPair(SKEY, VAL));
            }
        }
    }

    /**
     * New a JSON with object type. Its key-value pairs are from {@code obj}.
     * @param obj The key-value pairs of the object.
     */
    public SimpleJSON(@Nullable Map<?, ? extends SimpleJSON> obj) {
        if (obj == null) {
            this.type = JSONType.NULL;
            this.content = null;
            this.ifArray = null;
            this.ifObject = null;
        } else {
            this.type = JSONType.OBJECT;
            this.content = null;
            this.ifArray = null;
            this.ifObject = new ArrayList<KVPair>();
            for (Entry<?, ? extends SimpleJSON> i : obj.entrySet()) {
                final Object KEY = i.getKey();
                final String SKEY = (KEY == null) ? null : KEY.toString();
                this.ifObject.add(new KVPair(SKEY, i.getValue()));
            }
        }
    }

    /**
     * Return whether the JSON is null. No side effect.
     * @return A boolean reflects whether the JSON is null.
     */
    @Contract(pure = true)
    public boolean isNull() {
        return this.type == null || this.type == JSONType.NULL;
    }

    /**
     * Dump the JSON to a string.
     * @return The dumped string.
     */
    @Override
    @Contract(pure = true)
    public String toString() {
        if (this.isNull()) {
            return NULL_IDENTIFIER;
        }
        switch (this.type) {
            case NUMBER: return this.content;
            case STRING: {
                String body = this.content.replace(
                    "\\","\\\\"
                ).replace("\"","\\\"");
                for (char i : WHITESPACES.toCharArray()) {
                    if (!containsChar(STRING_ALLOWED_WHITESPACES, i)) {
                        body = body.replace(String.valueOf(i),
                                            String.format("\\u%04x", (int)i));
                    }
                }
                return "\"" + body + "\"";
            }
            case BOOLEAN: return (this.content == null) ? FALSE_IDENTIFIER :
                                 TRUE_IDENTIFIER;
            case ARRAY: {
                if (this.ifArray.isEmpty()) {
                    return "[]";
                }
                final StringBuilder SB = new StringBuilder();
                SB.append(ARRAY_START);
                boolean firstPassed = false;
                for (SimpleJSON i : this.ifArray) {
                    if (firstPassed) {
                        SB.append(", ");
                    }
                    SB.append(i.toString());
                    firstPassed = true;
                }
                SB.append(ARRAY_END);
                return SB.toString();
            }
            case OBJECT: {
                if (this.ifObject.isEmpty()) {
                    return "{}";
                }
                final StringBuilder SB = new StringBuilder();
                SB.append(OBJECT_START);
                boolean firstPassed = false;
                for (KVPair i : this.ifObject) {
                    if (firstPassed) {
                        SB.append(", ");
                    }
                    SB.append(STRING_WRAP);
                    SB.append(i.getKey().replace("\\", "\\\\")
                              .replace("\"", "\\\""));
                    SB.append(STRING_WRAP);
                    SB.append(": ");
                    SB.append(i.getValue().toString());
                    firstPassed = true;
                }
                SB.append(OBJECT_END);
                return SB.toString();
            }
        }
        return "";
    }

    @Override
    @Contract(value = "null -> false", pure = true)
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof SimpleJSON)) {
            return false;
        }
        SimpleJSON that = (SimpleJSON)obj;
        if (this.isNull()) {
            return that.isNull();
        }
        if (this.type != that.type) {
            return false;
        }
        switch (this.type) {
            case NUMBER: return new BigDecimal(this.content).compareTo(
                new BigDecimal(that.content)
            ) == 0;
            case STRING: return this.content.equals(that.content);
            case BOOLEAN: return (this.content==null) == (that.content==null);
            case ARRAY: return this.ifArray.equals(that.ifArray);
            case OBJECT: return this.ifObject.equals(that.ifObject);
        }
        return false;
    }

    @Override
    @Contract(pure = true)
    public int hashCode() {
        return ((this.type == null) ? JSONType.NULL : this.type).hashCode() ^
               ((this.content == null) ? 0 : this.content.hashCode()) ^
               ((this.ifArray == null) ? 0 : this.ifArray.hashCode()) ^
               ((this.ifObject == null) ? 0 : this.ifObject.hashCode());
    }

    /**
     * Get the number of the JSON.
     * @return The number, in {@link java.math.BigDecimal}.
     * @throws IllegalArgumentException if the JSON is not as number.
     */
    public BigDecimal getAsNumber() throws IllegalArgumentException {
        if (this.type != JSONType.NUMBER) {
            throw new IllegalArgumentException("JSON is not as number");
        }
        return new BigDecimal(this.content);
    }

    /**
     * Get the string of the JSON.
     * @return The string, in {@link java.lang.String}.
     * @throws IllegalArgumentException if the JSON is not as string.
     */
    public String getAsString() throws IllegalArgumentException {
        if (this.type != JSONType.STRING) {
            throw new IllegalArgumentException("JSON is not as string");
        }
        return this.content;
    }

    /**
     * Get the boolean value of the JSON.
     * @return The boolean, in primitive boolean.
     * @throws IllegalArgumentException if the JSON is not as boolean.
     */
    public boolean getAsBoolean() throws IllegalArgumentException {
        if (this.type != JSONType.BOOLEAN) {
            throw new IllegalArgumentException("JSON is not as boolean");
        }
        return this.content != null;
    }

    /**
     * Get the property(subscript) of the JSON.
     * @param sub The subscript.
     * @return The property, in {@link SimpleJSON}. Return {@code null} if the
     * JSON is in number or boolean, or if the subscript to the JSON in string
     * or array is invalid or out of range, or if the subscript to the JSON in
     * object is absent.
     * @throws IllegalArgumentException if the JSON is as null.
     */
    public @Nullable SimpleJSON getProperty(Object sub) throws
    IllegalArgumentException {
        if (this.isNull()) {
            throw new IllegalArgumentException(
                "Cannot read properties of null"
            );
        }
        if (sub instanceof SimpleJSON) {
            final SimpleJSON JSON = (SimpleJSON)sub;
            if (!JSON.isNull()) {
                switch (JSON.type) {
                    case NUMBER: {
                        sub = JSON.getAsNumber().stripTrailingZeros()
                              .toPlainString();
                        break;
                    }
                    case STRING: sub = JSON.getAsString();
                }
            }
        }
        switch (this.type) {
            case STRING: {
                try {
                    final BigInteger BI = new BigInteger(sub.toString());
                    if (BI.compareTo(BigInteger.ZERO) >= 0 &&
                        BI.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) <=
                        0) {
                        return new SimpleJSON(String.valueOf(
                            this.content.charAt(BI.intValue())
                        ));
                    }
                } catch (NumberFormatException ignored) {}
                catch (ArithmeticException ignored) {}
                catch (StringIndexOutOfBoundsException ignored) {}
                return null;
            }
            case ARRAY: {
                try {
                    final BigInteger BI = new BigInteger(sub.toString());
                    if (BI.compareTo(BigInteger.ZERO) >= 0 &&
                        BI.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) <=
                        0) {
                        return this.ifArray.get(BI.intValue());
                    }
                } catch (NumberFormatException ignored) {}
                catch (ArithmeticException ignored) {}
                catch (StringIndexOutOfBoundsException ignored) {}
                return null;
            }
            case OBJECT: {
                final String STR =
                (sub == null) ? NULL_IDENTIFIER : sub.toString();
                for (KVPair i : this.ifObject) {
                    if (i.getKey().equals(STR)) {
                        return i.getValue();
                    }
                }
                return null;
            }
        }
        return null;
    }

    /**
     * Set the property(subscript) of the JSON.
     * @param sub The subscript.
     * @param json The JSON to set with.
     * @return A boolean reflects whether you were capable set the property.
     * @throws IllegalArgumentException if the JSON is as null.
     */
    public boolean setProperty(Object sub, SimpleJSON json)
    throws IllegalArgumentException {
        if (this.isNull()) {
            throw new IllegalArgumentException(
                "Cannot set properties of null"
            );
        }
        if (sub instanceof SimpleJSON) {
            final SimpleJSON JSON = (SimpleJSON)sub;
            if (!JSON.isNull()) {
                switch (JSON.type) {
                    case NUMBER: {
                        sub = JSON.getAsNumber().stripTrailingZeros()
                              .toPlainString();
                        break;
                    }
                    case STRING: sub = JSON.getAsString();
                }
            }
        }
        switch (this.type) {
            case ARRAY: {
                try {
                    final BigInteger BI = new BigInteger(sub.toString());
                    if (BI.compareTo(BigInteger.ZERO) >= 0 &&
                        BI.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) <=
                        0) {
                        final int INDEX = BI.intValue();
                        for (int size = this.ifArray.size(); size <= INDEX;
                             size++) {
                            this.ifArray.add(new SimpleJSON((String)null));
                        }
                        if (INDEX >= 0) {
                            this.ifArray
                            .set(INDEX, (json == null) ?
                                        new SimpleJSON((String)null) : json);
                            return true;
                        }
                    }
                } catch (NumberFormatException ignored) {}
                catch (ArithmeticException ignored) {}
                catch (StringIndexOutOfBoundsException ignored) {}
                return false;
            }
            case OBJECT: {
                final String STR =
                (sub == null) ? NULL_IDENTIFIER : sub.toString();
                for (KVPair i : this.ifObject) {
                    if (i.getKey().equals(STR)) {
                        i.setValue(json);
                        return true;
                    }
                }
                this.ifObject.add(new KVPair(STR, json));
                return true;
            }
        }
        return false;
    }

    /**
     * Delete the property(subscript) of the JSON. If the JSON is as array,
     * then try to delete the element at the index and move all elements after
     * the index 1 step lower; If the JSON is as object, then try to remove the
     * key-value pair.
     * @param sub The subscript.
     * @return A boolean reflects whether the deletion is succeeded.
     * @throws IllegalArgumentException if the JSON is as null.
     */
    public boolean deleteProperty(Object sub) throws IllegalArgumentException {
        if (this.isNull()) {
            throw new IllegalArgumentException(
                "Cannot set properties of null"
            );
        }
        if (sub instanceof SimpleJSON) {
            final SimpleJSON JSON = (SimpleJSON)sub;
            if (!JSON.isNull()) {
                switch (JSON.type) {
                    case NUMBER: {
                        sub = JSON.getAsNumber().stripTrailingZeros()
                              .toPlainString();
                        break;
                    }
                    case STRING: sub = JSON.getAsString();
                }
            }
        }
        switch (this.type) {
            case ARRAY: {
                try {
                    final BigInteger BI = new BigInteger(sub.toString());
                    if (BI.compareTo(BigInteger.ZERO) >= 0 &&
                        BI.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) <=
                        0) {
                        final int INDEX = BI.intValue();
                        this.ifArray.remove(INDEX);
                        return true;
                    }
                } catch (NumberFormatException ignored) {}
                catch (ArithmeticException ignored) {}
                catch (StringIndexOutOfBoundsException ignored) {}
                catch (IndexOutOfBoundsException ignored) {}
                return false;
            }
            case OBJECT: {
                final String STR =
                (sub == null) ? NULL_IDENTIFIER : sub.toString();
                int index = 0;
                for (KVPair i : this.ifObject) {
                    if (i.getKey().equals(STR)) {
                        this.ifObject.remove(index);
                        return true;
                    }
                    index++;
                }
                return false;
            }
        }
        return false;
    }

    /**
     * Get the length of the JSON.
     * @return The length of the JSON as string, array or object in
     * {@link java.lang.Integer}, or null otherwise.
     * @throws IllegalArgumentException if the JSON is as null.
     */
    public @Nullable Integer length() throws IllegalArgumentException {
        if (this.isNull()) {
            throw new IllegalArgumentException(
                "Cannot read properties of null"
            );
        }
        switch (this.type) {
            case STRING: return this.content.length();
            case ARRAY: return this.ifArray.size();
            case OBJECT: return this.ifObject.size();
        }
        return null;
    }

    @Contract(pure = true)
    public @Nullable ArrayList<SimpleJSON> toList() {
        if (this.isNull()) {
            return null;
        }
        switch (this.type) {
            case STRING: {
                final ArrayList<SimpleJSON> RES = new ArrayList<SimpleJSON>();
                for (char i : this.content.toCharArray()) {
                    RES.add(new SimpleJSON(String.valueOf(i)));
                }
                return RES;
            }
            case ARRAY: return new ArrayList<SimpleJSON>(this.ifArray);
            case OBJECT: {
                final ArrayList<SimpleJSON> RES = new ArrayList<SimpleJSON>();
                for (KVPair i : this.ifObject) {
                    RES.add(new SimpleJSON(i.getKey()));
                }
                return RES;
            }
        }
        return null;
    }

    /**
     * Return all keys of the JSON object.
     * @return All keys of the JSON objects, ordered, in an
     * {@link java.util.ArrayList}&lt;{@link java.lang.String}&gt;.
     * @throws IllegalArgumentException if the JSON is not as object.
     */
    public ArrayList<String> getKeys() throws IllegalArgumentException {
        if (this.type != JSONType.OBJECT) {
            throw new IllegalArgumentException("JSON is not as object");
        }
        final ArrayList<String> RES = new ArrayList<String>();
        for (KVPair i : this.ifObject) {
            RES.add(i.getKey());
        }
        return RES;
    }

    /**
     * Return all values of the JSON object.
     * @return All values of the JSON objects, ordered, in an
     * {@link java.util.ArrayList}&lt;{@link SimpleJSON}&gt;.
     * @throws IllegalArgumentException if the JSON is not as object.
     */
    public ArrayList<SimpleJSON> getValues() throws IllegalArgumentException {
        if (this.type != JSONType.OBJECT) {
            throw new IllegalArgumentException("JSON is not as object");
        }
        final ArrayList<SimpleJSON> RES = new ArrayList<SimpleJSON>();
        for (KVPair i : this.ifObject) {
            RES.add(i.getValue());
        }
        return RES;
    }
}
