/*
 * Decompiled with CFR 0.152.
 */
package me.artificial.autoserver.velocity;

import com.velocitypowered.api.proxy.Player;
import com.velocitypowered.api.proxy.server.RegisteredServer;
import com.velocitypowered.api.scheduler.ScheduledTask;
import com.velocitypowered.api.scheduler.Scheduler;
import java.io.IOException;
import java.net.Socket;
import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import me.artificial.autoserver.velocity.AutoServer;
import me.artificial.autoserver.velocity.AutoServerLogger;
import me.artificial.autoserver.velocity.Messenger;
import me.artificial.autoserver.velocity.ServerStatus;
import me.artificial.autoserver.velocity.startable.LocalStartable;
import me.artificial.autoserver.velocity.startable.RemoteStartable;
import me.artificial.autoserver.velocity.startable.Startable;

public class ServerManager {
    private final AutoServerLogger logger;
    private final AutoServer plugin;
    private final HashSet<String> startingServers = new HashSet();
    private final HashMap<Player, String> queuePlayers = new HashMap();
    private final Map<String, ServerStatus> serverStatusCache = new ConcurrentHashMap<String, ServerStatus>();
    private final Map<String, ScheduledTask> shutdownScheduledTask = new ConcurrentHashMap<String, ScheduledTask>();

    public ServerManager(AutoServer plugin) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
    }

    public CompletableFuture<String> startServer(RegisteredServer server) {
        String serverName = server.getServerInfo().getName();
        if (this.startingServers.contains(serverName)) {
            this.logger.debug("Server {} is already starting", serverName);
            return CompletableFuture.completedFuture("Server is already starting.");
        }
        this.startingServers.add(serverName);
        this.logger.debug("Attempting to start server: {}", serverName);
        Startable startableStrategy = this.getServerStrategy(server);
        return ((CompletableFuture)this.isServerOnline(server).thenCompose(isOnline -> {
            if (isOnline.booleanValue()) {
                this.moveQueuedPlayersToServer(server);
                return CompletableFuture.completedFuture("Server already running");
            }
            return startableStrategy.start().thenCompose(result -> this.waitForServerToBecomeResponsive(server).thenApply(isResponsive -> {
                if (isResponsive.booleanValue()) {
                    this.moveQueuedPlayersToServer(server);
                    return "Server started and is responsive.";
                }
                throw new RuntimeException("Server started but is not responsive.");
            }));
        })).whenComplete((result, ex) -> {
            this.startingServers.remove(serverName);
            if (ex != null) {
                this.logger.error("Failed to start server: {}", ex.getMessage());
            }
        });
    }

    public CompletableFuture<String> stopServer(RegisteredServer server) {
        String serverName = server.getServerInfo().getName();
        if (this.getServerStatus(server).isStopping()) {
            this.logger.debug("Server {} is already stopping", serverName);
            return CompletableFuture.completedFuture("Server is already stopping.");
        }
        this.getServerStatus(server).setStatus(ServerStatus.Status.STOPPING);
        this.logger.info("Attempting to stop server: {}", serverName);
        Startable startableStrategy = this.getServerStrategy(server);
        return ((CompletableFuture)this.isServerOnline(server).thenCompose(isOnline -> {
            if (!isOnline.booleanValue()) {
                return CompletableFuture.completedFuture("Server already stopped");
            }
            return startableStrategy.stop().thenCompose(result -> {
                long shutdownDelay = this.plugin.getConfig().getShutdownDelay(server);
                try {
                    this.logger.info("Sleeping for {} seconds before checking if server has stopped.", shutdownDelay);
                    Thread.sleep(shutdownDelay * 1000L);
                }
                catch (InterruptedException e) {
                    this.logger.warn("Stop delay sleep interrupted: {}", e.getMessage());
                    Thread.currentThread().interrupt();
                }
                return this.isServerOnline(server).thenApply(isOnline2 -> {
                    if (isOnline2.booleanValue()) {
                        throw new RuntimeException("Failed to stop server.");
                    }
                    return "Server stopped.";
                });
            });
        })).whenComplete((result, ex) -> {
            if (ex != null) {
                this.logger.error("Failed to stop server: {}", ex.getMessage());
                this.getServerStatus(server).setStatus(ServerStatus.Status.UNKNOWN);
            } else {
                this.getServerStatus(server).setStatus(ServerStatus.Status.STOPPED);
            }
        });
    }

    public CompletableFuture<Boolean> isServerOnline(RegisteredServer server) {
        return this.pingServer(server, 5000);
    }

    public CompletableFuture<Boolean> isServerResponsive(RegisteredServer server) {
        String serverName = server.getServerInfo().getName();
        ServerStatus cachedStatus = this.getServerStatus(server);
        if (cachedStatus.is(ServerStatus.Status.STOPPED)) {
            this.logger.debug("Cache check for server '{}' is OFFLINE", serverName);
            return CompletableFuture.completedFuture(false);
        }
        if (!server.getPlayersConnected().isEmpty()) {
            this.logger.debug("Players detected on server '{}', assuming ONLINE", serverName);
            return CompletableFuture.completedFuture(true);
        }
        return this.pingServer(server, 50);
    }

    public ServerStatus getServerStatus(RegisteredServer server) {
        String serverName = server.getServerInfo().getName();
        if (!this.serverStatusCache.containsKey(serverName)) {
            this.serverStatusCache.put(serverName, new ServerStatus());
            try {
                this.isServerOnline(server).get();
            }
            catch (InterruptedException | ExecutionException exception) {
                // empty catch block
            }
        }
        return this.serverStatusCache.get(serverName);
    }

    public void queuePlayerForServerJoin(Player player, String serverName) {
        this.queuePlayers.put(player, serverName);
    }

    public void scheduleShutdownServer(RegisteredServer server) {
        assert (server != null);
        this.logger.trace("scheduleShutdownServer: {}", server.getServerInfo().getName());
        long autoShutdownDelay = this.plugin.getConfig().getAutoShutdownDelay(server);
        if (autoShutdownDelay <= 0L) {
            return;
        }
        assert (!this.shutdownScheduledTask.containsKey(server.getServerInfo().getName())) : "Server already has task scheduled for shutdown.";
        this.logger.info("Scheduling shutdown of server {} in {}", server.getServerInfo().getName(), autoShutdownDelay);
        Scheduler.TaskBuilder taskBuilder = this.plugin.getProxy().getScheduler().buildTask((Object)this.plugin, () -> {
            assert (server.getPlayersConnected().isEmpty()) : "Server is not empty";
            this.stopServer(server).whenComplete((result, ex) -> {
                if (ex != null) {
                    this.logger.error("error: {}", ex.getMessage());
                } else {
                    this.logger.info("Message: {}", result);
                }
            });
        }).delay(Duration.ofSeconds(autoShutdownDelay));
        ScheduledTask scheduledTask = taskBuilder.schedule();
        this.shutdownScheduledTask.put(server.getServerInfo().getName(), scheduledTask);
    }

    public void cancelShutdownServer(RegisteredServer server) {
        String serverName = server.getServerInfo().getName();
        if (this.shutdownScheduledTask.containsKey(serverName)) {
            this.logger.info("Cancelling auto shutdown: {}", serverName);
            this.shutdownScheduledTask.get(serverName).cancel();
            this.shutdownScheduledTask.remove(serverName);
        }
    }

    public void validateServers(Collection<RegisteredServer> servers) {
        this.logger.trace("Validating Server status...", new Object[0]);
        for (RegisteredServer server : servers) {
            this.pingServer(server, 5000).thenApply(isOnline -> {
                if (isOnline.booleanValue() && server.getPlayersConnected().isEmpty()) {
                    String serverName = server.getServerInfo().getName();
                    if (this.shutdownScheduledTask.containsKey(serverName)) {
                        this.logger.trace("Server {} is already scheduled to stop", serverName);
                        return null;
                    }
                    this.scheduleShutdownServer(server);
                }
                return null;
            });
        }
    }

    private Startable getServerStrategy(RegisteredServer server) {
        Optional<Boolean> remote = this.plugin.getConfig().isRemoteServer(server);
        if (remote.isPresent() && remote.get().booleanValue()) {
            return new RemoteStartable(this.plugin, server);
        }
        return new LocalStartable(this.plugin, server);
    }

    private CompletableFuture<Boolean> pingServer(RegisteredServer server, int pingTimeout) {
        String serverName = server.getServerInfo().getName();
        this.logger.debug("Pinging server {}...", serverName);
        return ((CompletableFuture)server.ping().orTimeout(pingTimeout, TimeUnit.MILLISECONDS).thenApply(serverPing -> {
            this.logger.debug("ping success {} is {}online{}", serverName, "\u001b[0;32m", "\u001b[0m");
            if (!this.getServerStatus(server).isStopping()) {
                this.getServerStatus(server).setStatus(ServerStatus.Status.RUNNING);
            }
            return true;
        })).exceptionallyCompose(e -> {
            String msg;
            this.logger.debug("ping failed for {}: {}", serverName, e.getMessage());
            Throwable cause = e.getCause() != null ? e.getCause() : e;
            String string = msg = cause.getMessage() != null ? cause.getMessage() : "";
            if (msg.contains("A packet did not decode successfully")) {
                this.logger.debug("failed to decode packet, likely online trying socket connect", new Object[0]);
                return CompletableFuture.supplyAsync(() -> {
                    Boolean bl;
                    Socket socket = new Socket();
                    try {
                        socket.connect(server.getServerInfo().getAddress());
                        this.logger.warn("Socket connection to {} succeeded, treating as online.", serverName);
                        if (!this.getServerStatus(server).isStopping()) {
                            this.getServerStatus(server).setStatus(ServerStatus.Status.RUNNING);
                        }
                        bl = true;
                    }
                    catch (Throwable throwable) {
                        try {
                            try {
                                socket.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                            throw throwable;
                        }
                        catch (IOException ioe) {
                            this.logger.warn("Socket connection to {} failed after ping error.", serverName);
                            if (!this.getServerStatus(server).isStarting()) {
                                this.getServerStatus(server).setStatus(ServerStatus.Status.STOPPED);
                            }
                            return false;
                        }
                    }
                    socket.close();
                    return bl;
                });
            }
            this.logger.debug("ping failed {} is {}offline{}", serverName, "\u001b[0;31m", "\u001b[0m");
            if (!this.getServerStatus(server).isStarting()) {
                this.getServerStatus(server).setStatus(ServerStatus.Status.STOPPED);
            }
            return CompletableFuture.completedFuture(false);
        });
    }

    private CompletableFuture<Boolean> waitForServerToBecomeResponsive(RegisteredServer server) {
        return CompletableFuture.supplyAsync(() -> {
            int retires = 10;
            int delayBetweenRetries = 5;
            long startupDelay = this.plugin.getConfig().getStartUpDelay(server);
            try {
                this.logger.info("Sleeping for {} seconds before checking if server has started.", startupDelay);
                Thread.sleep(startupDelay * 1000L);
            }
            catch (InterruptedException e) {
                this.logger.warn("Ping delay sleep interrupted: {}", e.getMessage());
                Thread.currentThread().interrupt();
            }
            while (retires > 0) {
                try {
                    if (this.pingServer(server, 5000).get().booleanValue()) {
                        this.logger.info("Server {} is {}online{}. Moving queued players...", server.getServerInfo().getName(), "\u001b[0;32m", "\u001b[0m");
                        return true;
                    }
                    this.logger.debug("Failed to ping server {}. Retrying in {} seconds.", server.getServerInfo().getName(), delayBetweenRetries);
                }
                catch (InterruptedException | ExecutionException e) {
                    this.logger.debug("Failed to ping server {}: {}. Retrying in {} seconds.", server.getServerInfo().getName(), e.getMessage(), delayBetweenRetries);
                }
                if (--retires <= 0) continue;
                try {
                    Thread.sleep((long)delayBetweenRetries * 1000L);
                }
                catch (InterruptedException e) {
                    this.logger.warn("Ping retry sleep interrupted: {}", e.getMessage());
                    Thread.currentThread().interrupt();
                }
            }
            return false;
        });
    }

    private void moveQueuedPlayersToServer(RegisteredServer server) {
        this.queuePlayers.forEach((player, serverName) -> {
            if (serverName.equals(server.getServerInfo().getName()) && player.isActive()) {
                if (player.getCurrentServer().isPresent()) {
                    Messenger.send(player, this.plugin.getConfig().getMessage("notify").orElse(""), serverName);
                    this.plugin.getProxy().getScheduler().buildTask((Object)this.plugin, () -> {
                        this.plugin.internalTransfer((Player)player);
                        player.createConnectionRequest(server).connect().whenComplete((result, throwable) -> {
                            if (throwable != null) {
                                Messenger.send(player, this.plugin.getConfig().getMessage("failed").orElse(""), serverName);
                                this.logger.error("Failed to connect player to server {}", throwable.getMessage());
                            } else {
                                this.logger.info("Player {} successfully moved to server {}", player.getUsername(), serverName);
                            }
                            this.queuePlayers.remove(player);
                        });
                    }).delay(5L, TimeUnit.SECONDS).schedule();
                } else {
                    player.createConnectionRequest(server).connect().whenComplete((result, throwable) -> {
                        if (throwable != null) {
                            Messenger.send(player, this.plugin.getConfig().getMessage("failed").orElse(""), serverName);
                            this.logger.error("Failed to connect player to server {}", throwable.getMessage());
                        } else {
                            this.logger.info("Player {} successfully moved to server {}", player.getUsername(), serverName);
                        }
                        this.queuePlayers.remove(player);
                    });
                }
            }
        });
    }
}

