package io.github.fishstiz.packed_packs.gui.screens;

import com.google.common.collect.ImmutableList;
import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.components.contextmenu.*;
import io.github.fishstiz.fidgetz.gui.layouts.FlexLayout;
import io.github.fishstiz.fidgetz.gui.renderables.ColoredRect;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.Sprite;
import io.github.fishstiz.fidgetz.gui.shapes.Size;
import io.github.fishstiz.fidgetz.util.debounce.ImmediateDebouncer;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.compat.ModAdditions;
import io.github.fishstiz.packed_packs.config.Config;
import io.github.fishstiz.packed_packs.config.Folder;
import io.github.fishstiz.packed_packs.config.Preferences;
import io.github.fishstiz.packed_packs.gui.components.contextmenu.DirectoryMenuItem;
import io.github.fishstiz.packed_packs.gui.components.contextmenu.PackMenuHeader;
import io.github.fishstiz.packed_packs.gui.components.pack.*;
import io.github.fishstiz.packed_packs.gui.layouts.pack.AvailablePacksLayout;
import io.github.fishstiz.packed_packs.gui.layouts.pack.CurrentPacksLayout;
import io.github.fishstiz.packed_packs.gui.layouts.pack.PackLayout;
import io.github.fishstiz.packed_packs.gui.metadata.Toggleable;
import io.github.fishstiz.packed_packs.pack.PackGroup;
import io.github.fishstiz.packed_packs.pack.PackWatcher;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionModelAccessor;
import io.github.fishstiz.packed_packs.util.ToastUtil;
import io.github.fishstiz.packed_packs.util.constants.Theme;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.pack.PackRepositoryHelper;
import io.github.fishstiz.packed_packs.config.Profile;
import io.github.fishstiz.packed_packs.gui.layouts.*;
import io.github.fishstiz.packed_packs.gui.components.events.*;
import io.github.fishstiz.packed_packs.gui.history.HistoryManager;
import io.github.fishstiz.packed_packs.gui.history.Restorable;
import io.github.fishstiz.packed_packs.gui.metadata.PackSelectionScreenArgs;
import io.github.fishstiz.packed_packs.transform.mixin.gui.HeaderAndFooterLayoutAccess;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionScreenAccessor;
import io.github.fishstiz.packed_packs.util.PackUtil.PathValidationResults;
import io.github.fishstiz.packed_packs.util.ResourceUtil;
import io.github.fishstiz.packed_packs.util.lang.CollectionsUtil;
import io.github.fishstiz.packed_packs.util.lang.ObjectsUtil;
import it.unimi.dsi.fastutil.booleans.BooleanConsumer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_3288;
import net.minecraft.class_332;
import net.minecraft.class_403;
import net.minecraft.class_410;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_5375;
import net.minecraft.class_7919;
import net.minecraft.class_8016;
import net.minecraft.class_8132;
import net.minecraft.class_8667;
import net.minecraft.class_8669;

import static net.minecraft.class_3675.field_31986;
import static net.minecraft.class_3675.field_31947;
import static io.github.fishstiz.packed_packs.util.InputUtil.*;
import static io.github.fishstiz.packed_packs.util.PackUtil.*;
import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.*;

public class PackedPacksScreen extends PackListEventHandler implements
        ToggleableDialogContainer,
        ContextMenuContainer,
        Restorable<PackedPacksScreen.Snapshot> {
    private static final class_2561 ACTION_BAR_INFO = ResourceUtil.getText("toggle_actionbar.info");
    private static final class_2561 ORIGINAL_SCREEN_INFO = ResourceUtil.getText("original_screen.info");
    private static final class_2561 OPTIONS_TEXT = ResourceUtil.getText("options.title");
    private static final class_2561 OPEN_FOLDER_TEXT = class_2561.method_43471("pack.openFolder");
    private static final class_2561 OPEN_FOLDER_INFO_TEXT = class_2561.method_43471("pack.folderInfo");
    private static final class_2561 APPLY_TEXT = ResourceUtil.getText("apply");
    private static final class_2561 REFRESH_PACKS_TEXT = ResourceUtil.getText("refresh");
    private static final class_2561 RESET_ENABLED_TEXT = ResourceUtil.getText("reset_enabled");
    private final class_437 previous;
    private final PackSelectionScreenArgs original;
    private final PackRepositoryHelper repository;
    private final class_8132 layout = new class_8132(this);
    private final HistoryManager<io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot> history = new HistoryManager<>();
    private final ImmediateDebouncer<String> searchListener = new ImmediateDebouncer<>(this::clearHistory, 250);
    private final AvailablePacksLayout availablePacks;
    private final CurrentPacksLayout currentPacks;
    private final Config.Packs packsConfig;
    private final ProfilesLayout profiles;
    private final FolderDialog folderDialog;
    private final Modal<class_8667> options;
    private final FileRenameModal fileRenameModal;
    private final ContextMenu contextMenu;
    private final List<ToggleableDialog<?>> dialogs;
    private final List<PackList> packLists;
    private List<Path> additionalFolders;
    private CompletableFuture<Void> refreshFuture;
    private PackWatcher watcher;
    private boolean showActionBar = PackedPacks.CONFIG.isShowActionBar();
    private boolean initialized = false;

    private PackedPacksScreen(class_437 previous, PackSelectionScreenArgs original, @Nullable Profile profile, @Nullable PackGroup packs) {
        super(ResourceUtil.getModName());

        this.previous = previous;
        this.original = original;
        this.packsConfig = PackedPacks.CONFIG.get(original.packType());
        this.profiles = new ProfilesLayout(profile, this.packsConfig, this);
        this.repository = new PackRepositoryHelper(this.original.repository(), this.original.packDir(), this.packsConfig, this.profiles::getProfile);
        this.availablePacks = new AvailablePacksLayout(this.repository, this);
        this.currentPacks = new CurrentPacksLayout(this.repository, this);
        this.options = Modal.builder(this, new OptionsLayout().layout())
                .setBackdrop(new ColoredRect(Theme.BLACK.withAlpha(0.5f)))
                .setCaptureFocus(true)
                .build();
        this.folderDialog = FolderDialog.create(this, this.repository);
        this.fileRenameModal = new FileRenameModal(this, this.repository);
        this.contextMenu = ContextMenu.builder(this)
                .setSpacing(SPACING)
                .setBackground(Theme.GRAY_800.getARGB())
                .setBorderColor(Theme.GRAY_500.getARGB())
                .build();
        this.dialogs = List.of(this.options, this.contextMenu, this.fileRenameModal, this.profiles.getSidebar(), this.folderDialog);
        this.packLists = List.of(this.folderDialog.root(), this.availablePacks.getList(), this.currentPacks.getList());
        this.initAdditionalFolders();

        if (profile != null) {
            this.applyProfile(profile);
        } else if (packs == null) {
            this.useSelected();
        } else {
            this.applyPacks(packs.unselected(), packs.selected());
        }
    }

    public PackedPacksScreen(class_437 previous, PackSelectionScreenArgs original) {
        this(previous, original, null, null);
    }

    public PackedPacksScreen(class_437 previous, PackSelectionScreenArgs original, Profile profile) {
        this(previous, original, profile, null);
    }

    public PackedPacksScreen(class_437 previous, PackSelectionScreenArgs original, PackGroup packs) {
        this(previous, original, null, packs);
    }

    @Override
    public void method_49589() {
        if (this.initialized) {
            this.refreshPacks();
            this.initAdditionalFolders();
            this.createWatcher();
        }
    }

    @Override
    public void method_25432() {
        this.closeWatcher();
        this.syncProfile(this.profiles.getProfile());
        PackedPacks.CONFIG.save();
        Preferences.INSTANCE.save();
    }

    @Override
    protected void method_25426() {
        if (this.initialized) return;

        this.layout.method_48992(this.createHeader());
        this.layout.method_48999(this.createContents());
        this.layout.method_48996(this.createFooter());

        this.folderDialog.root().method_48206(this.folderDialog::addRenderableWidget);
        this.profiles.initContents();
        this.profiles.getSidebar().getCloseButton().addListener(this::method_56131);
        this.options.root().method_48206(this.options::addRenderableWidget);

        this.method_25429(this.options);
        this.method_25429(this.contextMenu);
        this.method_25429(this.fileRenameModal);
        this.method_25429(this.profiles.getSidebar());
        this.method_25429(this.folderDialog);
        this.layout.method_48206(this::method_37063);
        this.method_37060(this.folderDialog);
        this.method_37060(this.profiles.getSidebar());
        this.method_37060(this.fileRenameModal);
        this.method_37060(this.contextMenu);
        this.method_37060(this.options);

        this.clearHistory();
        this.method_48640();

        this.refreshPacks();
        this.createWatcher();

        this.initialized = true;
    }

    private FlexLayout createHeader() {
        FlexLayout header = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        final boolean devMode = PackedPacks.CONFIG.isDevMode();

        header.addChild(
                FidgetzButton.builder()
                        .makeSquare()
                        .setMessage(ProfilesLayout.TITLE_TEXT)
                        .setTooltip(class_7919.method_47407(ProfilesLayout.TITLE_TEXT))
                        .setSprite(HAMBURGER_SPRITE)
                        .setOnPress(this.profiles.getSidebar()::toggle)
                        .build()
        );

        if (devMode || Preferences.INSTANCE.actionBarWidget.get()) {
            header.addChild(
                    Toggleable.applyPref(Preferences.INSTANCE.actionBarWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setTooltip(class_7919.method_47407(ACTION_BAR_INFO))
                            .setSprite(new Sprite(ResourceUtil.getIcon("filter"), Size.of16()))
                            .setOnPress(this::toggleActionBar)
                            .build()
            );
        }
        header.addChild(this.profiles.getToggleNameButton());
        header.addFlexChild(this.profiles.getNameField());

        class_5375 packSelectionScreen = this.previous instanceof class_5375 s ? s : this.original.createDummy();
        ModAdditions.addToHeader(this.repository.isResourcePacks(), header, packSelectionScreen);

        if (devMode || Preferences.INSTANCE.optionsWidget.get()) {
            header.addChild(
                    Toggleable.applyPref(Preferences.INSTANCE.optionsWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setMessage(OPTIONS_TEXT)
                            .setTooltip(class_7919.method_47407(OPTIONS_TEXT))
                            .setSprite(new Sprite(ResourceUtil.getIcon("gear"), Size.of16()))
                            .setOnPress(this.options::toggle)
                            .build()
            );
        }
        if (devMode || Preferences.INSTANCE.originalScreenWidget.get()) {
            header.addChild(
                    Toggleable.applyPref(Preferences.INSTANCE.originalScreenWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setTooltip(class_7919.method_47407(ORIGINAL_SCREEN_INFO))
                            .setSprite(new Sprite(ResourceUtil.getIcon("exit"), Size.of16()))
                            .setOnPress(this::setOriginalScreen)
                            .build()
            );
        }
        return header;
    }

    private FlexLayout createContents() {
        FlexLayout contents = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        this.availablePacks.init(contents.addFlexChild(FlexLayout.vertical(this.layout::method_57727).spacing(SPACING), false));
        this.currentPacks.init(contents.addFlexChild(FlexLayout.vertical(this.layout::method_57727).spacing(SPACING), false));
        this.currentPacks.getSearchField().addListener(this.searchListener);
        this.availablePacks.getSearchField().addListener(this.searchListener);
        return contents;
    }

    private FlexLayout createFooter() {
        FlexLayout footer = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        FlexLayout firstColumn = FlexLayout.horizontal().spacing(SPACING);
        FlexLayout secondColumn = firstColumn.copyLayout();

        firstColumn.addFlexChild(
                FidgetzButton.builder()
                        .setMessage(OPEN_FOLDER_TEXT)
                        .setTooltip(class_7919.method_47407(OPEN_FOLDER_INFO_TEXT))
                        .setOnPress(this.repository::openDir)
                        .build()
        );

        if (this.repository.isResourcePacks()) {
            secondColumn.addFlexChild(FidgetzButton.builder().setMessage(APPLY_TEXT).setOnPress(this::commit).build());
        }

        secondColumn.addFlexChild(FidgetzButton.builder().setMessage(class_5244.field_24334).setOnPress(this::method_25419).build());

        footer.addFlexChild(firstColumn);
        footer.addFlexChild(secondColumn);
        return footer;
    }

    public int getMaxWidth() {
        return this.field_22789 - SPACING * 2;
    }

    @Override
    protected void method_41843() {
        if (this.field_22787 == null) return;

        PackedPacksScreen screen;
        Profile profile = this.profiles.getProfile();

        if (profile != null) {
            profile.setPacks(this.currentPacks.getList().copyFlattenedPacks());
            screen = new PackedPacksScreen(this.previous, this.original, profile);
        } else {
            PackGroup packs = PackGroup.of(this.currentPacks.getList().copyPacks(), this.availablePacks.getList().copyPacks());
            screen = new PackedPacksScreen(this.previous, this.original, packs);
        }

        this.field_22787.method_1507(screen);
    }

    @Override
    public void method_29638(List<Path> packs) {
        if (this.field_22787 != null) {
            this.field_22787.method_1507(new class_410(
                    this.confirmFileDrop(packs),
                    class_2561.method_43471("pack.dropConfirm"),
                    class_2561.method_43470(joinPackNames(packs))
            ));
        }
    }

    private BooleanConsumer confirmFileDrop(List<Path> packs) {
        return confirmed -> {
            if (this.field_22787 == null) {
                return;
            }
            if (!confirmed) {
                this.field_22787.method_1507(this);
                return;
            }
            PathValidationResults results = validatePaths(packs);

            if (!results.symlinkWarnings().isEmpty()) {
                this.field_22787.method_1507(class_8669.method_52750(() -> this.field_22787.method_1507(this)));
                return;
            }
            if (!results.valid().isEmpty()) {
                class_5375.method_29669(this.field_22787, results.valid(), this.original.packDir());
                this.refreshPacks();
            }
            if (!results.rejected().isEmpty()) {
                String rejectedNames = joinPackNames(results.rejected());
                this.field_22787.method_1507(new class_403(
                        () -> this.field_22787.method_1507(this),
                        class_2561.method_43471("pack.dropRejected.title"),
                        class_2561.method_43469("pack.dropRejected.message", rejectedNames)
                ));
                return;
            }
            this.field_22787.method_1507(this);
        };
    }

    private void setOriginalScreen() {
        if (this.previous instanceof class_5375) {
            this.method_25419();
        } else if (this.field_22787 != null) {
            class_5375 originalScreen = this.original.createScreen();
            ((PackSelectionScreenAccessor) originalScreen).packed_packs$setPrevious(this.previous);
            this.field_22787.method_1507(originalScreen);
        }
    }

    @Override
    public void method_25419() {
        if (this.field_22787 == null) return;

        String commitRequestor = ModAdditions.shouldCommit(this.repository.isResourcePacks());
        if (commitRequestor != null) {
            this.commit();
            PackedPacks.LOGGER.info("[packed_packs] Commiting packs on close at the request of mod '{}'.", commitRequestor);
        } else if (!(this.packsConfig instanceof Config.ResourcePacks resourceConfig) || resourceConfig.isApplyOnClose()) {
            this.commit();
        }

        if (!this.repository.isResourcePacks() && !(this.previous instanceof class_5375)) {
            this.original.output().accept(this.repository.getRepository()); // validate datapacks
            return;
        }

        if (this.previous instanceof PackSelectionScreenAccessor packScreen) {
            ((PackSelectionModelAccessor) packScreen.getModel()).packed_packs$reset();
            packScreen.invokeReload();
        }

        this.field_22787.method_1507(this.previous);
    }

    @Override
    public void method_25393() {
        if (this.watcher != null) {
            this.watcher.poll();
        }
    }

    private void createWatcher() {
        if (this.watcher == null) {
            try {
                List<Path> paths = CollectionsUtil.mutableListOf(this.repository.getBaseDir());
                paths.addAll(this.additionalFolders);
                this.watcher = new PackWatcher(paths, this::refreshPacks);
            } catch (Exception e) {
                PackedPacks.LOGGER.error("[packed_packs] Failed to initialize pack directory watcher.", e);
                this.closeWatcher();
            }
        }
    }

    private void closeWatcher() {
        if (this.watcher != null) {
            this.watcher.close();
            this.watcher = null;
        }
    }

    private void initAdditionalFolders() {
        this.additionalFolders = CollectionsUtil.deduplicate(CollectionsUtil.addAll(
                mapValidDirectories(this.packsConfig.getAdditionalFolders()),
                this.repository.getAdditionalDirs()
        ));
    }

    private void repositionLists() {
        this.availablePacks.setHeaderVisibility(this.showActionBar);
        this.currentPacks.setHeaderVisibility(this.showActionBar);
    }

    @Override
    protected void method_48640() {
        this.layout.method_48222();
        ((HeaderAndFooterLayoutAccess) this.layout).getContentsFrame().method_46419(this.layout.method_48998());
        this.profiles.getSidebar().repositionElements();
        this.options.repositionElements();
        this.fileRenameModal.repositionElements();
        this.contextMenu.setOpen(false);
        this.repositionLists();
    }

    public void toggleActionBar() {
        this.showActionBar = !this.showActionBar;
        PackedPacks.CONFIG.setShowActionBar(this.showActionBar);
        this.repositionLists();
    }

    public void commit() {
        this.currentPacks.getSearchField().method_1852("");
        this.syncProfile(this.profiles.getProfile());
        this.repository.selectPacks(this.currentPacks.getList().copyPacks());

        if (this.repository.isResourcePacks()) {
            this.original.output().accept(this.repository.getRepository());
        }
    }

    private void replacePacks(PackList list, ImmutableList<class_3288> packs) {
        list.replaceState(new PackList.Snapshot(list, packs, list.copySelection(), list.copyQuery()));
    }

    private void revalidateFolder() {
        if (this.folderDialog.isOpen()) {
            FolderPack folderPack = this.folderDialog.getFolderPack();
            if (folderPack == null || this.repository.getFolderConfig(folderPack) == null) {
                this.folderDialog.setOpen(false);
            } else {
                this.replacePacks(this.folderDialog.root(), ImmutableList.copyOf(this.repository.getNestedPacks(folderPack)));
            }
        }
    }

    public void revalidatePacks() {
        PackList availableList = this.availablePacks.getList();
        PackList currentList = this.currentPacks.getList();
        PackGroup packs = this.repository.validatePacks(availableList.copyPacks(), currentList.copyPacks());
        this.repository.clearIconCache();
        this.replacePacks(availableList, packs.unselected());
        this.replacePacks(currentList, packs.selected());
        this.revalidateFolder();
        this.clearHistory();
    }

    public void refreshPacks() {
        this.refreshFuture = CompletableFuture.runAsync(this.repository::refresh, class_156.method_18349())
                .thenRunAsync(this::revalidatePacks, this.field_22787);
    }

    public void reset() {
        PackGroup packs = this.repository.getPacksByRequirement();
        this.availablePacks.getList().reload(packs.unselected());
        this.currentPacks.getList().reload(packs.selected());
        this.clearHistory();
    }

    public void useSelected() {
        PackGroup packs = this.repository.getPacksBySelected();
        this.availablePacks.getList().reload(packs.unselected());
        this.currentPacks.getList().reload(packs.selected());
        this.clearHistory();
    }

    public void resetToEnabled() {
        this.onEvent(new BasicEvent(true));
        this.useSelected();
    }

    public void onProfileChange(@Nullable Profile previous, @Nullable Profile current) {
        if (previous != null) {
            previous.setPacks(this.currentPacks.getList().copyFlattenedPacks());
        }

        if (current == null) {
            this.useSelected();
        } else if (!current.getPackIds().isEmpty()) {
            this.applyProfile(current);
        } else {
            this.reset();
        }

        boolean unlocked = current == null || !current.isLocked();
        this.availablePacks.getSearchField().method_1852("");
        this.availablePacks.getTransferButton().field_22763 = unlocked;
        this.currentPacks.getSearchField().method_1852("");
        this.currentPacks.getTransferButton().field_22763 = unlocked;

        this.method_48640();
    }

    public void onProfileCopy(@Nullable Profile original, @NotNull Profile copy) {
        copy.setPacks(this.currentPacks.getList().copyFlattenedPacks());
    }

    private void applyProfile(@NotNull Profile profile) {
        List<class_3288> available = this.availablePacks.getList().copyPacks();
        List<class_3288> current = this.repository.getPacksByFlattenedIds(profile.getPackIds());
        this.applyPacks(available, current);
    }

    private void applyPacks(List<class_3288> available, List<class_3288> current) {
        PackGroup packs = this.repository.validatePacks(available, current);
        this.availablePacks.getList().reload(packs.unselected());
        this.currentPacks.getList().reload(packs.selected());
        this.clearHistory();
    }

    public void syncProfile(@Nullable Profile profile) {
        if (profile != null) {
            profile.syncPacks(this.repository.getFlattenedPacks(), this.currentPacks.getList().copyFlattenedPacks());
        }
    }

    private boolean isUnlocked() {
        Profile profile = this.profiles.getProfile();
        return profile == null || !profile.isLocked();
    }

    @Override
    public @NotNull List<PackList> getPackLists() {
        return this.packLists;
    }

    @Override
    public @Nullable PackList getDestination(PackList source) {
        if (source == this.availablePacks.getList()) {
            return this.currentPacks.getList();
        } else if (source == this.currentPacks.getList()) {
            return this.availablePacks.getList();
        }
        return null;
    }

    @Override
    protected void transferFocus(PackList source, PackList destination) {
        super.transferFocus(source, destination);

        if (destination == currentPacks.getList()) {
            currentPacks.getList().scrollToLastSelected();
        }
    }

    private void onFolderOpen(FolderOpenEvent event) {
        this.folderDialog.root().reload(this.repository.getNestedPacks(event.opened()));
        this.folderDialog.updateFolder(event.target(), event.opened(), this.repository);
        this.folderDialog.setOpen(true);
    }

    private void onFolderClose(FolderCloseEvent event) {
        this.folderDialog.setOpen(false);

        FolderPack folderPack = event.folderPack();
        if (folderPack == null) return;

        Folder folder = this.repository.getFolderConfig(folderPack);
        if (folder != null && this.isUnlocked()) {
            if (folder.trySetPacks(this.repository.validateAndOrderNestedPacks(folderPack, event.target().copyPacks()))) {
                folderPack.saveConfig(folder);
            }
            this.focusList(ObjectsUtil.firstNonNullOrDefault(this.availablePacks.getList(), this.folderDialog.getParent()));
        }
    }

    private void onFileRename(FileRenameEvent event) {
        if (this.folderDialog.isOpen()) {
            this.folderDialog.onRename(event.renamed(), event.newName());
        }
        this.refreshPacks();
    }

    @Override
    protected void handleMoveEvent(MoveEvent event) {
        if (event.target() != this.folderDialog.root()) {
            super.handleMoveEvent(event);
            return;
        }

        PackList.Entry entry = event.target().getEntry(event.trigger());
        if (entry != null) {
            this.focus(class_8016.method_48194(entry, event.target(), this.folderDialog, this));
        } else {
            this.focus(class_8016.method_48194(event.target(), this.folderDialog, this));
        }
    }

    @Override
    public void onEvent(PackListEvent event) {
        super.onEvent(event);

        this.profiles.getSidebar().setOpen(false);
        this.contextMenu.setOpen(false);
        this.fileRenameModal.setOpen(false);

        boolean notFolderDialogEvent = event.target() != this.folderDialog.root();
        if (notFolderDialogEvent) {
            this.folderDialog.setOpen(false);
        }

        switch (event) {
            case FileDeleteEvent ignore -> this.revalidatePacks();
            case FileRenameOpenEvent e -> this.fileRenameModal.open(e.target(), e.trigger());
            case FileRenameEvent e -> this.onFileRename(e);
            case FileRenameCloseEvent e -> this.focusList(e.target());
            case FolderOpenEvent e -> this.onFolderOpen(e);
            case FolderCloseEvent e -> this.onFolderClose(e);
            default -> {
            }
        }

        if (event.pushToHistory() && notFolderDialogEvent) {
            this.history.push(this.captureState());
        }
    }

    public @Nullable PackLayout<?> getLayoutFromSelectedList() {
        return ObjectsUtil.firstNonNull(
                ObjectsUtil.<PackLayout<?>>pick(this.availablePacks, this.currentPacks, pl -> pl.getList() == this.method_25399()),
                ObjectsUtil.<PackLayout<?>>pick(this.availablePacks, this.currentPacks, pl -> pl.getList().method_49606()),
                ObjectsUtil.<PackLayout<?>>pick(this.availablePacks, this.currentPacks, pl -> pl.getList().method_25370())
        );
    }

    public ToggleableEditBox<Void> focusSearchField(@NotNull PackLayout<?> packLayout) {
        if (!this.showActionBar) this.toggleActionBar();
        ToggleableEditBox<Void> searchField = packLayout.getSearchField();
        this.focus(searchField);
        return searchField;
    }

    @Override
    public boolean method_25400(class_11905 charEvent) {
        if (super.method_25400(charEvent)) {
            return true;
        }
        if (charEvent.comp_4793() != field_31947 && noModifiers(charEvent.comp_4794())) {
            PackLayout<?> packLayout = this.getLayoutFromSelectedList();
            if (packLayout != null && !packLayout.getSearchField().method_25370()) {
                return this.focusSearchField(packLayout).method_25400(charEvent);
            }
        }
        return false;
    }

    public void toggleDevMode() {
        PackedPacks.CONFIG.setDevMode(!PackedPacks.CONFIG.isDevMode());
        ToastUtil.onDevModeToggleToast(PackedPacks.CONFIG.isDevMode());
        this.method_41843();
    }

    @Override
    public boolean method_25404(class_11908 keyEvent) {
        this.contextMenu.setOpen(false);

        if (isDeveloperMode(keyEvent)) {
            this.toggleDevMode();
            return true;
        }
        if (isRefresh(keyEvent) && (this.refreshFuture == null || this.refreshFuture.isDone())) {
            this.refreshPacks();
            return true;
        }
        if (super.method_25404(keyEvent)) {
            return true;
        }
        if (isRedo(keyEvent)) {
            return this.history.redo();
        }
        if (isUndo(keyEvent)) {
            return this.history.undo();
        }
        if (keyEvent.comp_4795() == field_31986) {
            PackLayout<?> packLayout = this.getLayoutFromSelectedList();
            if (packLayout != null) {
                ToggleableEditBox<Void> searchField = packLayout.getSearchField();
                if (!searchField.method_25370() && !searchField.method_1882().isEmpty()) {
                    return this.focusSearchField(packLayout).method_25404(keyEvent);
                }
            }
        }
        return false;
    }

    private boolean hasHeader(List<MenuItem> items) {
        return !items.isEmpty() && items.getFirst() instanceof PackMenuHeader;
    }

    private void openContextMenu(int mouseX, int mouseY) {
        if (this.contextMenu.method_25405(mouseX, mouseY)) return;

        this.buildItems(mouseX, mouseY)
                .when(PackedPacks.CONFIG.isDevMode())
                .ifTrue(dev -> dev.separatorIfNonEmpty()
                        .whenNonNull(this.profiles.getProfile())
                        .ifTrue((profile, b) -> b.
                                add(devItem(ResourceUtil.getText("profile.save"))
                                        .action(() -> profile.setPacks(this.currentPacks.getList().copyFlattenedPacks()))
                                        .build())
                                .separator())
                        .add(devItem(ResourceUtil.getText("preferences"))
                                .addChildren(Toggleable.preferences())
                                .addChild(devItem(ResourceUtil.getText("preferences.reset"))
                                        .action(Preferences.INSTANCE::reset)
                                        .build())
                                .build())
                )
                .separatorIfNonEmpty()
                .simpleItem(RESET_ENABLED_TEXT, this::isUnlocked, this::resetToEnabled)
                .simpleItem(REFRESH_PACKS_TEXT, this::canRefresh, this::refreshPacks)
                .when(this.additionalFolders, List::isEmpty)
                .ifTrue(b -> b.simpleItem(OPEN_FOLDER_TEXT, this.repository::openDir))
                .ifFalse((dirs, b) -> b
                        .parent(OPEN_FOLDER_TEXT, p -> p
                                .add(new DirectoryMenuItem(this.repository.getBaseDir()))
                                .separator()
                                .addAll(dirs.stream().map(DirectoryMenuItem::new).toList())))
                .peek(items -> {
                    int yOffset = this.hasHeader(items) ? this.contextMenu.getItemHeight() : 0;
                    this.contextMenu.open(mouseX, mouseY - yOffset, items);
                });
    }

    @Override
    public boolean method_25402(class_11909 mouseEvent, boolean doubleClicked) {
        this.setDragged(null);
        if (isRightClick(mouseEvent) && !this.options.method_25405(mouseEvent.comp_4798(), mouseEvent.comp_4799())) {
            this.openContextMenu((int) mouseEvent.comp_4798(), (int) mouseEvent.comp_4799());
            return true;
        }
        if (ToggleableDialogContainer.super.method_25402(mouseEvent, doubleClicked)) {
            return true;
        }
        if (isClickForward(mouseEvent)) {
            return this.history.redo();
        }
        if (isClickBack(mouseEvent)) {
            return this.history.undo();
        }
        if (isLeftClick(mouseEvent) && !(this.method_25399() instanceof PackList)) {
            this.method_25395(this.method_25396().getFirst());
            this.layout.method_48206(w -> w.method_25365(false));
        }
        this.contextMenu.setOpen(false);
        return false;
    }


    @Override
    public List<ToggleableDialog<?>> getDialogs() {
        return this.dialogs;
    }

    @Override
    public void method_25394(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
        super.method_25394(guiGraphics, mouseX, mouseY, partialTick);

        if (PackedPacks.CONFIG.isDevMode()) {
            float scale = 0.5f;
            int y = (int) ((field_22790 - this.field_22793.field_2000 * scale) / scale);

            guiGraphics.method_51448().pushMatrix();
            guiGraphics.method_51448().scale(scale);
            guiGraphics.method_27535(this.field_22793, ResourceUtil.getText("dev_mode", DEV_MODE_SHORTCUT), 0, y, Theme.WHITE.getARGB());
            guiGraphics.method_51448().popMatrix();
        }
    }

    public boolean canRefresh() {
        return this.refreshFuture == null || this.refreshFuture.isDone();
    }

    public void clearHistory() {
        this.history.reset(this.captureState());
    }

    @Override
    public @NotNull PackedPacksScreen.Snapshot captureState() {
        return new io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot(this, this.availablePacks.getList().captureState(), this.currentPacks.getList().captureState());
    }

    @Override
    public void replaceState(@NotNull io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot snapshot) {
        List<class_3288> validPacks = this.repository.getPacks();
        this.availablePacks.getSortButton().setValueSilently(snapshot.availablePacks.query().getSort());
        this.availablePacks.getCompatButton().setValueSilently(snapshot.availablePacks.query().isHideIncompatible());
        snapshot.availablePacks.validate(validPacks).restore();
        snapshot.currentPacks.validate(validPacks).restore();
        this.availablePacks.getList().scrollToLastSelected();
        this.currentPacks.getList().scrollToLastSelected();
    }

    public record Snapshot(
            PackedPacksScreen target,
            PackList.Snapshot availablePacks,
            PackList.Snapshot currentPacks
    ) implements Restorable.Snapshot<io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot> {
    }
}
