package io.github.fishstiz.packed_packs.pack;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.config.Config;
import io.github.fishstiz.packed_packs.config.Folder;
import io.github.fishstiz.packed_packs.config.Profile;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.transform.interfaces.FilteredPackSelectionModel;
import io.github.fishstiz.packed_packs.transform.interfaces.IPack;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionModelAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.FolderRepositorySourceAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.PackRepositoryAccessor;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.lang.CollectionsUtil;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.apache.commons.lang3.function.Consumers;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.*;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3283;
import net.minecraft.class_3288;
import net.minecraft.class_5369;
import net.minecraft.class_7172;
import net.minecraft.class_9225;

public class PackRepositoryHelper implements PackAssets {
    private final Map<String, class_2960> cachedIcons = new Object2ObjectOpenHashMap<>();
    private final Map<String, class_3288> availablePacks = new Object2ObjectLinkedOpenHashMap<>();
    private final Map<String, List<class_3288>> folderPacks = new Object2ObjectOpenHashMap<>();
    private final Map<String, CompletableFuture<Folder>> folderConfigs = new Object2ObjectOpenHashMap<>();
    private final PackOptionsResolver resolver;
    private final class_3283 repository;
    private final Path packDir;
    private final boolean resourcePacks;
    private class_5369 model;
    private Map<String, class_2960> staleIcons;

    public PackRepositoryHelper(class_3283 repository, Path packDir, Config.Packs config, Supplier<@Nullable Profile> profileSupplier) {
        this.resolver = new PackOptionsResolver(profileSupplier, config);
        this.repository = repository;
        this.packDir = packDir;
        this.resourcePacks = config instanceof Config.ResourcePacks;

        this.refreshModel();
        this.regenerateAvailablePacks();
    }

    public class_3283 getRepository() {
        return this.repository;
    }

    private List<class_3288> getSelectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getSelectedPacks();
    }

    private List<class_3288> getUnselectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getUnselectedPacks();
    }

    private void refreshModel() {
        this.model = new class_5369(Consumers.nop(), PackAssets::getDefaultIcon, this.repository, Consumers.nop());
        ((FilteredPackSelectionModel) this.model).packed_packs$filterHidden(false);
    }

    public List<class_3288> getPacks() {
        return List.copyOf(this.availablePacks.values());
    }

    public List<class_3288> getFlattenedPacks() {
        return this.flattenPacks(List.copyOf(this.availablePacks.values()));
    }

    public PackGroup getPacksByRequirement() {
        List<class_3288> required = new ObjectArrayList<>();
        List<class_3288> optional = new ObjectArrayList<>();

        for (class_3288 pack : this.availablePacks.values()) {
            if (this.isRequired(pack)) {
                this.getPosition(pack).method_14468(required, pack, this::getSelectionConfig, true);
            } else {
                optional.add(pack);
            }
        }

        return PackGroup.of(required, optional);
    }

    public PackGroup getPacksBySelected() {
        return this.validateAndGroupPacks(this.getUnselectedPacks(), this.getSelectedPacks());
    }

    private void removePack(class_3288 pack) {
        this.repository.method_49428(pack.method_14463());
        this.availablePacks.remove(pack.method_14463());
        try {
            this.getSelectedPacks().remove(pack);
            this.getUnselectedPacks().remove(pack);
        } catch (UnsupportedOperationException e) {
            // just in case
            PackedPacks.LOGGER.warn("[packed_packs] Failed to mutate PackSelectionModel lists. Report this issue to mod author.");
        }
    }

    @Override
    public boolean deletePack(class_3288 pack) {
        if (!PackAssets.super.deletePack(pack)) {
            return false;
        }
        if (pack instanceof FolderPack) {
            Optional.ofNullable(this.folderPacks.get(pack.method_14463())).ifPresent(packs -> packs.forEach(this::removePack));
            this.folderPacks.remove(pack.method_14463());
            this.folderConfigs.remove(pack.method_14463());
        }
        this.removePack(pack);
        this.regenerateAvailablePacks();
        return true;
    }

    /**
     * @param unselected ungrouped list of unselected packs
     * @param selected   ungrouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validateAndGroupPacks(List<class_3288> unselected, List<class_3288> selected) {
        return this.validatePacks(this.groupByFolders(unselected), this.groupByFolders(selected));
    }

    public void validateOverrides(class_3288 pack) {
        Profile profile = this.resolver.profileSupplier().get();
        if (profile == null) return;

        Profile defaultProfile = this.getConfig().getDefaultProfile();
        if (defaultProfile != null && defaultProfile.overridesRequired(pack)) {
            return;
        }

        if (profile.overridesRequired(pack) && !profile.isRequired(pack)) {
            profile.setRequired(null, pack);
        }
    }

    private void addValidPacks(List<class_3288> source, Set<class_3288> seen, ObjectOpenHashSet<class_3288> validPacks, List<class_3288> target) {
        for (class_3288 pack : source) {
            this.validateOverrides(pack);
            class_3288 validPack = validPacks.get(pack); // metadata can change
            if (validPack != null && (seen.add(pack))) {
                target.add(validPack);
            }
        }
    }

    private void addValidPacks(
            List<class_3288> source,
            Set<class_3288> seen,
            ObjectOpenHashSet<class_3288> validPacks,
            List<class_3288> targetUnselected,
            List<class_3288> targetSelected
    ) {
        for (class_3288 pack : source) {
            this.validateOverrides(pack);
            class_3288 validPack = validPacks.get(pack);
            if (validPack != null && seen.add(pack)) {
                if (this.isRequired(validPack)) {
                    this.getPosition(validPack).method_14468(targetSelected, validPack, this::getSelectionConfig, true);
                } else {
                    targetUnselected.add(validPack);
                }
            }
        }
    }

    /**
     * @param unselected grouped list of unselected packs
     * @param selected   grouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validatePacks(List<class_3288> unselected, List<class_3288> selected) {
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(this.availablePacks.values());
        List<class_3288> validSelected = new ObjectArrayList<>(selected.size());
        List<class_3288> validUnselected = new ObjectArrayList<>(unselected.size());

        this.addValidPacks(selected, seen, validPacks, validSelected);
        this.addValidPacks(unselected, seen, validPacks, validUnselected, validSelected);

        for (class_3288 validPack : validPacks) {
            if (seen.add(validPack)) {
                if (this.isRequired(validPack)) {
                    this.getPosition(validPack).method_14468(validSelected, validPack, this::getSelectionConfig, true);
                } else {
                    validUnselected.add(validPack);
                }
            }
        }
        return PackGroup.of(validSelected, validUnselected);
    }

    /**
     * @param folderPack   the folder pack
     * @param orderedPacks the nested packs that define the preferred order
     * @return a validated and ordered list of all packs under the folder pack
     */
    public List<class_3288> validateAndOrderNestedPacks(FolderPack folderPack, List<class_3288> orderedPacks) {
        List<class_3288> orderedValidPacks = this.folderPacks.get(folderPack.method_14463());
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(orderedValidPacks);
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        List<class_3288> finalOrderedPacks = new ObjectArrayList<>();

        this.addValidPacks(orderedPacks, seen, validPacks, finalOrderedPacks);

        for (class_3288 validPack : orderedValidPacks) {
            if (seen.add(validPack)) {
                finalOrderedPacks.add(validPack);
            }
        }

        this.folderPacks.put(folderPack.method_14463(), finalOrderedPacks);
        return finalOrderedPacks;
    }

    /**
     * @param folderPack   the folder pack
     * @param orderedPacks the nested pack ids that define the preferred order
     * @return a validated and ordered list of all packs under the folder pack
     */
    public List<class_3288> validateAndOrderNestedPackIds(FolderPack folderPack, List<String> orderedPacks) {
        return this.validateAndOrderNestedPacks(folderPack, this.getPacksById(orderedPacks, this.folderPacks.get(folderPack.method_14463())));
    }

    /**
     * @param packIds grouped pack ids
     * @param source  grouped packs
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds, Map<String, class_3288> source) {
        return CollectionsUtil.lookup(packIds, source);
    }

    /**
     * @param packIds grouped pack ids
     * @param source  grouped packs
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds, Collection<class_3288> source) {
        return CollectionsUtil.lookup(packIds, CollectionsUtil.toMap(source, class_3288::method_14463));
    }

    /**
     * @param packIds grouped pack ids
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds) {
        return this.getPacksById(packIds, this.availablePacks);
    }

    public List<class_3288> getPacksByFlattenedIds(Collection<String> packIds) {
        List<class_3288> folderPacks = CollectionsUtil.filter(this.availablePacks.values(), FolderPack.class::isInstance, ObjectArrayList::new);
        List<class_3288> available = CollectionsUtil.addAll(folderPacks, this.repository.method_14441());
        return this.groupByFolders(this.getPacksById(packIds, available));
    }

    /**
     * @param packs ungrouped collection of packs
     */
    private void populateAvailablePacks(Collection<class_3288> packs) {
        for (class_3288 pack : packs) {
            IPack _pack = (IPack) pack;
            if (_pack.packed_packs$nestedPack()) {
                Path folderPath = Objects.requireNonNull(_pack.packed_packs$getPath()).getParent();
                String folderName = PackUtil.generatePackName(folderPath);
                String folderId = PackUtil.generatePackId(folderName);
                if (!this.availablePacks.containsKey(folderId)) {
                    FolderPack folderPack = new FolderPack(folderId, folderName, folderPath);
                    this.folderConfigs.put(folderId, folderPack.loadConfig());
                    this.availablePacks.put(folderId, folderPack);
                }
                this.folderPacks.computeIfAbsent(folderId, id -> new ObjectArrayList<>()).add(pack);
            } else {
                this.availablePacks.put(pack.method_14463(), pack);
            }
        }
    }

    private void regenerateAvailablePacks() {
        this.availablePacks.clear();
        this.folderPacks.clear();
        this.folderConfigs.clear();
        this.populateAvailablePacks(this.getSelectedPacks());
        this.populateAvailablePacks(this.getUnselectedPacks());
    }

    public void refresh() {
        ((PackSelectionModelAccessor) this.model).packed_packs$reset();
        this.model.method_29981();
        this.regenerateAvailablePacks();
    }

    /**
     * @param selected grouped list of selected packs
     */
    public void selectPacks(List<class_3288> selected) {
        boolean hasHighContrast = false;
        List<String> packIds = new ObjectArrayList<>();

        for (class_3288 pack : this.flattenPacks(selected).reversed()) {
            String packId = pack.method_14463();

            packIds.add(packId);
            if (!hasHighContrast && packId.equals(PackUtil.HIGH_CONTRAST_ID)) {
                hasHighContrast = true;
            }
        }

        this.repository.method_14447(ImmutableList.copyOf(packIds));

        class_7172<Boolean> highContrastOption = class_310.method_1551().field_1690.method_49600();
        if (highContrastOption.method_41753() != hasHighContrast) {
            highContrastOption.method_41748(hasHighContrast);
        }

        this.refreshModel();
    }

    /**
     * @param groupedPacks grouped list of packs
     * @return flattened list of packs
     */
    @Override
    public List<class_3288> flattenPacks(List<class_3288> groupedPacks) {
        if (this.folderPacks.isEmpty()) return groupedPacks;

        List<class_3288> flattened = new ObjectArrayList<>(groupedPacks);
        for (int i = flattened.size() - 1; i >= 0; i--) {
            if (flattened.get(i) instanceof FolderPack folderPack) {
                // do not remove folder pack; solves headaches
//                flattened.remove(i);
                List<class_3288> nested = this.getNestedPacks(folderPack);
                if (nested != null && !nested.isEmpty()) {
                    for (class_3288 pack : Lists.reverse(nested)) {
                        flattened.add(i + 1, pack); // insert after folder pack
                    }
                }
            }
        }
        return flattened;
    }

    /**
     * @param flatPacks ungrouped list of packs
     * @return grouped list of packs
     */
    public List<class_3288> groupByFolders(List<class_3288> flatPacks) {
        if (this.folderPacks.isEmpty()) return flatPacks;

        Map<String, String> packToFolder = new Object2ObjectOpenHashMap<>();
        for (Map.Entry<String, List<class_3288>> entry : this.folderPacks.entrySet()) {
            for (class_3288 pack : entry.getValue()) {
                packToFolder.put(pack.method_14463(), entry.getKey());
            }
        }

        Set<String> seenFolders = new ObjectOpenHashSet<>();
        List<class_3288> grouped = new ObjectArrayList<>(flatPacks.size());

        for (class_3288 pack : flatPacks) {
            String folderId = packToFolder.get(pack.method_14463());
            if (folderId != null) {
                if (seenFolders.add(folderId)) {
                    grouped.add(this.availablePacks.get(folderId));
                }
            } else {
                grouped.add(pack);
            }
        }
        return grouped;
    }

    public void openDir() {
        class_156.method_668().method_60932(this.packDir);
    }

    @Override
    public void getOrLoadIcon(class_3288 pack, Consumer<class_2960> iconCallback) {
        if (this.staleIcons != null) {
            class_2960 staleIcon = this.staleIcons.get(pack.method_14463());
            if (staleIcon != null) {
                iconCallback.accept(staleIcon);
            }
        }

        class_2960 cachedIcon = this.cachedIcons.get(pack.method_14463());
        if (cachedIcon != null) {
            iconCallback.accept(cachedIcon);
        } else {
            PackAssets.loadPackIcon(pack).thenAcceptAsync(location -> {
                this.cachedIcons.put(pack.method_14463(), location);
                iconCallback.accept(location);
            }, class_310.method_1551());
        }
    }

    public void clearIconCache() {
        this.staleIcons = new Object2ObjectOpenHashMap<>(this.cachedIcons);
        this.cachedIcons.clear();
    }

    @Override
    public boolean isResourcePacks() {
        return this.resourcePacks;
    }

    @Override
    public boolean isLocked() {
        Profile profile = this.resolver.profileSupplier().get();
        return profile != null && profile.isLocked();
    }

    @Override
    public boolean isEnabled(class_3288 pack) {
        Set<String> selectedIds = new ObjectOpenHashSet<>(this.repository.method_29210());

        if (pack instanceof FolderPack folderPack) {
            try {
                for (class_3288 nestedPack : this.getNestedPacks(folderPack)) {
                    if (selectedIds.contains(nestedPack.method_14463())) {
                        return true;
                    }
                }
            } catch (NullPointerException ignore) {
            }
            return false;
        }

        return selectedIds.contains(pack.method_14463());
    }

    @Override
    public boolean isHidden(class_3288 pack) {
        return this.resolver.isHidden(pack);
    }

    @Override
    public boolean isRequired(class_3288 pack) {
        return this.resolver.isRequiredOrDefault(pack);
    }

    @Override
    public boolean isFixed(class_3288 pack) {
        return this.resolver.isFixedOrDefault(pack);
    }

    @Override
    public @NotNull class_3288.class_3289 getPosition(class_3288 pack) {
        return this.resolver.getPositionOrDefault(pack);
    }

    @Override
    public @NotNull class_9225 getSelectionConfig(class_3288 pack) {
        return this.resolver.getSelectionConfigOrDefault(pack);
    }

    @Override
    public Config.Packs getConfig() {
        return this.resolver.config();
    }

    public Path getBaseDir() {
        return this.packDir;
    }

    public List<Path> getAdditionalDirs() {
        Path normalizedBaseDir = this.getBaseDir().toAbsolutePath().normalize();

        return ((PackRepositoryAccessor) this.repository).packed_packs$getSources().stream()
                .filter(FolderRepositorySourceAccessor.class::isInstance)
                .map(source -> ((FolderRepositorySourceAccessor) source).packed_packs$getFolder())
                .map(path -> path.toAbsolutePath().normalize())
                .filter(path -> !path.equals(normalizedBaseDir))
                .distinct()
                .toList();
    }

    @Override
    public @Nullable Folder getFolderConfig(@Nullable FolderPack folderPack) {
        if (folderPack == null) return null;
        CompletableFuture<Folder> future = this.folderConfigs.get(folderPack.method_14463());
        return future != null ? future.join() : null;
    }

    @Override
    public @Nullable Profile getProfile() {
        return this.resolver.profileSupplier().get();
    }

    public List<class_3288> getNestedPacks(FolderPack folderPack) {
        return this.validateAndOrderNestedPackIds(folderPack, Objects.requireNonNull(this.getFolderConfig(folderPack)).getPackIds());
    }
}
