package io.github.fishstiz.packed_packs.gui.components.pack;

import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.layouts.FlexLayout;
import io.github.fishstiz.fidgetz.util.DrawUtil;
import io.github.fishstiz.packed_packs.gui.components.events.FileRenameCloseEvent;
import io.github.fishstiz.packed_packs.gui.components.events.FileRenameEvent;
import io.github.fishstiz.packed_packs.gui.components.events.PackListEventListener;
import io.github.fishstiz.packed_packs.pack.PackAssetManager;
import io.github.fishstiz.packed_packs.pack.PackFileOperations;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.ToastUtil;
import org.apache.commons.io.FilenameUtils;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.util.Objects;
import java.util.regex.Pattern;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_3288;
import net.minecraft.class_332;
import net.minecraft.class_3675;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_8016;
import net.minecraft.class_8023;
import net.minecraft.class_8027;
import net.minecraft.class_8028;
import net.minecraft.class_8667;

import static io.github.fishstiz.packed_packs.util.PackUtil.ZIP_PACK_EXTENSION;
import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.CROSS_SPRITE;
import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.SPACING;

public class FileRenameModal extends Modal<class_8667> {
    private static final int MAX_LENGTH = 255;
    private static final int CONTENT_WIDTH = 256;
    private static final int SHADOW_SIZE = 24;
    private static final Pattern ILLEGAL_CHAR_PATTERN = Pattern.compile(".*[<>:\"/\\\\|?*].*");
    private final RenderableRectWidget<Void> sprite;
    private final FidgetzText<Void> title;
    private final ToggleableEditBox<Void> nameEditor;
    private final FidgetzButton<Void> saveButton;
    private final PackFileOperations fileOps;
    private final PackAssetManager assets;
    private PackList packList;
    private class_3288 pack;
    private String oldName;

    public <S extends class_437 & ToggleableDialogContainer & PackListEventListener> FileRenameModal(
            S screen,
            PackFileOperations fileOps,
            PackAssetManager assets
    ) {
        super(Modal.builder(screen, class_8667.method_52741().method_52735(SPACING)).padding(SPACING));
        this.fileOps = fileOps;
        this.assets = assets;

        this.sprite = RenderableRectWidget.<Void>builder(PackAssetManager.DEFAULT_ICON)
                .makeSquare()
                .build();
        this.title = FidgetzText.<Void>builder()
                .makeSquare()
                .setOffsetY(1)
                .setShadow(true)
                .build();
        FidgetzButton<Void> closeButton = FidgetzButton.<Void>builder()
                .makeSquare()
                .setOnPress(this::closeModal)
                .setSprite(CROSS_SPRITE)
                .build();

        this.nameEditor = ToggleableEditBox.<Void>builder()
                .setWidth(CONTENT_WIDTH)
                .setEditable(true)
                .addListener(this::handleChange)
                .setMaxLength(MAX_LENGTH)
                .setFilter(this::testInput)
                .build();

        FidgetzButton<Void> cancelButton = FidgetzButton.<Void>builder()
                .setOnPress(this::closeModal)
                .setMessage(class_5244.field_24335)
                .build();
        this.saveButton = FidgetzButton.<Void>builder()
                .setOnPress(this::saveName)
                .setMessage(class_5244.field_24334)
                .build();

        FlexLayout titleLayout = FlexLayout.horizontal(this::getContentWidth).spacing(SPACING);
        titleLayout.addChild(this.sprite);
        titleLayout.addFlexChild(this.title);
        titleLayout.addChild(closeButton);

        FlexLayout buttonLayout = FlexLayout.horizontal(this::getContentWidth).spacing(SPACING);
        buttonLayout.addFlexChild(cancelButton);
        buttonLayout.addFlexChild(this.saveButton);

        this.root().layout().method_52736(titleLayout);
        this.root().layout().method_52736(this.nameEditor);
        this.root().layout().method_52736(buttonLayout);

        this.root().method_48206(this::addRenderableWidget);

        this.addListener(this::onClose);
    }

    private int getContentWidth() {
        return CONTENT_WIDTH;
    }

    private void clearReferences() {
        this.packList = null;
        this.pack = null;
        this.oldName = null;
        this.sprite.setRenderableRect(PackAssetManager.DEFAULT_ICON);
        this.title.method_25355(class_5244.field_39003);
        this.nameEditor.method_1852("");
    }

    public void open(PackList packList, class_3288 pack) {
        this.packList = packList;
        this.pack = pack;
        this.sprite.setRenderableRect(this.assets.getIcon(pack));
        this.title.method_25355(pack.method_14457());

        this.oldName = sanitizeNameForEdit(pack);
        this.nameEditor.method_1852(this.oldName);
        this.nameEditor.method_1887(PackUtil.isZipPack(pack) ? ZIP_PACK_EXTENSION : null);
        this.saveButton.field_22763 = false;

        this.repositionElements();
        this.setOpen(true);
    }

    private boolean testInput(String input) {
        if (input == null || (!input.isEmpty() && input.isBlank())) {
            return false;
        }
        return testIllegalChars(input);
    }

    private boolean canSave(String input) {
        if (input == null || input.isBlank()) {
            return false;
        }
        if (this.pack == null || PackUtil.validatePackPath(pack) == null) {
            return false;
        }
        String trimmed = input.trim();
        if (Objects.equals(this.oldName, trimmed)) {
            return false;
        }
        return testIllegalChars(input);
    }

    private void handleChange(String name) {
        this.saveButton.field_22763 = this.canSave(name);
    }

    private void onClose(boolean open) {
        if (open) return;

        PackList target = this.packList;
        class_3288 trigger = this.pack;

        if (target != null) {
            ((PackListEventListener) this.screen).onEvent(new FileRenameCloseEvent(target, trigger));
        }

        this.clearReferences();
    }

    private void saveName() {
        String newName = this.nameEditor.method_1882();
        if (!this.canSave(newName)) {
            return;
        }

        String sanitizedName = sanitizeNameForSave(this.pack, newName);
        if (this.fileOps.renamePack(this.pack, sanitizedName)) {
            class_2561 sanitizedNameText = class_2561.method_43470(sanitizedName);
            if (this.packList != null) {
                PackList.Entry entry = this.packList.getEntry(this.pack);
                if (entry != null) {
                    entry.onRename(sanitizedNameText);
                }
            }

            ((PackListEventListener) this.screen).onEvent(new FileRenameEvent(this.packList, this.pack, sanitizedNameText));
            this.setOpen(false);
            this.clearReferences();
        } else {
            ToastUtil.onFileFailToast(ToastUtil.getRenameFailText(pack.method_14457().getString(), newName));
        }
    }

    private static String sanitizeNameForEdit(class_3288 pack) {
        String name = pack.method_14457().getString();
        return PackUtil.isZipPack(pack) ? name.replaceFirst(Pattern.quote(ZIP_PACK_EXTENSION) + "$", "") : name;
    }

    private static String sanitizeNameForSave(class_3288 pack, String newName) {
        newName = FilenameUtils.getName(newName).trim();
        return PackUtil.isZipPack(pack) ? newName + ZIP_PACK_EXTENSION : newName;
    }

    private static boolean testIllegalChars(@NonNull String input) {
        input = input.trim();
        if (!input.equals(FilenameUtils.getName(input))) {
            return false;
        }
        return !ILLEGAL_CHAR_PATTERN.matcher(input).matches();
    }

    @Override
    protected void renderBackground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        DrawUtil.renderDropShadow(guiGraphics, x, y, width, height, SHADOW_SIZE);
        super.renderBackground(guiGraphics, x, y, width, height, mouseX, mouseY, partialTick);
    }

    @Override
    public boolean method_25404(@NonNull class_11908 keyEvent) {
        boolean keyPressed = super.method_25404(keyEvent);
        if (!keyPressed && this.isOpen() && keyEvent.comp_4795() == class_3675.field_31957 && this.canSave(this.nameEditor.method_1882())) {
            this.saveName();
            return true;
        }
        return keyPressed;
    }

    @Override
    public boolean method_25400(@NonNull class_11905 characterEvent) {
        boolean charTyped = super.method_25400(characterEvent);

        if (!charTyped && this.isOpen() && !this.nameEditor.method_25370()) {
            this.method_25395(this.nameEditor);
            return this.nameEditor.method_25400(characterEvent);
        }

        return charTyped;
    }

    @Override
    public @Nullable class_8016 method_48205(@NonNull class_8023 event) {
        if (this.nameEditor.method_25370() &&
            event instanceof class_8023.class_8024(class_8028 direction) &&
            direction.method_48237() == class_8027.field_41822) {
            if (!class_310.method_1551().method_74187()) {
                this.nameEditor.method_1884(this.nameEditor.method_1881());
            }
            return class_8016.method_48194(this.nameEditor, this);
        }

        return super.method_48205(event);
    }
}
