package io.github.fishstiz.fidgetz.gui.components.contextmenu;

import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.renderables.RenderableRect;
import io.github.fishstiz.fidgetz.gui.shapes.GuiRectangle;
import io.github.fishstiz.fidgetz.util.DrawUtil;
import io.github.fishstiz.fidgetz.util.GuiUtil;
import io.github.fishstiz.fidgetz.util.ITheme;
import io.github.fishstiz.packed_packs.util.constants.Theme;
import io.github.fishstiz.packed_packs.util.lang.ObjectsUtil;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import net.minecraft.class_1144;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_364;
import net.minecraft.class_4068;
import net.minecraft.class_437;
import net.minecraft.class_6381;
import net.minecraft.class_6382;
import net.minecraft.class_8021;
import net.minecraft.class_8030;
import net.minecraft.class_8667;

import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.SPACING;

public class ContextMenu extends ToggleableDialog<LayoutWrapper<class_8667>> {
    static final BooleanSupplier DEFAULT_ACTIVE_SUPPLIER = () -> true;
    static final int DEFAULT_TEXT_INACTIVE_COLOR = Theme.GRAY_500.withAlpha(0.5f);
    static final int DEFAULT_BORDER_COLOR = Theme.GRAY_500.getARGB();
    static final int DEFAULT_BACKGROUND_COLOR = Theme.GRAY_800.getARGB();
    private static final int ITEM_HEIGHT = 20;
    private static final int MIN_WIDTH = 150;
    private static final int MENU_POINT_OFFSET = 1;
    private static final int DROP_SHADOW_SIZE = 16;
    private final io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder;
    private final List<ContextMenu> childMenus = new ArrayList<>();
    private final int borderColor;
    private final ContextMenu parentMenu;
    private Direction direction;

    protected ContextMenu(io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder) {
        super(builder);

        this.builder = builder;
        this.borderColor = builder.borderColor;
        this.parentMenu = builder.parentMenu;
        this.direction = builder.direction;
        this.addListener(open -> {
            if (!open) this.direction = this.builder.direction;
        });
    }

    @Override
    protected void clearWidgets() {
        super.clearWidgets();
        this.visitChildren(ContextMenu::clearWidgets);
        this.childMenus.clear();
    }

    private <T extends class_8021> T addChild(T child) {
        return this.root().layout().method_52736(child);
    }

    private ItemWidget<?> createItemWidget(MenuItem current, MenuItem next) {
        Integer separatorColor = ObjectsUtil.pick(ObjectsUtil.anyIdentity(next, null, MenuItem.SEPARATOR), null, this.borderColor);
        switch (current) {
            case ParentMenuItem parentItem -> {
                ContextMenu childMenu = Builder.ofChild(this.builder, this).setDirection(this.direction).build();
                this.childMenus.add(this.prependWidget(childMenu));
                return new ParentItemWidget(this.root().getWidth(), separatorColor, parentItem, this, childMenu);
            }
            case RenderableMenuItem renderableMenuItem -> {
                return new CustomItemWidget(this.root().getWidth(), separatorColor, renderableMenuItem, this);
            }
            default -> {
                return new ItemWidget<>(this.root().getWidth(), separatorColor, current, this);
            }
        }
    }

    private void setItems(List<MenuItem> items) {
        this.clearWidgets();
        this.root().setLayout(emptyLayout());

        for (int i = 0; i < items.size(); i++) {
            MenuItem current = items.get(i);
            MenuItem next = (i + 1 < items.size()) ? items.get(i + 1) : null;

            if (current == MenuItem.SEPARATOR) {
                this.addRenderableWidget(this.addChild(new Separator(this.root().method_25368(), this.borderColor)));
            } else {
                this.addRenderableWidget(this.addChild(this.createItemWidget(current, next)));
            }
        }
    }

    private void open(int x, int y, Direction direction, List<MenuItem> items) {
        if (items.isEmpty()) return;

        this.direction = direction;
        this.setItems(items);
        this.root().method_48222();
        this.root().method_48229(this.clampX(x), this.clampY(y));
        this.setOpen(true);
    }

    public void open(int x, int y, List<MenuItem> items) {
        this.open(x, y, this.builder.direction, items);
    }

    private int clampX(int x) {
        GuiRectangle bounds = this.getBoundingBox();
        this.direction = this.direction.next(this.screen, bounds, x);
        return this.direction.clamp(this.screen, bounds, x);
    }

    private int clampY(int y) {
        GuiRectangle bounds = this.getBoundingBox();
        y = y - MENU_POINT_OFFSET;
        return y + bounds.getHeight() > this.screen.field_22790 ? Math.max(0, this.screen.field_22790 - bounds.getHeight()) : y;
    }

    public int getItemHeight() {
        return ITEM_HEIGHT;
    }

    @Override
    protected void renderBackground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        DrawUtil.renderDropShadow(guiGraphics, x, y, width, height, DROP_SHADOW_SIZE);
        super.renderBackground(guiGraphics, x, y, width, height, mouseX, mouseY, partialTick);
    }

    @Override
    protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        for (ContextMenu childMenu : this.childMenus) {
            childMenu.method_25394(guiGraphics, mouseX, mouseY, partialTick);
        }
    }

    private boolean isChildMenuHovered(int mouseX, int mouseY) {
        if (this.isOpen()) {
            for (ContextMenu child : this.childMenus) {
                if (child.isOpen()) {
                    if (child.isChildMenuHovered(mouseX, mouseY)) {
                        return true;
                    }
                    if (child.isMouseOverBounds(mouseX, mouseY) || child.direction.isHovered(mouseX, mouseY, child.getBoundingBox())) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public void visitChildren(Consumer<ContextMenu> visitor) {
        for (ContextMenu child : this.childMenus) {
            visitor.accept(child);
            child.visitChildren(visitor);
        }
    }

    public void visitParents(Consumer<ContextMenu> visitor) {
        if (this.parentMenu != null) {
            visitor.accept(this.parentMenu);
            this.parentMenu.visitParents(visitor);
        }
    }

    private static class_8667 emptyLayout() {
        return class_8667.method_52741();
    }

    public static <S extends class_437 & ToggleableDialogContainer> ContextMenu.Builder builder(S screen) {
        return new ContextMenu.Builder(screen, new LayoutWrapper<>(emptyLayout(), MIN_WIDTH, 0));
    }

    public static class Builder extends ToggleableDialog.Builder<LayoutWrapper<class_8667>, io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder> {
        protected Direction direction = Direction.RIGHT;
        protected int backgroundColor = DEFAULT_BACKGROUND_COLOR;
        protected int borderColor = DEFAULT_BORDER_COLOR;
        private ContextMenu parentMenu = null;

        protected <S extends class_437 & ToggleableDialogContainer> Builder(S screen, LayoutWrapper<class_8667> root) {
            super(screen, root);
            this.focusOnOpen = false;
        }

        @SuppressWarnings("unchecked")
        protected static <S extends class_437 & ToggleableDialogContainer> io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder ofChild(io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder, ContextMenu parentMenu) {
            io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder copy = builder((S) builder.screen);
            copy.backgroundColor = builder.backgroundColor;
            copy.borderColor = builder.borderColor;
            copy.background = builder.background;
            copy.autoClose = builder.autoClose;
            copy.focusOnOpen = builder.focusOnOpen;
            copy.autoLoseFocus = builder.autoLoseFocus;
            copy.parentMenu = parentMenu;
            return copy;
        }

        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setBorderColor(int borderColor) {
            this.borderColor = borderColor;
            return this;
        }

        @Override
        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setBackground(int color) {
            this.backgroundColor = color;
            return this;
        }

        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setDirection(Direction direction) {
            this.direction = direction;
            return this;
        }

        @Override
        public ContextMenu build() {
            if (this.background == null || this.background instanceof DefaultMenuBackground) {
                this.background = new DefaultMenuBackground(this.backgroundColor, this.borderColor);
            }

            return new ContextMenu(this);
        }
    }

    private record DefaultMenuBackground(int backgroundColor, int borderColor) implements RenderableRect {
        @Override
        public void render(class_332 guiGraphics, int x, int y, int width, int height, float partialTick) {
            guiGraphics.method_25294(x, y, x + width, y + height, this.backgroundColor);
            guiGraphics.method_49601(x, y, width, height, this.borderColor);
        }
    }

    private static class ItemWidget<T extends MenuItem> extends class_339 {
        private static final int HOVER_OVERLAY_COLOR = Theme.WHITE.withAlpha(0.1f);
        private final FidgetzText<Void> text;
        private final Runnable onPress;
        private final Integer separator;
        protected final ContextMenu parent;
        protected final T item;

        private ItemWidget(int width, Integer separator, T item, ContextMenu parent) {
            super(0, 0, width, ITEM_HEIGHT, item.text());

            this.text = FidgetzText.<Void>builder().setMessage(item.text()).alignLeft().build();
            this.separator = ObjectsUtil.mapOrNull(separator, color -> ITheme.withAlpha(color, 0.15f));
            this.onPress = item.action();
            this.parent = parent;
            this.item = item;
        }

        @Override
        public boolean method_37303() {
            return super.method_37303() && this.item.active();
        }

        @Override
        public void method_25354(class_1144 handler) {
            if (this.item.active()) {
                super.method_25354(handler);
            }
        }

        protected void closeParents() {
            this.parent.setOpen(false);
            this.parent.visitParents(menu -> menu.setOpen(false));
        }

        @Override
        public void method_25348(double mouseX, double mouseY) {
            if (this.item.active()) {
                this.onPress.run();
            }
            this.closeParents();
        }

        protected void renderSeparator(class_332 guiGraphics, int minX, int maxX, int y) {
            if (this.separator != null) {
                guiGraphics.method_25292(minX, maxX, y, this.separator);
            }
        }

        protected void renderText(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
            this.text.method_46438(this.item.textColor());
            this.text.method_48229(x, y);
            this.text.method_55445(width, height);
            this.text.method_48579(guiGraphics, mouseX, mouseY, partialTick);
        }

        protected void renderHighlight(class_332 guiGraphics, int x, int y, int right, int bottom, boolean hoveredOrActive, float partialTick) {
            if (hoveredOrActive && this.item.active()) {
                guiGraphics.method_25294(x, y, right, bottom, HOVER_OVERLAY_COLOR);
            }
        }

        @SuppressWarnings("unused")
        protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, boolean hovered, double mouseX, double mouseY, float partialTick) {
            // for subclass
        }

        @Override
        protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            int x = this.method_46426();
            int y = this.method_46427();
            int width = this.method_25368();
            int height = this.method_25364();
            int right = x + width;
            int bottom = y + height;

            boolean hovered = this.method_25405(mouseX, mouseY);
            this.renderHighlight(guiGraphics, x, y, right, bottom, hovered, partialTick);
            this.renderSeparator(guiGraphics, x, right - 1, bottom - 1);
            if (this.method_25370()) guiGraphics.method_49601(x, y, width, height, Theme.WHITE.getARGB());

            int textX = x + SPACING;
            int textY = y + SPACING;
            int textWidth = width - SPACING * 2;
            int textHeight = height - SPACING * 2;

            this.renderText(guiGraphics, textX, textY, textWidth, textHeight, mouseX, mouseY, partialTick);
            this.renderForeground(guiGraphics, x, y, width, height, hovered, mouseX, mouseY, partialTick);
        }

        @Override
        public boolean method_25405(double mouseX, double mouseY) {
            if (!this.parent.isOpen()) {
                return false;
            }
            for (ContextMenu siblingChild : this.parent.childMenus) {
                if (siblingChild.isOpen() && siblingChild.isMouseOverBounds(mouseX, mouseY)) {
                    return false;
                }
            }
            return super.method_25405(mouseX, mouseY);
        }

        @Override
        protected void method_47399(class_6382 narrationElementOutput) {
            narrationElementOutput.method_37034(class_6381.field_33788, this.method_25369());
        }
    }

    private static class ParentItemWidget extends ItemWidget<ParentMenuItem> {
        private static final String CARET_RIGHT = ">";
        private final ContextMenu child;
        private boolean forcedOpen;

        ParentItemWidget(int width, Integer separator, ParentMenuItem item, ContextMenu parent, ContextMenu child) {
            super(width, separator, item, parent);
            this.child = child;
        }

        private void openChild() {
            GuiRectangle parentBounds = this.parent.getBoundingBox();
            GuiRectangle childBounds = this.child.getBoundingBox();
            Direction parentDirection = this.parent.direction;
            Direction nextDirection = parentDirection.next(parent.screen, childBounds, parentDirection.getX(parentBounds));
            int x = nextDirection.getX(parentBounds);
            int y = this.method_46427() + MENU_POINT_OFFSET;
            this.parent.visitChildren(menu -> {
                if (menu != this.child) menu.setOpen(false);
            });
            this.child.open(x, y, nextDirection, this.item.children());
        }

        private void closeChildren() {
            this.child.setOpen(false);
            this.child.visitChildren(menu -> menu.setOpen(false));
        }

        @Override
        public void method_25348(double mouseX, double mouseY) {
            if (this.item.active()) {
                this.forcedOpen = !this.forcedOpen;
            }
        }

        @Override
        protected void renderText(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
            int textWidth = width - (height + SPACING);
            super.renderText(guiGraphics, x, y, textWidth, height, mouseX, mouseY, partialTick);

            class_327 font = class_310.method_1551().field_1772;
            int caretWidth = font.method_1727(CARET_RIGHT);
            int caretHeight = font.field_2000;
            int caretX = (x + textWidth + SPACING) + (height - caretWidth) / 2;
            int caretY = y + (height - caretHeight) / 2;
            int color = this.item.textColor();

            guiGraphics.method_51433(font, CARET_RIGHT, caretX, caretY, color, false);
        }

        @Override
        protected void renderHighlight(class_332 guiGraphics, int x, int y, int right, int bottom, boolean hoveredOrActive, float partialTick) {
            super.renderHighlight(guiGraphics, x, y, right, bottom, hoveredOrActive || this.child.isOpen(), partialTick);
        }

        @Override
        protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            super.method_48579(guiGraphics, mouseX, mouseY, partialTick);

            if (!this.item.active()) {
                if (this.child.isOpen()) this.closeChildren();
                this.forcedOpen = false;
                return;
            }

            if (!this.forcedOpen && !this.method_49606() && !this.parent.isChildMenuHovered(mouseX, mouseY)) {
                this.closeChildren();
            } else if (this.method_49606() && !this.child.isOpen() && !this.parent.isChildMenuHovered(mouseX, mouseY)) {
                this.openChild();
            }
        }
    }

    private static class CustomItemWidget extends ItemWidget<RenderableMenuItem> {
        private CustomItemWidget(int width, Integer separator, RenderableMenuItem item, ContextMenu parent) {
            super(width, separator, item, parent);
        }

        @Override
        protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, boolean hovered, double mouseX, double mouseY, float partialTick) {
            RenderableRect renderer = this.item.renderer();
            if (renderer instanceof MenuItemRenderer menuItemRenderer) {
                menuItemRenderer.render(guiGraphics, x, y, width, height, SPACING, hovered, mouseX, mouseY, partialTick);
            } else {
                renderer.render(guiGraphics, x, y, width, height);
            }
        }
    }

    private static class Separator extends AbstractLayoutElement implements class_364, class_4068 {
        private final int color;

        private Separator(int width, int color) {
            this.color = color;
            this.setSize(width, 1);
        }

        @Override
        public void method_25365(boolean focused) {
            // no-op
        }

        @Override
        public boolean method_25370() {
            return false;
        }

        @Override
        public @NotNull class_8030 method_48202() {
            return super.method_48202();
        }

        @Override
        public boolean method_25405(double mouseX, double mouseY) {
            return GuiUtil.containsPoint(this, mouseX, mouseY);
        }

        @Override
        public void method_25394(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            guiGraphics.method_25292(this.method_46426(), this.getRight() - 1, this.getMidY(), this.color);
        }
    }

    public enum Direction {
        LEFT {
            @Override
            protected Direction next(class_437 screen, GuiRectangle bounds, int x) {
                return x - MENU_POINT_OFFSET - bounds.getWidth() / 2 < 0 ? RIGHT : this;
            }

            @Override
            protected int clamp(class_437 screen, GuiRectangle bounds, int x) {
                return Math.max(0, x - MENU_POINT_OFFSET - bounds.getWidth());
            }

            @Override
            protected int getX(GuiRectangle bounds) {
                return bounds.getX();
            }

            @Override
            protected boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds) {
                return mouseX >= bounds.getX() &&
                       mouseX <= bounds.getX() + bounds.getWidth() + HOVER_LEEWAY &&
                       mouseY >= bounds.getY() &&
                       mouseY <= bounds.getY() + bounds.getHeight();
            }
        },
        RIGHT {
            @Override
            protected Direction next(class_437 screen, GuiRectangle bounds, int x) {
                return x + MENU_POINT_OFFSET + bounds.getWidth() / 2 > screen.field_22789 ? LEFT : this;
            }

            @Override
            protected int clamp(class_437 screen, GuiRectangle bounds, int x) {
                int clamped = x + MENU_POINT_OFFSET;
                return clamped + bounds.getWidth() > screen.field_22789 ? screen.field_22789 - bounds.getWidth() : clamped;
            }

            @Override
            protected int getX(GuiRectangle bounds) {
                return bounds.getRight();
            }

            @Override
            protected boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds) {
                return mouseX >= bounds.getX() - HOVER_LEEWAY &&
                       mouseX <= bounds.getX() + bounds.getWidth() &&
                       mouseY >= bounds.getY() &&
                       mouseY <= bounds.getY() + bounds.getHeight();
            }
        };

        private static final int HOVER_LEEWAY = 5;

        protected abstract Direction next(class_437 screen, GuiRectangle bounds, int x);

        protected abstract int clamp(class_437 screen, GuiRectangle bounds, int x);

        protected abstract int getX(GuiRectangle bounds);

        protected abstract boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds);
    }
}
