package io.github.fishstiz.packed_packs.pack;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.Runnables;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.config.Folder;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.transform.interfaces.IPack;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionModelAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.FolderRepositorySourceAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.PackRepositoryAccessor;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.lang.CollectionsUtil;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.apache.commons.lang3.function.Consumers;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3283;
import net.minecraft.class_3288;
import net.minecraft.class_5369;

public class PackRepositoryHelper implements PackAssets {
    private final Map<String, class_2960> cachedIcons = new Object2ObjectOpenHashMap<>();
    private final Map<String, class_3288> availablePacks = new Object2ObjectLinkedOpenHashMap<>();
    private final Map<String, List<class_3288>> folderPacks = new Object2ObjectOpenHashMap<>();
    private final Map<String, CompletableFuture<Folder>> folderConfigs = new Object2ObjectOpenHashMap<>();
    private final class_3283 repository;
    private final Path packDir;
    private final class_5369 model;
    private final boolean resourcePacks;
    private Map<String, class_2960> staleIcons;

    public PackRepositoryHelper(class_3283 repository, Path packDir) {
        this.repository = repository;
        this.packDir = packDir;

        // Fabric API workaround
        this.model = new class_5369(Runnables.doNothing(), PackAssets::getDefaultIcon, this.repository, Consumers.nop());

        this.resourcePacks = this.repository == class_310.method_1551().method_1520();

        this.regenerateAvailablePacks();
    }

    public class_3283 getRepository() {
        return this.repository;
    }

    private List<class_3288> getSelectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getSelectedPacks();
    }

    private List<class_3288> getUnselectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getUnselectedPacks();
    }

    private void refreshModel() {
        ((PackSelectionModelAccessor) this.model).packed_packs$reset();
    }

    public ImmutableList<class_3288> getPacks() {
        return ImmutableList.copyOf(this.availablePacks.values());
    }

    public PackGroup getPacksByRequirement() {
        List<class_3288> required = new ArrayList<>();
        List<class_3288> optional = new ArrayList<>();

        for (class_3288 pack : this.availablePacks.values()) {
            if (pack.method_14464()) {
                pack.method_14466().method_14468(required, pack, class_3288::method_56934, true);
            } else {
                optional.add(pack);
            }
        }

        return PackGroup.of(required, optional);
    }

    public PackGroup getPacksBySelected() {
        return this.validateAndGroupPacks(this.getUnselectedPacks(), this.getSelectedPacks());
    }

    private void removePack(class_3288 pack) {
        this.repository.method_49428(pack.method_14463());
        this.availablePacks.remove(pack.method_14463());
        try {
            this.getSelectedPacks().remove(pack);
            this.getUnselectedPacks().remove(pack);
        } catch (UnsupportedOperationException e) {
            // just in case
            PackedPacks.LOGGER.warn("[packed_packs] Failed to mutate PackSelectionModel lists. Report this issue to mod author.");
        }
    }

    @Override
    public boolean deletePack(class_3288 pack) {
        if (!PackAssets.super.deletePack(pack)) {
            return false;
        }
        if (pack instanceof FolderPack) {
            Optional.ofNullable(this.folderPacks.get(pack.method_14463())).ifPresent(packs -> packs.forEach(this::removePack));
            this.folderPacks.remove(pack.method_14463());
            this.folderConfigs.remove(pack.method_14463());
        }
        this.removePack(pack);
        this.regenerateAvailablePacks();
        return true;
    }

    /**
     * @param unselected ungrouped list of unselected packs
     * @param selected   ungrouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validateAndGroupPacks(List<class_3288> unselected, List<class_3288> selected) {
        return this.validatePacks(this.groupByFolders(unselected), this.groupByFolders(selected));
    }

    private void addValidPacks(List<class_3288> source, Set<class_3288> seen, ObjectOpenHashSet<class_3288> validPacks, List<class_3288> target) {
        for (class_3288 pack : source) {
            class_3288 validPack = validPacks.get(pack); // metadata can change
            // Folder packs can show up in both columns in case user messes around in original screen. PackListBase ignores duplicates anyway.
            if (validPack != null && (seen.add(pack) || pack instanceof FolderPack)) {
                target.add(validPack);
            }
        }
    }

    /**
     * @param unselected grouped list of unselected packs
     * @param selected   grouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validatePacks(List<class_3288> unselected, List<class_3288> selected) {
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(this.availablePacks.values());
        List<class_3288> validSelected = new ArrayList<>(selected.size());
        List<class_3288> validUnselected = new ArrayList<>(unselected.size());

        this.addValidPacks(selected, seen, validPacks, validSelected);
        this.addValidPacks(unselected, seen, validPacks, validUnselected);

        for (class_3288 validPack : validPacks) {
            if (seen.add(validPack)) {
                if (validPack.method_14464()) {
                    validPack.method_14466().method_14468(validSelected, validPack, class_3288::method_56934, true);
                } else {
                    validUnselected.add(validPack);
                }
            }
        }
        return PackGroup.of(validSelected, validUnselected);
    }

    /**
     * @param folderPack   the folder pack
     * @param orderedPacks the nested packs that define the preferred order
     * @return a validated and ordered list of all packs under the folder pack
     */
    public List<class_3288> validateAndOrderNestedPacks(FolderPack folderPack, List<class_3288> orderedPacks) {
        List<class_3288> orderedValidPacks = this.folderPacks.get(folderPack.method_14463());
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(orderedValidPacks);
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        List<class_3288> finalOrderedPacks = new ArrayList<>();

        this.addValidPacks(orderedPacks, seen, validPacks, finalOrderedPacks);

        for (class_3288 validPack : orderedValidPacks) {
            if (seen.add(validPack)) {
                finalOrderedPacks.add(validPack);
            }
        }

        this.folderPacks.put(folderPack.method_14463(), finalOrderedPacks);
        return finalOrderedPacks;
    }

    public List<class_3288> validateAndOrderNestedPackIds(FolderPack folderPack, List<String> orderedPacks) {
        return this.validateAndOrderNestedPacks(folderPack, this.getPacksById(orderedPacks, this.folderPacks.get(folderPack.method_14463())));
    }

    public List<class_3288> getPacksById(List<String> packIds, Map<String, class_3288> source) {
        return CollectionsUtil.lookup(packIds, source);
    }

    public List<class_3288> getPacksById(List<String> packIds, List<class_3288> source) {
        return CollectionsUtil.lookup(packIds, CollectionsUtil.toMap(source, class_3288::method_14463));
    }

    public List<class_3288> getPacksById(List<String> packIds) {
        return this.getPacksById(packIds, this.availablePacks);
    }

    /**
     * @param packs ungrouped collection of packs
     */
    private void populateAvailablePacks(Collection<class_3288> packs) {
        for (class_3288 pack : packs) {
            IPack _pack = (IPack) pack;
            if (_pack.packed_packs$nestedPack()) {
                Path folderPath = Objects.requireNonNull(_pack.packed_packs$getPath()).getParent();
                String folderName = PackUtil.generatePackName(folderPath);
                String folderId = PackUtil.generatePackId(folderName);
                if (!this.availablePacks.containsKey(folderId)) {
                    FolderPack folderPack = new FolderPack(folderId, folderName, folderPath);
                    this.folderConfigs.put(folderId, folderPack.loadConfig());
                    this.availablePacks.put(folderId, folderPack);
                }
                this.folderPacks.computeIfAbsent(folderId, id -> new ArrayList<>()).add(pack);
            } else {
                this.availablePacks.put(pack.method_14463(), pack);
            }
        }
    }

    private void regenerateAvailablePacks() {
        this.availablePacks.clear();
        this.folderPacks.clear();
        this.folderConfigs.clear();
        this.populateAvailablePacks(this.getSelectedPacks());
        this.populateAvailablePacks(this.getUnselectedPacks());
    }

    public void refresh() {
        ((PackSelectionModelAccessor) this.model).packed_packs$reset();
        this.model.method_29981();
        this.regenerateAvailablePacks();
    }

    /**
     * @param selected grouped list of selected packs
     */
    public void selectPacks(List<class_3288> selected) {
        this.repository.method_14447(Lists.reverse(this.flattenPacks(selected)).stream().map(class_3288::method_14463).collect(ImmutableList.toImmutableList()));
        this.refreshModel();
    }

    /**
     * @param groupedPacks grouped list of packs
     * @return flattened list of packs
     */
    public List<class_3288> flattenPacks(List<class_3288> groupedPacks) {
        if (this.folderPacks.isEmpty()) return groupedPacks;

        List<class_3288> flattened = new ArrayList<>(groupedPacks);
        for (int i = flattened.size() - 1; i >= 0; i--) {
            if (flattened.get(i) instanceof FolderPack folderPack) {
                flattened.remove(i);
                List<class_3288> nested = this.getNestedPacks(folderPack);
                if (nested != null) {
                    for (class_3288 pack : Lists.reverse(nested)) {
                        flattened.add(i, pack);
                    }
                }
            }
        }
        return flattened;
    }

    /**
     * @param flatPacks ungrouped list of packs
     * @return grouped list of packs
     */
    public List<class_3288> groupByFolders(List<class_3288> flatPacks) {
        if (this.folderPacks.isEmpty()) return flatPacks;

        Map<String, String> packToFolder = new Object2ObjectOpenHashMap<>();
        for (Map.Entry<String, List<class_3288>> entry : this.folderPacks.entrySet()) {
            for (class_3288 pack : entry.getValue()) {
                packToFolder.put(pack.method_14463(), entry.getKey());
            }
        }

        Set<String> seenFolders = new ObjectOpenHashSet<>();
        List<class_3288> grouped = new ArrayList<>(flatPacks.size());

        for (class_3288 pack : flatPacks) {
            String folderId = packToFolder.get(pack.method_14463());
            if (folderId != null) {
                if (seenFolders.add(folderId)) {
                    grouped.add(this.availablePacks.get(folderId));
                }
            } else {
                grouped.add(pack);
            }
        }
        return grouped;
    }

    public void openDir() {
        class_156.method_668().method_60932(this.packDir);
    }

    @Override
    public void getOrLoadIcon(class_3288 pack, Consumer<class_2960> iconCallback) {
        if (this.staleIcons != null) {
            class_2960 staleIcon = this.staleIcons.get(pack.method_14463());
            if (staleIcon != null) {
                iconCallback.accept(staleIcon);
            }
        }

        class_2960 cachedIcon = this.cachedIcons.get(pack.method_14463());
        if (cachedIcon != null) {
            iconCallback.accept(cachedIcon);
        } else {
            PackAssets.loadPackIcon(pack).thenAcceptAsync(location -> {
                this.cachedIcons.put(pack.method_14463(), location);
                iconCallback.accept(location);
            }, class_310.method_1551());
        }
    }

    public void clearIconCache() {
        this.staleIcons = new Object2ObjectOpenHashMap<>(this.cachedIcons);
        this.cachedIcons.clear();
    }

    @Override
    public boolean isResourcePacks() {
        return this.resourcePacks;
    }

    @Override
    public boolean isEnabled(class_3288 pack) {
        Set<String> selectedIds = new ObjectOpenHashSet<>(this.repository.method_29210());

        if (pack instanceof FolderPack folderPack) {
            try {
                for (class_3288 nestedPack : this.getNestedPacks(folderPack)) {
                    if (selectedIds.contains(nestedPack.method_14463())) {
                        return true;
                    }
                }
            } catch (NullPointerException ignore) {
            }
            return false;
        }

        return selectedIds.contains(pack.method_14463());
    }

    public Path getBaseDir() {
        return this.packDir;
    }

    public List<Path> getAdditionalDirs() {
        Path normalizedBaseDir = this.getBaseDir().toAbsolutePath().normalize();

        return ((PackRepositoryAccessor) this.repository).packed_packs$getSources().stream()
                .filter(FolderRepositorySourceAccessor.class::isInstance)
                .map(source -> ((FolderRepositorySourceAccessor) source).packed_packs$getFolder())
                .map(path -> path.toAbsolutePath().normalize())
                .filter(path -> !path.equals(normalizedBaseDir))
                .distinct()
                .toList();
    }

    public @Nullable Folder getFolderConfig(FolderPack folderPack) {
        CompletableFuture<Folder> future = this.folderConfigs.get(folderPack.method_14463());
        return future != null ? future.join() : null;
    }

    public List<class_3288> getNestedPacks(FolderPack folderPack) {
        return this.validateAndOrderNestedPackIds(folderPack, Objects.requireNonNull(this.getFolderConfig(folderPack)).getPackIds());
    }

    public record PackGroup(ImmutableList<class_3288> selected, ImmutableList<class_3288> unselected) {
        private static PackGroup of(List<class_3288> selected, List<class_3288> unselected) {
            return new PackGroup(ImmutableList.copyOf(selected), ImmutableList.copyOf(unselected));
        }
    }
}
