package io.github.fishstiz.fidgetz.gui.components.contextmenu;

import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.renderables.ColoredRect;
import io.github.fishstiz.fidgetz.gui.renderables.RenderableRect;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.Sprite;
import io.github.fishstiz.fidgetz.gui.shapes.GuiRectangle;
import io.github.fishstiz.fidgetz.transform.interfaces.UnpaddedScrollableLayout;
import io.github.fishstiz.fidgetz.util.DrawUtil;
import io.github.fishstiz.fidgetz.util.GuiUtil;
import io.github.fishstiz.fidgetz.util.ARGBColor;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import net.minecraft.class_1144;
import net.minecraft.class_11467;
import net.minecraft.class_11875;
import net.minecraft.class_11909;
import net.minecraft.class_124;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_6381;
import net.minecraft.class_6382;
import net.minecraft.class_8667;
import net.minecraft.class_9848;

public class ContextMenu extends ToggleableDialog<LayoutWrapper<class_11467>> {
    static final int DEFAULT_BORDER_COLOR = ARGBColor.withAlpha(Objects.requireNonNull(class_124.field_1080.method_532()), 1);
    static final int DEFAULT_TEXT_INACTIVE_COLOR = ARGBColor.withAlpha(DEFAULT_BORDER_COLOR, 0.5f);
    static final int DEFAULT_BACKGROUND_COLOR = ARGBColor.withAlpha(Objects.requireNonNull(class_124.field_1063.method_532()), 1);
    private static final int DEFAULT_SPACING = class_4185.field_46856;
    private static final int ITEM_HEIGHT = 20;
    private static final int MAX_HEIGHT = ITEM_HEIGHT * 10;
    private static final int MIN_WIDTH = 150;
    private static final int MENU_POINT_OFFSET = 1;
    private static final int DROP_SHADOW_SIZE = 16;
    private final io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder;
    private final List<ContextMenu> childMenus = new ArrayList<>();
    private final int spacing;
    private final int borderColor;
    private final int softSeparatorColor;
    private final ContextMenu parentMenu;
    private Direction direction;
    private boolean forceOpen;

    protected ContextMenu(io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder) {
        super(builder);

        this.builder = builder;
        this.spacing = builder.spacing;
        this.borderColor = builder.borderColor;
        this.softSeparatorColor = ARGBColor.withAlpha(this.borderColor, 0.15f);
        this.parentMenu = builder.parentMenu;
        this.direction = builder.direction;
        this.addListener(open -> {
            if (!open) {
                this.direction = this.builder.direction;
                this.forceOpen = false;
            }
        });
    }

    @Override
    protected void clearWidgets() {
        super.clearWidgets();
        this.visitChildren(ContextMenu::clearWidgets);
        this.childMenus.clear();
    }

    private ItemWidget<?> createItemWidget(MenuItem item) {
        switch (item) {
            case ParentMenuItem parentItem -> {
                ContextMenu childMenu = Builder.ofChild(this.builder, this).setDirection(this.direction).build();
                this.childMenus.add(childMenu);
                return new ParentItemWidget(MIN_WIDTH, this.spacing, parentItem, this, childMenu);
            }
            case RenderableMenuItem renderableMenuItem -> {
                return new CustomItemWidget(MIN_WIDTH, this.spacing, renderableMenuItem, this);
            }
            default -> {
                return new ItemWidget<>(MIN_WIDTH, this.spacing, item, this);
            }
        }
    }

    private void setItems(List<? extends MenuItem> items) {
        this.clearWidgets();

        class_8667 content = class_8667.method_52741();

        for (int i = 0; i < items.size(); i++) {
            MenuItem current = items.get(i);
            MenuItem next = (i + 1 < items.size()) ? items.get(i + 1) : null;

            if (current == MenuItem.SEPARATOR) {
                content.method_52736(new Separator(MIN_WIDTH, this.borderColor));
                continue;
            }

            content.method_52736(this.createItemWidget(current));

            if (current.shouldAutoSeparate() && next != null && next.shouldAutoSeparate()) {
                content.method_52736(new Separator(MIN_WIDTH, this.softSeparatorColor));
            }
        }

        this.childMenus.forEach(this::addWidget);

        content.method_48222();
        class_11467 layout = new class_11467(class_310.method_1551(), content, content.method_25364());
        ((UnpaddedScrollableLayout) layout).fidgetz$setUnpadded(true);

        layout.method_71807(MAX_HEIGHT);
        layout.method_48206(this::addRenderableWidget);

        this.root().setLayout(layout);
        this.root().method_48206(this::addRenderableWidget);
    }

    private void open(int x, int y, Direction direction, List<? extends MenuItem> items) {
        if (items.isEmpty()) return;

        this.direction = direction;
        this.setItems(items);
        this.root().method_48222();
        this.root().method_48229(this.clampX(x), this.clampY(y));
        this.setOpen(true);
    }

    public void open(int x, int y, List<MenuItem> items) {
        this.open(x, y, this.builder.direction, items);
    }

    private int clampX(int x) {
        GuiRectangle bounds = this.getBoundingBox();
        this.direction = this.direction.next(this.screen, bounds, x);
        return this.direction.clamp(this.screen, bounds, x);
    }

    private int clampY(int y) {
        GuiRectangle bounds = this.getBoundingBox();
        return y + bounds.getHeight() > this.screen.field_22790 ? Math.max(0, this.screen.field_22790 - bounds.getHeight()) : y;
    }

    public int getItemHeight() {
        return ITEM_HEIGHT;
    }

    @Override
    protected void renderBackground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        DrawUtil.renderDropShadow(guiGraphics, x, y, width, height, DROP_SHADOW_SIZE);
        super.renderBackground(guiGraphics, x, y, width, height, mouseX, mouseY, partialTick);
    }

    @Override
    protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        DrawUtil.renderOutline(guiGraphics, x, y, width, height, this.borderColor);

        for (ContextMenu childMenu : this.childMenus) {
            childMenu.method_25394(guiGraphics, mouseX, mouseY, partialTick);
        }

        GuiRectangle bounds = this.getBoundingBox();
        if (GuiUtil.containsPoint(bounds.getX() + MENU_POINT_OFFSET, bounds.getY(), bounds.getWidth(), bounds.getHeight(), mouseX, mouseY)) {
            guiGraphics.method_74037(class_11875.field_62449);
        }
    }

    public @Nullable ContextMenu getOpenedChildMenu() {
        for (ContextMenu child : this.childMenus) {
            if (child.isOpen()) return child;
        }
        return null;
    }

    private boolean isHoveredAtDirection(int mouseX, int mouseY) {
        return this.isOpen() && (this.isMouseOverBounds(mouseX, mouseY) || this.direction.isHovered(mouseX, mouseY, this.getBoundingBox()));
    }

    @Override
    public boolean method_25405(double mouseX, double mouseY) {
        if (!this.isOpen()) return false;

        ContextMenu child = this.getOpenedChildMenu();
        boolean withinBounds = child != null
                ? this.getBoundingBox().containsPoint(mouseX, mouseY) || child.getBoundingBox().containsPoint(mouseX, mouseY)
                : this.getBoundingBox().containsPoint(mouseX, mouseY);

        return withinBounds && super.method_25405(mouseX, mouseY);
    }

    public void visitChildren(Consumer<ContextMenu> visitor) {
        for (ContextMenu child : this.childMenus) {
            visitor.accept(child);
            child.visitChildren(visitor);
        }
    }

    public void visitParents(Consumer<ContextMenu> visitor) {
        if (this.parentMenu != null) {
            visitor.accept(this.parentMenu);
            this.parentMenu.visitParents(visitor);
        }
    }

    private void closeCascade() {
        this.setOpen(false);
        this.visitParents(parent -> parent.setOpen(false));
    }

    public static <S extends class_437 & ToggleableDialogContainer> io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder(S screen) {
        return new io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder(screen, new LayoutWrapper<>(new class_11467(class_310.method_1551(), class_8667.method_52741(), MAX_HEIGHT), MIN_WIDTH, 0));
    }

    public static class Builder extends ToggleableDialog.Builder<LayoutWrapper<class_11467>, io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder> {
        protected Direction direction = Direction.RIGHT;
        protected int backgroundColor = DEFAULT_BACKGROUND_COLOR;
        protected int borderColor = DEFAULT_BORDER_COLOR;
        protected int spacing = DEFAULT_SPACING;
        private ContextMenu parentMenu = null;

        protected <S extends class_437 & ToggleableDialogContainer> Builder(S screen, LayoutWrapper<class_11467> root) {
            super(screen, root);
            this.focusOnOpen = false;
        }

        @SuppressWarnings("unchecked")
        protected static <S extends class_437 & ToggleableDialogContainer> io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder ofChild(io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder builder, ContextMenu parentMenu) {
            io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder copy = builder((S) builder.screen);
            copy.spacing = builder.spacing;
            copy.backgroundColor = builder.backgroundColor;
            copy.borderColor = builder.borderColor;
            copy.background = builder.background;
            copy.autoClose = builder.autoClose;
            copy.focusOnOpen = builder.focusOnOpen;
            copy.autoLoseFocus = builder.autoLoseFocus;
            copy.parentMenu = parentMenu;
            return copy;
        }

        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setSpacing(int spacing) {
            this.spacing = spacing;
            return this;
        }

        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setBorderColor(int borderColor) {
            this.borderColor = borderColor;
            return this;
        }

        @Override
        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setBackground(int color) {
            this.backgroundColor = color;
            return this;
        }

        public io.github.fishstiz.fidgetz.gui.components.contextmenu.ContextMenu.Builder setDirection(Direction direction) {
            this.direction = direction;
            return this;
        }

        @Override
        public ContextMenu build() {
            if (this.background == null) {
                this.background = new ColoredRect(this.backgroundColor);
            }

            return new ContextMenu(this);
        }
    }

    private static class ItemWidget<T extends MenuItem> extends class_339 implements Fidgetz {
        private static final int HOVER_OVERLAY_COLOR = class_9848.method_61317(0.1f);
        private final FidgetzText<Void> text;
        protected final ContextMenu parent;
        protected final T item;
        protected final int spacing;

        private ItemWidget(int width, int spacing, T item, ContextMenu parent) {
            super(0, 0, width, ITEM_HEIGHT, item.text());
            this.spacing = spacing;
            this.text = FidgetzText.<Void>builder()
                    .setOffsetY(MENU_POINT_OFFSET)
                    .setShadow(true)
                    .setMessage(item.text())
                    .build();
            this.parent = parent;
            this.item = item;
        }

        @Override
        public void method_25354(class_1144 handler) {
            if (this.item.active()) {
                super.method_25354(handler);
            }
        }

        @Override
        public void method_25348(class_11909 mouseButtonEvent, boolean doubleClicked) {
            if (this.item.active()) {
                this.item.action().run();
            }
            if (this.item.shouldCloseOnInteract()) {
                this.parent.closeCascade();
            }
        }

        protected void renderBackground(class_332 guiGraphics, int x, int y, int width, int height, float partialTick) {
            RenderableRect background = this.item.background();
            if (background != null) {
                background.render(guiGraphics, x, y, width, height, partialTick);
            }
        }

        protected void renderIcon(class_332 guiGraphics, int x, int y, int width, int height, float partialTick) {
            Sprite icon = this.item.icon();
            if (icon != null) {
                icon.render(guiGraphics, x, y, width, height, partialTick);
            }
        }

        protected void renderText(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
            this.text.method_46438(this.item.textColor());
            this.text.method_48229(x, y);
            this.text.method_55445(width, height);
            this.text.method_48579(guiGraphics, mouseX, mouseY, partialTick);
        }

        protected void renderHighlight(class_332 guiGraphics, int x, int y, int right, int bottom, boolean hovered, float partialTick) {
            if (hovered && this.item.active()) {
                guiGraphics.method_25294(x, y, right, bottom, HOVER_OVERLAY_COLOR);
            }
        }

        @SuppressWarnings("unused")
        protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, boolean hovered, double mouseX, double mouseY, float partialTick) {
            // for subclass
        }

        @Override
        protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            this.field_22762 = this.field_22762 && this.method_25405(mouseX, mouseY);

            int x = this.method_46426();
            int y = this.method_46427();
            int width = this.method_25368();
            int height = this.method_25364();
            int right = x + width;
            int bottom = y + height;

            this.renderBackground(guiGraphics, x, y, width, height, partialTick);
            this.renderHighlight(guiGraphics, x, y, right, bottom, this.field_22762, partialTick);

            int size = class_310.method_1551().field_1772.field_2000;
            int innerX = x + this.spacing;
            int innerY = y + this.spacing;
            int innerWidth = width - this.spacing * 2;
            int innerHeight = height - this.spacing * 2;
            int iconY = innerY + (innerHeight - size) / 2;
            int textX = this.item.icon() != null ? innerX + size + this.spacing : innerX;
            int textWidth = this.item.icon() != null ? innerWidth - size - this.spacing : innerWidth;

            this.renderIcon(guiGraphics, innerX, iconY, size, size, partialTick);
            this.renderText(guiGraphics, textX, innerY, textWidth, innerHeight, mouseX, mouseY, partialTick);
            this.renderForeground(guiGraphics, x, y, width, height, this.field_22762, mouseX, mouseY, partialTick);
        }

        @Override
        public boolean method_25405(double mouseX, double mouseY) {
            if (!this.parent.isOpen()) {
                return false;
            }
            for (ContextMenu siblingChild : this.parent.childMenus) {
                if (siblingChild.isOpen() && siblingChild.isMouseOverBounds(mouseX, mouseY)) {
                    return false;
                }
            }
            return super.method_25405(mouseX, mouseY);
        }

        @Override
        protected void method_47399(class_6382 narrationElementOutput) {
            narrationElementOutput.method_37034(class_6381.field_33788, this.method_25369());
        }
    }

    private static class ParentItemWidget extends ItemWidget<ParentMenuItem> {
        private static final String CARET_RIGHT = ">";
        private final ContextMenu child;

        ParentItemWidget(int width, int spacing, ParentMenuItem item, ContextMenu parent, ContextMenu child) {
            super(width, spacing, item, parent);
            this.child = child;
        }

        private void openChild() {
            GuiRectangle parentBounds = this.parent.getBoundingBox();
            GuiRectangle childBounds = this.child.getBoundingBox();
            Direction parentDirection = this.parent.direction;
            Direction nextDirection = parentDirection.next(parent.screen, childBounds, parentDirection.getX(parentBounds));
            int x = nextDirection.getX(this);
            int y = this.method_46427();
            this.parent.visitChildren(menu -> {
                if (menu != this.child) menu.setOpen(false);
            });
            this.child.open(x, y, nextDirection, this.item.children());
        }

        private void closeChildren() {
            this.child.setOpen(false);
            this.child.visitChildren(menu -> menu.setOpen(false));
        }

        @Override
        public void method_25348(class_11909 mouseButtonEvent, boolean doubleClicked) {
            if (this.item.active()) {
                this.item.action().run();
                this.child.forceOpen = !this.child.forceOpen || !this.child.isOpen();
                if (!this.child.isOpen()) {
                    this.openChild();
                }
            } else if (this.item.shouldCloseOnInteract()) {
                this.parent.closeCascade();
            }
        }

        private boolean isWithinParentXBounds(int mouseX, int mouseY) {
            if (this.method_49606()) return true;

            GuiRectangle parentBox = this.parent.getBoundingBox();
            return mouseX >= parentBox.getX() &&
                   mouseX <= parentBox.getRight() &&
                   mouseY >= this.method_46427() &&
                   mouseY <= this.method_55443();
        }

        @Override
        protected void renderText(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
            int textWidth = width - (height + this.spacing);
            super.renderText(guiGraphics, x, y, textWidth, height, mouseX, mouseY, partialTick);

            if (this.item.active()) {
                class_327 font = class_310.method_1551().field_1772;
                int caretWidth = font.method_1727(CARET_RIGHT);
                int caretHeight = font.field_2000;
                int caretX = (x + textWidth + this.spacing) + (height - caretWidth) / 2;
                int caretY = y + (height - caretHeight) / 2;
                int color = this.item.textColor();

                guiGraphics.method_51433(font, CARET_RIGHT, caretX, caretY, color, false);
            }
        }

        @Override
        protected void renderHighlight(class_332 guiGraphics, int x, int y, int right, int bottom, boolean hovered, float partialTick) {
            super.renderHighlight(guiGraphics, x, y, right, bottom, hovered || this.child.isOpen(), partialTick);
        }

        @Override
        protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            super.method_48579(guiGraphics, mouseX, mouseY, partialTick);

            if (!this.item.active()) {
                if (this.child.isOpen()) this.closeChildren();
                this.child.forceOpen = false;
                return;
            }

            boolean hovered = this.isWithinParentXBounds(mouseX, mouseY) && guiGraphics.method_58135(mouseX, mouseY);
            ContextMenu sibling = this.parent.getOpenedChildMenu();

            if (!hovered && this.child.isOpen() && !this.child.forceOpen && (sibling == null || !sibling.isHoveredAtDirection(mouseX, mouseY))) {
                this.closeChildren();
            } else if (hovered && !this.child.isOpen() && (sibling == null || !sibling.forceOpen && !sibling.isHoveredAtDirection(mouseX, mouseY))) {
                this.openChild();
            }
        }
    }

    private static class CustomItemWidget extends ItemWidget<RenderableMenuItem> {
        private CustomItemWidget(int width, int spacing, RenderableMenuItem item, ContextMenu parent) {
            super(width, spacing, item, parent);
        }

        @Override
        protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, boolean hovered, double mouseX, double mouseY, float partialTick) {
            this.item.renderer().render(guiGraphics, x, y, width, height, partialTick);
        }
    }

    private static class Separator extends class_339 implements Fidgetz {
        private final int color;

        private Separator(int width, int color) {
            super(0, 0, width, 1, class_5244.field_39003);
            this.color = color;
        }

        @Override
        public boolean method_25370() {
            return false;
        }

        @Override
        public boolean method_25402(class_11909 mouseButtonEvent, boolean bl) {
            return false;
        }

        @Override
        public boolean method_25405(double mouseX, double mouseY) {
            return GuiUtil.containsPoint(this, mouseX, mouseY);
        }

        @Override
        protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            guiGraphics.method_51738(this.method_46426(), this.method_55442() - 1, this.getMidY(), this.color);
        }

        @Override
        protected void method_47399(class_6382 narrationElementOutput) {
            // no-op
        }
    }

    public enum Direction {
        LEFT {
            @Override
            protected Direction next(class_437 screen, GuiRectangle bounds, int x) {
                return x - bounds.getWidth() / 2 < 0 ? RIGHT : this;
            }

            @Override
            protected int clamp(class_437 screen, GuiRectangle bounds, int x) {
                return Math.max(0, x - bounds.getWidth());
            }

            @Override
            protected int getX(GuiRectangle bounds) {
                return bounds.getX();
            }

            @Override
            protected boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds) {
                return mouseX >= bounds.getX() &&
                       mouseX <= bounds.getX() + bounds.getWidth() + HOVER_LEEWAY &&
                       mouseY >= bounds.getY() &&
                       mouseY <= bounds.getY() + bounds.getHeight();
            }
        },
        RIGHT {
            @Override
            protected Direction next(class_437 screen, GuiRectangle bounds, int x) {
                return x + bounds.getWidth() / 2 > screen.field_22789 ? LEFT : this;
            }

            @Override
            protected int clamp(class_437 screen, GuiRectangle bounds, int x) {
                return x + bounds.getWidth() > screen.field_22789 ? screen.field_22789 - bounds.getWidth() : x;
            }

            @Override
            protected int getX(GuiRectangle bounds) {
                return bounds.getRight();
            }

            @Override
            protected boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds) {
                return mouseX >= bounds.getX() - HOVER_LEEWAY &&
                       mouseX <= bounds.getX() + bounds.getWidth() &&
                       mouseY >= bounds.getY() &&
                       mouseY <= bounds.getY() + bounds.getHeight();
            }
        };

        private static final int HOVER_LEEWAY = 5;

        protected abstract Direction next(class_437 screen, GuiRectangle bounds, int x);

        protected abstract int clamp(class_437 screen, GuiRectangle bounds, int x);

        protected abstract int getX(GuiRectangle bounds);

        protected abstract boolean isHovered(int mouseX, int mouseY, GuiRectangle bounds);
    }
}
