package io.github.fishstiz.packed_packs.gui.components.pack;

import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.Sprite;
import io.github.fishstiz.fidgetz.util.DrawUtil;
import io.github.fishstiz.packed_packs.gui.components.events.FileRenameCloseEvent;
import io.github.fishstiz.packed_packs.gui.components.events.FileRenameEvent;
import io.github.fishstiz.packed_packs.gui.components.events.PackListEventListener;
import io.github.fishstiz.packed_packs.pack.PackAssetManager;
import io.github.fishstiz.packed_packs.pack.PackFileOperations;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.ToastUtil;
import org.apache.commons.io.FilenameUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.regex.Pattern;
import net.minecraft.class_2561;
import net.minecraft.class_3288;
import net.minecraft.class_332;
import net.minecraft.class_3675;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_7847;
import net.minecraft.class_8016;
import net.minecraft.class_8023;
import net.minecraft.class_8027;
import net.minecraft.class_8667;

import static io.github.fishstiz.packed_packs.util.PackUtil.ZIP_PACK_EXTENSION;
import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.SPACING;

public class FileRenameModal extends Modal<class_8667> {
    private static final int MAX_LENGTH = 255;
    private static final int CONTENT_WIDTH = 200;
    private static final int SHADOW_SIZE = 24;
    private static final int TITLE_HEIGHT = 16;
    private static final Pattern ILLEGAL_CHAR_PATTERN = Pattern.compile(".*[<>:\"/\\\\|?*].*");
    private static final Sprite DEFAULT_SPRITE = Sprite.of16(PackAssetManager.DEFAULT_ICON);
    private final ToggleableEditBox<Void> nameEditor = ToggleableEditBox.<Void>builder()
            .setEditable(true)
            .addListener(this::handleChange)
            .setMaxLength(MAX_LENGTH)
            .setFilter(this::testInput)
            .build();
    private final FidgetzText<Void> title = FidgetzText.<Void>builder()
            .alignLeft()
            .setHeight(TITLE_HEIGHT)
            .setOffsetY(1)
            .setShadow(true)
            .build();
    private final FidgetzButton<Void> saveButton = FidgetzButton.<Void>builder()
            .setOnPress(this::saveName)
            .setMessage(class_5244.field_24334)
            .build();
    private final PackFileOperations fileOps;
    private final PackAssetManager assets;
    private Sprite sprite = DEFAULT_SPRITE;
    private PackList packList;
    private class_3288 pack;
    private String oldName;

    public <S extends class_437 & ToggleableDialogContainer & PackListEventListener> FileRenameModal(
            S screen,
            PackFileOperations fileOps,
            PackAssetManager assets
    ) {
        super(Modal.builder(screen, class_8667.method_52741()));

        this.fileOps = fileOps;
        this.assets = assets;

        class_7847 rootLayoutSettings = class_7847.method_46481().method_46477(SPACING).method_46471(SPACING);
        this.root().layout().method_52737(this.title, rootLayoutSettings);
        this.root().layout().method_52737(this.nameEditor, rootLayoutSettings);

        FidgetzButton<Void> cancelButton = FidgetzButton.<Void>builder()
                .setOnPress(() -> this.setOpen(false))
                .setMessage(class_5244.field_24335)
                .build();
        class_8667 buttonLayout = class_8667.method_52742();
        buttonLayout.method_52737(cancelButton, class_7847.method_46481().method_46477(SPACING));
        buttonLayout.method_52736(this.saveButton);

        this.root().layout().method_52737(buttonLayout, class_7847.method_46481().method_46471(SPACING).method_46475((int) (SPACING * 1.5)));
        this.root().layout().method_48206(widget -> widget.method_25358(CONTENT_WIDTH));
        buttonLayout.method_48206(widget -> widget.method_25358((CONTENT_WIDTH - SPACING) / 2));

        this.root().method_48222();
        this.root().method_48206(this::addRenderableWidget);

        this.addListener(this::onClose);
    }

    @Override
    public void repositionElements() {
        this.title.method_25358(this.title.method_25368() - this.sprite.width - SPACING);
        super.repositionElements();
        this.title.method_46421(this.title.method_46426() + this.sprite.width + SPACING);
    }

    private void clearReferences() {
        this.packList = null;
        this.pack = null;
        this.oldName = null;
        this.sprite = DEFAULT_SPRITE;
        this.title.method_25355(class_5244.field_39003);
        this.nameEditor.method_1852("");
    }

    public void open(PackList packList, class_3288 pack) {
        this.packList = packList;
        this.pack = pack;
        this.sprite = Sprite.of16(PackAssetManager.getDefaultIcon(pack));
        this.assets.getOrLoadIcon(pack, icon -> this.sprite = Sprite.of16(icon));
        this.title.method_25355(pack.method_14457());

        this.oldName = sanitizeNameForEdit(pack);
        this.nameEditor.method_1852(this.oldName);
        this.nameEditor.method_1887(PackUtil.isZipPack(pack) ? ZIP_PACK_EXTENSION : null);
        this.saveButton.field_22763 = false;

        this.setOpen(true);
    }

    private boolean testInput(String input) {
        if (input == null || (!input.isEmpty() && input.isBlank())) {
            return false;
        }
        return testIllegalChars(input);
    }

    private boolean canSave(String input) {
        if (input == null || input.isBlank()) {
            return false;
        }
        if (this.pack == null || PackUtil.validatePackPath(pack) == null) {
            return false;
        }
        String trimmed = input.trim();
        if (Objects.equals(this.oldName, trimmed)) {
            return false;
        }
        return testIllegalChars(input);
    }

    private void handleChange(String name) {
        this.saveButton.field_22763 = this.canSave(name);
    }

    private void onClose(boolean open) {
        if (open) return;

        PackList target = this.packList;
        class_3288 trigger = this.pack;

        if (target != null) {
            ((PackListEventListener) this.screen).onEvent(new FileRenameCloseEvent(target, trigger));
        }

        this.clearReferences();
    }

    private void saveName() {
        String newName = this.nameEditor.method_1882();
        if (!this.canSave(newName)) {
            return;
        }

        String sanitizedName = sanitizeNameForSave(this.pack, newName);
        if (this.fileOps.renamePack(this.pack, sanitizedName)) {
            class_2561 sanitizedNameText = class_2561.method_43470(sanitizedName);
            if (this.packList != null) {
                PackList.Entry entry = this.packList.getEntry(this.pack);
                if (entry != null) {
                    entry.onRename(sanitizedNameText);
                }
            }

            ((PackListEventListener) this.screen).onEvent(new FileRenameEvent(this.packList, this.pack, sanitizedNameText));
            this.setOpen(false);
            this.clearReferences();
        } else {
            ToastUtil.onFileFailToast(ToastUtil.getRenameFailText(pack.method_14457().getString(), newName));
        }
    }

    private static String sanitizeNameForEdit(class_3288 pack) {
        String name = pack.method_14457().getString();
        return PackUtil.isZipPack(pack) ? name.replaceFirst(Pattern.quote(ZIP_PACK_EXTENSION) + "$", "") : name;
    }

    private static String sanitizeNameForSave(class_3288 pack, String newName) {
        newName = FilenameUtils.getName(newName).trim();
        return PackUtil.isZipPack(pack) ? newName + ZIP_PACK_EXTENSION : newName;
    }

    private static boolean testIllegalChars(@NotNull String input) {
        input = input.trim();
        if (!input.equals(FilenameUtils.getName(input))) {
            return false;
        }
        return !ILLEGAL_CHAR_PATTERN.matcher(input).matches();
    }

    @Override
    protected void renderBackground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        DrawUtil.renderDropShadow(guiGraphics, x, y, width, height, SHADOW_SIZE);
        super.renderBackground(guiGraphics, x, y, width, height, mouseX, mouseY, partialTick);
    }

    @Override
    protected void renderForeground(class_332 guiGraphics, int x, int y, int width, int height, int mouseX, int mouseY, float partialTick) {
        int spriteX = x + SPACING;
        int spriteY = this.title.method_46427() + (this.title.method_25364() - this.sprite.height) / 2;
        this.sprite.render(guiGraphics, spriteX, spriteY);
    }

    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        boolean keyPressed = super.method_25404(keyCode, scanCode, modifiers);
        if (!keyPressed && this.isOpen() && keyCode == class_3675.field_31957 && this.canSave(this.nameEditor.method_1882())) {
            this.saveName();
            return true;
        }
        return keyPressed;
    }

    @Override
    public boolean method_25400(char codePoint, int modifiers) {
        boolean charTyped = super.method_25400(codePoint, modifiers);

        if (!charTyped && this.isOpen() && !this.nameEditor.method_25370()) {
            this.method_25395(this.nameEditor);
            return this.nameEditor.method_25400(codePoint, modifiers);
        }

        return charTyped;
    }

    @Override
    public @Nullable class_8016 method_48205(class_8023 event) {
        if (this.nameEditor.method_25370() &&
            event instanceof class_8023.class_8024(ScreenDirection direction) &&
            direction.method_48237() == class_8027.field_41822) {
            if (!class_437.method_25442()) {
                this.nameEditor.method_1884(this.nameEditor.method_1881());
            }
            return class_8016.method_48194(this.nameEditor, this);
        }

        return super.method_48205(event);
    }
}
