package io.github.fishstiz.packed_packs.pack;

import com.google.common.util.concurrent.Runnables;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.config.Folder;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.transform.interfaces.FilePack;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionModelAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.FolderRepositorySourceAccessor;
import io.github.fishstiz.packed_packs.transform.mixin.folders.additional.PackRepositoryAccessor;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.lang.CollectionsUtil;
import io.github.fishstiz.packed_packs.util.lang.ObjectsUtil;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.apache.commons.lang3.function.Consumers;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_156;
import net.minecraft.class_3283;
import net.minecraft.class_3288;
import net.minecraft.class_5369;

public class PackRepositoryManager {
    private final Map<String, class_3288> availablePacks = new Object2ObjectLinkedOpenHashMap<>();
    private final Map<String, List<class_3288>> folderPacks = new Object2ObjectOpenHashMap<>();
    private final Map<String, CompletableFuture<Folder>> folderConfigs = new Object2ObjectOpenHashMap<>();
    private final Set<String> selectedPacksCache = new ObjectOpenHashSet<>();
    private final class_3283 repository;
    private final PackOptionsContext options;
    private final Path packDir;
    private class_5369 model;

    public PackRepositoryManager(class_3283 repository, PackOptionsContext options, Path packDir) {
        this.repository = repository;
        this.options = options;
        this.packDir = packDir;

        this.refreshModel();
        this.regenerateAvailablePacks();
    }

    public class_3283 getRepository() {
        return this.repository;
    }

    private List<class_3288> getSelectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getSelectedPacks();
    }

    private List<class_3288> getUnselectedPacks() {
        return ((PackSelectionModelAccessor) this.model).getUnselectedPacks();
    }

    private void refreshModel() {
        this.model = new class_5369(Runnables.doNothing(), PackAssetManager::getDefaultIcon, this.repository, Consumers.nop());
        ((PackSelectionModelAccessor) this.model).packed_packs$filterHidden(false);
    }

    public List<class_3288> getPacks() {
        return List.copyOf(this.availablePacks.values());
    }

    public PackGroup getPacksByRequirement() {
        List<class_3288> required = new ObjectArrayList<>();
        List<class_3288> optional = new ObjectArrayList<>();

        for (class_3288 pack : this.availablePacks.values()) {
            if (this.options.isRequired(pack)) {
                this.options.getPosition(pack).method_14468(required, pack, this.options::getSelectionConfig, true);
            } else {
                optional.add(pack);
            }
        }

        return PackGroup.of(required, optional);
    }

    public PackGroup getPacksBySelected() {
        return this.validateAndGroupPacks(this.getUnselectedPacks(), this.getSelectedPacks());
    }

    public void removePack(class_3288 pack) {
        if (pack instanceof FolderPack) {
            ObjectsUtil.ifPresent(this.folderPacks.get(pack.method_14463()), packs -> packs.forEach(this::removePack));
            this.folderPacks.remove(pack.method_14463());
            this.folderConfigs.remove(pack.method_14463());
        }

        this.repository.method_49428(pack.method_14463());
        this.availablePacks.remove(pack.method_14463());

        try {
            this.getSelectedPacks().remove(pack);
            this.getUnselectedPacks().remove(pack);
        } catch (UnsupportedOperationException e) {
            // just in case
            PackedPacks.LOGGER.warn("[packed_packs] Failed to mutate PackSelectionModel lists. Report this issue to mod author.");
        }
    }

    /**
     * @param unselected ungrouped list of unselected packs
     * @param selected   ungrouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validateAndGroupPacks(List<class_3288> unselected, List<class_3288> selected) {
        return this.validatePacks(this.groupByFolders(unselected), this.groupByFolders(selected));
    }

    private void addValidPacks(List<class_3288> source, Set<class_3288> seen, ObjectOpenHashSet<class_3288> validPacks, List<class_3288> target) {
        for (class_3288 pack : source) {
            this.options.validate(pack);
            class_3288 validPack = validPacks.get(pack); // metadata can change
            if (validPack != null && (seen.add(pack))) {
                target.add(validPack);
            }
        }
    }

    private void addValidPacks(
            List<class_3288> source,
            Set<class_3288> seen,
            ObjectOpenHashSet<class_3288> validPacks,
            List<class_3288> targetUnselected,
            List<class_3288> targetSelected
    ) {
        for (class_3288 pack : source) {
            this.options.validate(pack);
            class_3288 validPack = validPacks.get(pack);
            if (validPack != null && seen.add(pack)) {
                if (this.options.isRequired(validPack)) {
                    this.options.getPosition(validPack).method_14468(targetSelected, validPack, this.options::getSelectionConfig, true);
                } else {
                    targetUnselected.add(validPack);
                }
            }
        }
    }

    /**
     * @param unselected grouped list of unselected packs
     * @param selected   grouped list of selected packs
     * @return validated and grouped list of packs
     */
    public PackGroup validatePacks(List<class_3288> unselected, List<class_3288> selected) {
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(this.availablePacks.values());
        List<class_3288> validSelected = new ObjectArrayList<>(selected.size());
        List<class_3288> validUnselected = new ObjectArrayList<>(unselected.size());

        this.addValidPacks(selected, seen, validPacks, validSelected);
        this.addValidPacks(unselected, seen, validPacks, validUnselected, validSelected);

        for (class_3288 validPack : validPacks) {
            if (seen.add(validPack)) {
                this.options.validate(validPack);
                if (this.options.isRequired(validPack)) {
                    this.options.getPosition(validPack).method_14468(validSelected, validPack, this.options::getSelectionConfig, true);
                } else {
                    validUnselected.add(validPack);
                }
            }
        }
        return PackGroup.of(validSelected, validUnselected);
    }

    /**
     * @param folderPack   the folder pack
     * @param orderedPacks the nested packs that define the preferred order
     * @return a validated and ordered list of all packs under the folder pack
     */
    public List<class_3288> validateAndOrderNestedPacks(FolderPack folderPack, List<class_3288> orderedPacks) {
        List<class_3288> orderedValidPacks = this.folderPacks.get(folderPack.method_14463());
        ObjectOpenHashSet<class_3288> validPacks = new ObjectOpenHashSet<>(orderedValidPacks);
        Set<class_3288> seen = new ObjectOpenHashSet<>();
        List<class_3288> finalOrderedPacks = new ObjectArrayList<>();

        this.addValidPacks(orderedPacks, seen, validPacks, finalOrderedPacks);

        for (class_3288 validPack : orderedValidPacks) {
            if (seen.add(validPack)) {
                finalOrderedPacks.add(validPack);
            }
        }

        this.folderPacks.put(folderPack.method_14463(), finalOrderedPacks);
        return finalOrderedPacks;
    }

    /**
     * @param folderPack   the folder pack
     * @param orderedPacks the nested pack ids that define the preferred order
     * @return a validated and ordered list of all packs under the folder pack
     */
    public List<class_3288> validateAndOrderNestedPackIds(FolderPack folderPack, List<String> orderedPacks) {
        return this.validateAndOrderNestedPacks(folderPack, this.getPacksById(orderedPacks, this.folderPacks.get(folderPack.method_14463())));
    }

    /**
     * @param packIds grouped pack ids
     * @param source  grouped packs
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds, Map<String, class_3288> source) {
        return CollectionsUtil.lookup(packIds, source);
    }

    /**
     * @param packIds grouped pack ids
     * @param source  grouped packs
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds, Collection<class_3288> source) {
        return CollectionsUtil.lookup(packIds, CollectionsUtil.toMap(source, class_3288::method_14463));
    }

    /**
     * @param packIds grouped pack ids
     * @return grouped packs by id
     */
    public List<class_3288> getPacksById(Collection<String> packIds) {
        return this.getPacksById(packIds, this.availablePacks);
    }

    public List<class_3288> getPacksByFlattenedIds(Collection<String> packIds) {
        List<class_3288> folderPacks = CollectionsUtil.filter(this.availablePacks.values(), FolderPack.class::isInstance, ObjectArrayList::new);
        List<class_3288> available = CollectionsUtil.addAll(folderPacks, this.repository.method_14441());
        return this.groupByFolders(this.getPacksById(packIds, available));
    }

    /**
     * @param packs ungrouped collection of packs
     */
    private void populateAvailablePacks(Collection<class_3288> packs) {
        for (class_3288 pack : packs) {
            FilePack filePack = (FilePack) pack;
            if (filePack.packed_packs$nestedPack()) {
                Path folderPath = Objects.requireNonNull(filePack.packed_packs$getPath()).getParent();
                String folderName = PackUtil.generatePackName(folderPath);
                String folderId = PackUtil.generatePackId(folderName);
                if (!this.availablePacks.containsKey(folderId)) {
                    FolderPack folderPack = new FolderPack(folderId, folderName, this.folderPacks::get, folderPath);
                    this.folderConfigs.put(folderId, folderPack.loadConfig());
                    this.availablePacks.put(folderId, folderPack);
                }
                this.folderPacks.computeIfAbsent(folderId, id -> new ObjectArrayList<>()).add(pack);
            } else {
                this.availablePacks.put(pack.method_14463(), pack);
            }
        }
    }

    private void regenerateAvailablePacks() {
        this.selectedPacksCache.clear();
        this.availablePacks.clear();
        this.folderPacks.clear();
        this.folderConfigs.clear();
        this.selectedPacksCache.addAll(this.repository.method_29210());
        this.populateAvailablePacks(this.getSelectedPacks());
        this.populateAvailablePacks(this.getUnselectedPacks());
    }

    public void refresh() {
        ((PackSelectionModelAccessor) this.model).packed_packs$reset();
        this.model.method_29981();
        this.regenerateAvailablePacks();
    }

    /**
     * @param selected grouped list of selected packs
     */
    public void selectPacks(List<class_3288> selected) {
        this.repository.method_14447(PackUtil.extractPackIds(PackUtil.flattenPacks(selected)).reversed());

        this.refreshModel();
        this.selectedPacksCache.clear();
        this.selectedPacksCache.addAll(this.repository.method_29210());
    }

    /**
     * @param flatPacks ungrouped list of packs
     * @return grouped list of packs
     */
    public List<class_3288> groupByFolders(List<class_3288> flatPacks) {
        if (this.folderPacks.isEmpty()) return flatPacks;

        Map<String, String> packToFolder = new Object2ObjectOpenHashMap<>();
        for (Map.Entry<String, List<class_3288>> entry : this.folderPacks.entrySet()) {
            for (class_3288 pack : entry.getValue()) {
                packToFolder.put(pack.method_14463(), entry.getKey());
            }
        }

        Set<String> seenFolders = new ObjectOpenHashSet<>();
        List<class_3288> grouped = new ObjectArrayList<>(flatPacks.size());

        for (class_3288 pack : flatPacks) {
            String folderId = packToFolder.get(pack.method_14463());
            if (folderId != null) {
                if (seenFolders.add(folderId)) {
                    grouped.add(this.availablePacks.get(folderId));
                }
            } else {
                grouped.add(pack);
            }
        }
        return grouped;
    }

    public void openDir() {
        class_156.method_668().method_60932(this.packDir);
    }

    public boolean isEnabled(class_3288 pack) {
        if (pack instanceof FolderPack folderPack) {
            for (class_3288 nestedPack : this.folderPacks.get(folderPack.method_14463())) {
                if (this.selectedPacksCache.contains(nestedPack.method_14463())) {
                    return true;
                }
            }
            return false;
        }

        return this.selectedPacksCache.contains(pack.method_14463());
    }

    public Path getBaseDir() {
        return this.packDir;
    }

    public List<Path> getAdditionalDirs() {
        Path normalizedBaseDir = this.getBaseDir().toAbsolutePath().normalize();

        return ((PackRepositoryAccessor) this.repository).packed_packs$getSources().stream()
                .filter(FolderRepositorySourceAccessor.class::isInstance)
                .map(source -> ((FolderRepositorySourceAccessor) source).packed_packs$getFolder())
                .map(path -> path.toAbsolutePath().normalize())
                .filter(path -> !path.equals(normalizedBaseDir))
                .distinct()
                .toList();
    }

    public @Nullable Folder getFolderConfig(@Nullable FolderPack folderPack) {
        if (folderPack == null) return null;
        CompletableFuture<Folder> future = this.folderConfigs.get(folderPack.method_14463());
        return future != null ? future.join() : null;
    }

    public List<class_3288> getNestedPacks(FolderPack folderPack) {
        Folder config = this.getFolderConfig(folderPack);
        if (config == null) return Collections.emptyList();
        return this.validateAndOrderNestedPackIds(folderPack, config.getPackIds());
    }
}
