package io.github.fishstiz.packed_packs.pack;

import com.google.common.hash.Hashing;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.config.Config;
import io.github.fishstiz.packed_packs.config.Folder;
import io.github.fishstiz.packed_packs.config.PackOptions;
import io.github.fishstiz.packed_packs.config.Profile;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.transform.interfaces.IPack;
import io.github.fishstiz.packed_packs.util.PackUtil;
import io.github.fishstiz.packed_packs.util.ResourceUtil;
import io.github.fishstiz.packed_packs.util.ToastUtil;
import org.jetbrains.annotations.Nullable;

import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_1060;
import net.minecraft.class_124;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3262;
import net.minecraft.class_3288;
import net.minecraft.class_5352;
import net.minecraft.class_5375;
import net.minecraft.class_7367;

public interface PackAssets extends PackOptions {
    String ICON_FILENAME = "pack.png";
    String ZIP_PACK_EXTENSION = ".zip";
    class_2960 DEFAULT_FOLDER_ICON = ResourceUtil.getResource("textures/misc/unknown_folder.png");
    class_2960 DEFAULT_ICON = class_2960.method_60656("textures/misc/unknown_pack.png");
    class_2561 OPEN_FILE_TEXT = ResourceUtil.getText("file.open");
    class_2561 OPEN_PARENT_TEXT = ResourceUtil.getText("file.parent.open");
    class_2561 RENAME_FILE_TEXT = ResourceUtil.getText("file.rename");
    class_2561 DELETE_FILE_TEXT = ResourceUtil.getText("file.delete");
    class_5352 SOURCE = class_5352.method_45281(name -> class_2561.method_43469("pack.nameAndSource", name, ResourceUtil.getModName().method_27692(class_124.field_1054))
            .method_27692(class_124.field_1080), false);

    void getOrLoadIcon(class_3288 pack, Consumer<class_2960> iconCallback);

    boolean isResourcePacks();

    boolean isLocked();

    boolean isEnabled(class_3288 pack);

    Config.Packs getConfig();

    Folder getFolderConfig(@Nullable FolderPack folderPack);

    @Nullable Profile getProfile();

    List<class_3288> flattenPacks(List<class_3288> grouped);

    default boolean deletePack(class_3288 pack) {
        if (pack == null || this.isEnabled(pack)) {
            return false;
        }

        Path path = validatePackPath(pack);
        if (path == null) {
            showFailToast(getDeleteFailText(pack.method_14457().getString()));
            return false;
        }

        if (!PackUtil.deletePath(path)) {
            showFailToast(getDeleteFailText(PackUtil.fileName(path)));
            return false;
        }

        return true;
    }

    default boolean renamePack(class_3288 pack, String newName) {
        if (pack == null || this.isEnabled(pack)) {
            return false;
        }

        Path path = validatePackPath(pack);
        if (path == null) {
            showFailToast(getRenameFailText(pack.method_14457().getString(), newName));
            return false;
        }

        Path newPath = path.getParent().resolve(newName);
        if (!PackUtil.renamePath(path, newPath)) {
            showFailToast(getRenameFailText(PackUtil.fileName(path), PackUtil.fileName(newPath)));
            return false;
        }

        return true;
    }

    static @Nullable Path validatePackPath(class_3288 pack) {
        if (pack == null) {
            return null;
        }
        Path path = ((IPack) pack).packed_packs$getPath();
        if (path == null) {
            return null;
        }

        try {
            return Files.exists(path) ? path : null;
        } catch (SecurityException e) {
            PackedPacks.LOGGER.error("[packed_packs] Could not read file: '{}'", path);
            return null;
        }
    }

    static boolean isZipPack(class_3288 pack) {
        Path path = PackAssets.validatePackPath(pack);
        return path != null && Files.isRegularFile(path) && PackUtil.fileName(path).endsWith(ZIP_PACK_EXTENSION);
    }

    private static void showFailToast(class_2561 message) {
        ToastUtil.onFileFailToast(message);
    }

    static class_2960 getDefaultIcon(class_3288 pack) {
        return pack instanceof FolderPack ? DEFAULT_FOLDER_ICON : DEFAULT_ICON;
    }

    static class_2561 getRenameFailText(String from, String to) {
        return ResourceUtil.getText("file.rename.fail", from, to);
    }

    static class_2561 getDeleteFailText(String fileName) {
        return ResourceUtil.getText("file.delete.fail", fileName);
    }

    /**
     * Copied from {@link class_5375#method_30289(class_1060, class_3288)}
     */
    @SuppressWarnings("all")
    static CompletableFuture<class_2960> loadPackIcon(class_3288 pack) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                class_2960 packIcon;
                try (class_3262 packResources = pack.method_14458()) {
                    class_7367<InputStream> iconIoSupplier = packResources.method_14410(ICON_FILENAME);

                    if (iconIoSupplier == null) {
                        return getDefaultIcon(pack);
                    }

                    String id = pack.method_14463();
                    class_2960 resourceLocation = class_2960.method_60656(
                            "pack/" + class_156.method_30309(id, class_2960::method_29184) + "/" + Hashing.sha1().hashUnencodedChars(id) + "/icon"
                    );
                    InputStream iconStream = iconIoSupplier.get();

                    try {
                        class_1011 nativeImage = class_1011.method_4309(iconStream);
                        class_1060 manager = class_310.method_1551().method_1531();
                        class_310.method_1551().execute(() -> manager.method_4616(resourceLocation, new class_1043(resourceLocation::toString, nativeImage)));
                        packIcon = resourceLocation;
                    } catch (Throwable e) {
                        if (iconStream != null) {
                            try {
                                iconStream.close();
                            } catch (Throwable e2) {
                                e.addSuppressed(e2);
                            }
                        }
                        throw e;
                    }
                    if (iconStream != null) {
                        iconStream.close();
                    }
                }
                return packIcon;
            } catch (Exception e) {
                if (!(e instanceof NoSuchFileException)) {
                    PackedPacks.LOGGER.warn("Failed to load icon from pack '{}'", pack.method_14463(), e);
                }
                return getDefaultIcon(pack);
            }
        }, class_156.method_18349());
    }
}
