package io.github.fishstiz.packed_packs.util;

import com.sun.jna.platform.FileUtils;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.pack.folder.FolderResources;
import io.github.fishstiz.packed_packs.transform.interfaces.IPack;
import io.github.fishstiz.packed_packs.transform.mixin.UtilAccess;
import io.github.fishstiz.packed_packs.util.lang.CollectionsUtil;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import net.fabricmc.fabric.impl.resource.loader.BuiltinModResourcePackSource;
import net.minecraft.class_156;
import net.minecraft.class_310;
import net.minecraft.class_3262;
import net.minecraft.class_3288;
import net.minecraft.class_5352;
import net.minecraft.class_8581;
import net.minecraft.class_8621;
import net.minecraft.class_9224;
import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.util.*;

public class PackUtil {
    public static final String HIGH_CONTRAST_ID = "high_contrast";
    public static final String VANILLA_ID = "vanilla";
    public static final String FABRIC_ID = "fabric";

    // Changing these fields would be breaking changes
    private static final String FILE_PREFIX = "file/";
    private static final String DELIMITER = "/";

    private PackUtil() {
    }

    public static String fileName(Path path) {
        return path.getFileName().toString();
    }

    public static String generatePackName(Path path) {
        return fileName(path);
    }

    public static String generatePackId(String name) {
        return FILE_PREFIX + name;
    }

    public static String generatePackId(Path path) {
        return generatePackId(generatePackName(path));
    }

    public static String generateNestedPackId(Path path) {
        return FILE_PREFIX + generatePackName(path.getParent()) + DELIMITER + generatePackName(path);
    }

    public static class_9224 replicateLocationInfo(class_9224 info, String id) {
        return new class_9224(id, info.comp_2330(), info.comp_2331(), info.comp_2332());
    }

    public static long getLastUpdatedEpochMs(class_3288 pack) {
        Path path = ((IPack) pack).packed_packs$getPath();
        if (path == null) {
            return -1;
        }

        try {
            return Files.getLastModifiedTime(path).toInstant().toEpochMilli();
        } catch (IOException e) {
            PackedPacks.LOGGER.error("Failed to get age of pack '{}'", pack.method_14463());
            return -1;
        }
    }

    public static List<String> extractPackIds(Collection<class_3288> packs) {
        return CollectionsUtil.extractNonNull(packs, class_3288::method_14463);
    }

    public static String joinPackNames(Collection<Path> paths) {
        return String.join(", ", CollectionsUtil.extractNonNull(paths, PackUtil::generatePackName));
    }

    public static boolean hasMcmeta(Path path) {
        return Files.isRegularFile(path.resolve(class_3262.field_29781), LinkOption.NOFOLLOW_LINKS);
    }

    public static boolean hasFolderConfig(Path path) {
        return Files.isRegularFile(path.resolve(FolderResources.FOLDER_CONFIG_FILENAME), LinkOption.NOFOLLOW_LINKS);
    }

    public static boolean isBuiltIn(class_3288 pack) {
        class_5352 packSource = pack.method_29483();
        //noinspection UnstableApiUsage
        return packSource == class_5352.field_25348 || packSource instanceof BuiltinModResourcePackSource;
    }

    public static boolean isEssential(class_3288 pack) {
        return pack.method_14463().equals(VANILLA_ID) || pack.method_14463().equals(FABRIC_ID);
    }

    public static boolean isFeature(class_3288 pack) {
        return pack.method_29483() == class_5352.field_40048;
    }

    public static boolean isNonPackDirectory(Path path) {
        return Files.isDirectory(path, LinkOption.NOFOLLOW_LINKS) && !hasMcmeta(path);
    }

    public static List<Path> mapValidDirectories(List<String> paths) {
        if (paths == null || paths.isEmpty()) return Collections.emptyList();

        return CollectionsUtil.extractNonNull(paths, path -> {
            try {
                Path resolved = Paths.get(path);
                if (Files.exists(resolved, LinkOption.NOFOLLOW_LINKS) && Files.isDirectory(resolved, LinkOption.NOFOLLOW_LINKS)) {
                    return resolved.toAbsolutePath().normalize();
                } else {
                    PackedPacks.LOGGER.error("[packed_packs] Path is not a valid directory: '{}', ignoring.", path);
                }
            } catch (Exception e) {
                PackedPacks.LOGGER.error("[packed_packs] Failed to resolve path: '{}', ignoring.", path, e);
            }
            return null;
        });
    }

    public static void openPack(class_3288 pack) {
        var path = ((IPack) pack).packed_packs$getPath();
        if (path != null) {
            class_156.method_668().method_60932(path);
        }
    }

    public static void openParent(class_3288 pack) {
        var path = ((IPack) pack).packed_packs$getPath();
        if (path != null) {
            PackUtil.openParent(path);
        }
    }

    public static void openParent(Path path) {
        File file = path.toFile();
        if (!file.exists()) return;

        try {
            switch (class_156.method_668()) {
                case field_1133 -> new ProcessBuilder("explorer.exe", "/select,", file.getAbsolutePath()).start();
                case field_1137 -> new ProcessBuilder("open", "-R", file.getAbsolutePath()).start();
                case field_1135 -> {
                    File parentFile = file.getParentFile();
                    if (parentFile != null) new ProcessBuilder("xdg-open", parentFile.getAbsolutePath()).start();
                }
                default -> {
                    Path parent = path.getParent();
                    if (parent != null) class_156.method_668().method_60932(parent);
                }
            }
        } catch (IOException e) {
            Path parent = path.getParent();
            if (parent != null) class_156.method_668().method_60932(parent);
        }
    }

    public static boolean deletePath(Path path) {
        FileUtils fileUtils = FileUtils.getInstance();

        if (fileUtils.hasTrash()) {
            try {
                fileUtils.moveToTrash(path.toFile());
                return true;
            } catch (IOException e) {
                PackedPacks.LOGGER.warn("[packed_packs] Failed to move to trash: '{}'", path, e);
            }
        }

        if (Files.isDirectory(path)) {
            try {
                org.apache.commons.io.FileUtils.deleteDirectory(path.toFile());
                return true;
            } catch (IOException e) {
                PackedPacks.LOGGER.error("[packed_packs] Failed to delete path: '{}'", path, e);
                return false;
            }
        }

        return UtilAccess.packed_packs$createDeleter(path).getAsBoolean();
    }

    public static boolean renamePath(Path path, Path newName) {
        return UtilAccess.packed_packs$createRenamer(path, newName).getAsBoolean();
    }

    public static PathValidationResults validatePaths(List<Path> packs) {
        class_8621<Path> packDetector = new class_8621<>(class_310.method_1551().method_52702()) {
            @Override
            protected Path method_52439(Path path) {
                return path;
            }

            @Override
            protected Path method_52438(Path path) {
                return path;
            }
        };

        PathValidationResults results = new PathValidationResults(packs);
        for (Path path : packs) {
            try {
                if (!isNonPackDirectory(path)) {
                    if (validatePath(path, packDetector, results.symlinkWarnings)) {
                        results.addValid(path);
                    }
                    continue;
                }

                try (DirectoryStream<Path> paths = Files.newDirectoryStream(path)) {
                    for (Path child : paths) {
                        if (validatePath(child, packDetector, results.symlinkWarnings)) {
                            results.addValid(path);
                            break;
                        }
                    }
                }
            } catch (IOException e) {
                PackedPacks.LOGGER.warn("Failed to check {} for packs", path, e);
            }
        }

        return results;
    }

    private static boolean validatePath(Path path, class_8621<Path> packDetector, List<class_8581> symlinkWarnings) throws IOException {
        Path detectedPack = packDetector.method_52441(path, symlinkWarnings);
        if (detectedPack == null) {
            PackedPacks.LOGGER.warn("Path {} does not seem like pack", path);
            return false;
        }
        return true;
    }

    public record PathValidationResults(
            List<Path> valid,
            Set<Path> rejected,
            List<class_8581> symlinkWarnings
    ) {
        private PathValidationResults(Collection<Path> packs) {
            this(new ArrayList<>(packs.size()), new ObjectOpenHashSet<>(packs), new ArrayList<>());
        }

        private void addValid(Path path) {
            this.valid.add(path);
            this.rejected.remove(path);
        }
    }
}
