package io.github.fishstiz.packed_packs.gui.components.pack;

import io.github.fishstiz.fidgetz.gui.components.CyclicButton;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.ButtonSprites;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.Sprite;
import io.github.fishstiz.fidgetz.gui.shapes.Size;
import io.github.fishstiz.packed_packs.util.ResourceUtil;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.util.PackUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import net.minecraft.class_2561;
import net.minecraft.class_3288;
import net.minecraft.class_7919;

public class Query {
    private boolean hideIncompatible = false;
    private SortOption sort;
    private String search;

    Query() {
    }

    Query(boolean hideIncompatible, SortOption sort, String search) {
        this.hideIncompatible = hideIncompatible;
        this.sort = sort;
        this.search = search;
    }

    boolean setHideIncompatible(boolean hideIncompatible) {
        boolean updated = this.hideIncompatible != hideIncompatible;
        this.hideIncompatible = hideIncompatible;
        return updated;
    }

    boolean setSort(SortOption sort) {
        boolean updated = !Objects.equals(this.sort, sort);
        this.sort = sort;
        return updated;
    }

    boolean setSearch(String search) {
        boolean updated = !Objects.equals(this.search, search);
        this.search = search;
        return updated;
    }

    boolean update(boolean incompatibleHidden, SortOption sort, String search) {
        boolean updated = this.setHideIncompatible(incompatibleHidden);
        updated |= this.setSort(sort);
        updated |= this.setSearch(search);
        return updated;
    }

    boolean update(Query query) {
        return this.update(query.hideIncompatible, query.sort, query.search);
    }

    void apply(final List<class_3288> packs) {
        Objects.requireNonNull(packs);

        if (this.hideIncompatible) {
            packs.removeIf(pack -> !pack.method_14460().method_14437());
        }
        if (this.search != null && !this.search.isEmpty()) {
            packs.removeIf(pack -> !normalizeTitle(pack.method_14457().getString()).toLowerCase().contains(this.search.toLowerCase()));
        }
        if (this.sort != null) {
            packs.sort(this.sort.comparator);
        }
    }

    boolean isQuerying() {
        return this.hideIncompatible || (this.search != null && !this.search.isEmpty()) || this.sort != null;
    }

    public Query copy() {
        return new Query(this.hideIncompatible, this.sort, this.search);
    }

    public boolean isHideIncompatible() {
        return this.hideIncompatible;
    }

    public SortOption getSort() {
        return this.sort;
    }

    public String getSearch() {
        return this.search;
    }

    public enum SortOption implements CyclicButton.SpriteOption {
        VANILLA("sort.vanilla", "sort_vanilla", (first, second) -> {
            boolean builtInFirst = PackUtil.isBuiltIn(first);
            boolean builtInSecond = PackUtil.isBuiltIn(second);
            if (builtInFirst != builtInSecond) return builtInFirst ? 1 : -1;

            boolean featureFirst = PackUtil.isFeature(first);
            boolean featureSecond = PackUtil.isFeature(second);
            if (featureFirst != featureSecond) return featureFirst ? 1 : -1;

            return first.method_14457().getString().compareTo(second.method_14457().getString());
        }),
        A_Z("sort.a_z", "sort_a_z", Comparator.comparing(
                pack -> normalizeTitle(pack.method_14457().getString()),
                String.CASE_INSENSITIVE_ORDER
        )),
        Z_A("sort.z_a", "sort_z_a", A_Z.comparator.reversed()),
        RECENT("sort.recent", "sort_recent", Comparator.comparingLong(PackUtil::getLastUpdatedEpochMs).reversed()),
        OLDEST("sort.oldest", "sort_oldest", RECENT.comparator.reversed());

        private final class_2561 component;
        private final class_7919 tooltip;
        private final ButtonSprites sprites;
        private final Comparator<class_3288> comparator;

        SortOption(String key, String icon, Comparator<class_3288> comparator) {
            this.component = ResourceUtil.getText(key);
            this.tooltip = class_7919.method_47407(this.component);
            this.sprites = ButtonSprites.of(new Sprite(ResourceUtil.getIcon(icon), Size.of16()));
            this.comparator = folderFirst(comparator);
        }

        @Override
        public @NotNull class_2561 text() {
            return this.component;
        }

        @Override
        public @Nullable class_7919 tooltip() {
            return this.tooltip;
        }

        @Override
        public @Nullable ButtonSprites sprites() {
            return this.sprites;
        }

        static Comparator<class_3288> folderFirst(Comparator<class_3288> base) {
            return Comparator.comparing((class_3288 pack) -> !(pack instanceof FolderPack)).thenComparing(base);
        }
    }

    private static String normalizeTitle(String title) {
        return title
                .replaceAll("§.", "") // remove formatting
                .trim();
    }
}
