package io.github.fishstiz.packed_packs.gui.screens;

import com.google.common.collect.ImmutableList;
import io.github.fishstiz.fidgetz.gui.components.*;
import io.github.fishstiz.fidgetz.gui.components.contextmenu.*;
import io.github.fishstiz.fidgetz.gui.layouts.FlexLayout;
import io.github.fishstiz.fidgetz.gui.renderables.ColoredRect;
import io.github.fishstiz.fidgetz.gui.renderables.sprites.Sprite;
import io.github.fishstiz.packed_packs.PackedPacks;
import io.github.fishstiz.packed_packs.compat.ModAdditions;
import io.github.fishstiz.packed_packs.config.*;
import io.github.fishstiz.packed_packs.gui.components.contextmenu.DirectoryMenuItem;
import io.github.fishstiz.packed_packs.gui.components.contextmenu.PackMenuHeader;
import io.github.fishstiz.packed_packs.gui.components.pack.*;
import io.github.fishstiz.packed_packs.gui.layouts.pack.AvailablePacksLayout;
import io.github.fishstiz.packed_packs.gui.layouts.pack.CurrentPacksLayout;
import io.github.fishstiz.packed_packs.gui.layouts.pack.PackAliasLayout;
import io.github.fishstiz.packed_packs.gui.layouts.pack.PackLayout;
import io.github.fishstiz.packed_packs.gui.components.ToggleableHelper;
import io.github.fishstiz.packed_packs.pack.*;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionModelAccessor;
import io.github.fishstiz.packed_packs.util.AsyncUtil;
import io.github.fishstiz.packed_packs.util.PackUtil.PathValidationResults;
import io.github.fishstiz.packed_packs.util.ToastUtil;
import io.github.fishstiz.packed_packs.util.constants.Theme;
import io.github.fishstiz.packed_packs.pack.folder.FolderPack;
import io.github.fishstiz.packed_packs.gui.layouts.*;
import io.github.fishstiz.packed_packs.gui.components.events.*;
import io.github.fishstiz.packed_packs.gui.history.HistoryManager;
import io.github.fishstiz.packed_packs.gui.history.Restorable;
import io.github.fishstiz.packed_packs.gui.metadata.PackSelectionScreenArgs;
import io.github.fishstiz.packed_packs.transform.mixin.PackSelectionScreenAccessor;
import io.github.fishstiz.packed_packs.util.ResourceUtil;
import io.github.fishstiz.fidgetz.util.lang.CollectionsUtil;
import io.github.fishstiz.fidgetz.util.lang.ObjectsUtil;
import it.unimi.dsi.fastutil.booleans.BooleanConsumer;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_3264;
import net.minecraft.class_3288;
import net.minecraft.class_332;
import net.minecraft.class_364;
import net.minecraft.class_403;
import net.minecraft.class_410;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import net.minecraft.class_5375;
import net.minecraft.class_7919;
import net.minecraft.class_8016;
import net.minecraft.class_8669;

import static net.minecraft.class_3675.field_31986;
import static net.minecraft.class_3675.field_31947;
import static io.github.fishstiz.packed_packs.util.InputUtil.*;
import static io.github.fishstiz.packed_packs.util.PackUtil.*;
import static io.github.fishstiz.packed_packs.util.constants.GuiConstants.*;

public class PackedPacksScreen extends PackListEventHandler implements
        HoverStateHandler,
        ToggleableDialogContainer,
        ContextMenuContainer,
        Restorable<PackedPacksScreen.Snapshot> {
    private static final class_2561 OPEN_FOLDER_TEXT = class_2561.method_43471("pack.openFolder");
    private final class_437 previous;
    private final PackSelectionScreenArgs original;
    private final HistoryManager<io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot> history;
    private final LayoutWrapper<FlexLayout> layout;
    private final ProfilesLayout profiles;
    private final PackOptionsContext options;
    private final PackRepositoryManager repository;
    private final AvailablePacksLayout availablePacks;
    private final CurrentPacksLayout currentPacks;
    private final FolderDialog folderDialog;
    private final List<PackList> packLists;
    private final FileRenameModal fileRenameModal;
    private final ContextMenu contextMenu;
    private final Modal<OptionsLayout> optionsModal;
    private final List<ToggleableDialog<?>> dialogs;
    private Modal<PackAliasLayout> aliasModal;
    private List<Path> additionalFolders;
    private CompletableFuture<Void> refreshFuture;
    private CompletableFuture<Void> watcherFuture;
    private PackWatcher watcher;
    private boolean showActionBar = Config.get().isShowActionBar();
    private @Nullable class_364 hoveredElement;
    private boolean initialized = false;

    private PackedPacksScreen(class_310 minecraft, class_437 previous, PackSelectionScreenArgs original, boolean initState) {
        super(minecraft, ResourceUtil.getModName());

        this.previous = previous;
        this.original = original;
        Config.Packs userConfig = Config.get().get(original.packType());
        DevConfig.Packs config = DevConfig.get().get(original.packType());

        this.history = new HistoryManager<>();
        this.layout = new LayoutWrapper<>(FlexLayout.vertical(this::getMaxHeight).spacing(SPACING));
        this.layout.setPadding(SPACING);

        this.profiles = new ProfilesLayout(this, userConfig, config, this::onProfileChange, this::onProfileCopy);
        this.options = new PackOptionsContext(this.profiles::getProfile, userConfig, config);
        this.repository = new PackRepositoryManager(this.original.repository(), this.options, this.original.packDir());

        WidgetFactory.PackedPacksWidgets widgets = WidgetFactory.createWidgets(this, this.options, this.repository, this.assetManager);
        this.availablePacks = widgets.availablePacksLayout();
        this.currentPacks = widgets.currentPacksLayout();
        this.folderDialog = widgets.folderDialog();
        this.packLists = widgets.packLists();
        this.fileRenameModal = widgets.fileRenameModal();
        this.contextMenu = widgets.contextMenu();
        this.optionsModal = Modal.builder(this, new OptionsLayout(this.field_22787, this.layout::method_25364, userConfig))
                .setBackdrop(new ColoredRect(Theme.BLACK.withAlpha(0.5f)))
                .setCaptureFocus(true)
                .padding(SPACING)
                .build();

        if (Config.get().isDevMode()) {
            PackAliasLayout packAliasLayout = new PackAliasLayout(config, this.assetManager);
            this.aliasModal = Modal.builder(this, packAliasLayout)
                    .addListener(open -> {
                        if (!open) this.aliasModal.root().layout().saveAliases();
                    })
                    .padding(SPACING)
                    .build();
            this.dialogs = List.of(this.optionsModal, this.contextMenu, this.aliasModal, this.fileRenameModal, this.profiles.getSidebar(), this.folderDialog);
        } else {
            this.dialogs = List.of(this.optionsModal, this.contextMenu, this.fileRenameModal, this.profiles.getSidebar(), this.folderDialog);
        }

        for (int i = 0; i < this.dialogs.size(); i++) {
            this.dialogs.get(i).setZ((this.dialogs.size() - i));
        }

        this.initAdditionalFolders();
        if (initState) {
            Profile lastViewed = userConfig.getLastViewedProfile();
            Profile defaultProfile = config.getDefaultProfile();
            if (Objects.equals(lastViewed, defaultProfile)) {
                lastViewed = defaultProfile;
            }
            this.profiles.setProfile(lastViewed);
        }
    }

    public PackedPacksScreen(class_310 minecraft, class_437 previous, PackSelectionScreenArgs original) {
        this(minecraft, previous, original, true);
    }

    public PackedPacksScreen(class_310 minecraft, class_437 previous, PackSelectionScreenArgs original, Profile profile) {
        this(minecraft, previous, original, false);
        this.profiles.setProfile(profile);
    }

    public PackedPacksScreen(class_310 minecraft, class_437 previous, PackSelectionScreenArgs original, PackGroup packs) {
        this(minecraft, previous, original, false);
        this.applyPacks(packs.unselected(), packs.selected());
    }

    @Override
    public void method_49589() {
        if (this.initialized) {
            this.refreshPacks();
            this.initAdditionalFolders();
            this.createWatcher();
        }
    }

    @Override
    public void method_25432() {
        this.closeWatcher();

        this.availablePacks.saveFilters();

        Profile profile = this.profiles.getProfile();
        this.syncProfile(profile);
        this.options.getUserConfig().setLastViewedProfile(profile);

        List<Profile> profiles = this.options.getUserConfig().getProfiles();
        this.options.getUserConfig().setProfileOrder(profiles);

        Runnable profileSaver = profile != null ? () -> Profiles.save(this.original.packType(), profile) : ObjectsUtil::nop;
        AsyncUtil.submitAndWait(
                class_156.method_18349(),
                profileSaver,
                Config.get()::save,
                DevConfig.get()::save,
                Preferences.INSTANCE::save
        );
    }

    @Override
    protected void method_25426() {
        if (this.initialized) return;

        this.profiles.init(this::method_56131);

        this.layout.layout().addChild(this.createHeader());
        this.layout.layout().addFlexChild(this.createContents());
        this.layout.layout().addChild(this.createFooter());

        this.dialogs.forEach(this::method_25429);
        this.layout.method_48206(this::method_37063);
        CollectionsUtil.forEachReverse(this.dialogs, this::method_37060);

        this.clearHistory();
        this.method_48640();

        this.refreshPacks();
        this.createWatcher();

        this.initialized = true;
    }

    private FlexLayout createHeader() {
        FlexLayout header = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        final boolean devMode = Config.get().isDevMode();

        header.addChild(
                FidgetzButton.builder()
                        .makeSquare()
                        .setMessage(ProfilesLayout.TITLE_TEXT)
                        .setTooltip(class_7919.method_47407(ProfilesLayout.TITLE_TEXT))
                        .setSprite(HAMBURGER_SPRITE)
                        .setOnPress(this.profiles.getSidebar()::toggle)
                        .build()
        );

        if (devMode || Preferences.INSTANCE.actionBarWidget.get()) {
            header.addChild(
                    ToggleableHelper.applyPref(Preferences.INSTANCE.actionBarWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setTooltip(class_7919.method_47407(ResourceUtil.getText("toggle_actionbar.info")))
                            .setSprite(Sprite.of16(ResourceUtil.getIcon("filter")))
                            .setOnPress(this::toggleActionBar)
                            .build()
            );
        }
        header.addChild(this.profiles.getToggleNameButton());
        header.addFlexChild(this.profiles.getNameField());

        class_5375 originalScreen = this.previous instanceof class_5375 s ? s : this.original.createDummy();
        ModAdditions.onCreateHeader(this.original.packType(), header, this, originalScreen);

        if (devMode || Preferences.INSTANCE.optionsWidget.get()) {
            header.addChild(
                    ToggleableHelper.applyPref(Preferences.INSTANCE.optionsWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setMessage(OPTIONS_TEXT)
                            .setTooltip(class_7919.method_47407(OPTIONS_TEXT.method_27661().method_10852(class_5244.field_39678)))
                            .setSprite(Sprite.of16(ResourceUtil.getIcon("gear")))
                            .setOnPress(this.optionsModal::toggle)
                            .build()
            );
        }
        if (devMode || Preferences.INSTANCE.originalScreenWidget.get()) {
            header.addChild(
                    ToggleableHelper.applyPref(Preferences.INSTANCE.originalScreenWidget, FidgetzButton.<Void>builder())
                            .makeSquare()
                            .setTooltip(class_7919.method_47407(ResourceUtil.getText("original_screen.info").method_10852(class_5244.field_39678)))
                            .setSprite(Sprite.of16(ResourceUtil.getIcon("exit")))
                            .setOnPress(this::setOriginalScreen)
                            .build()
            );
        }
        return header;
    }

    private FlexLayout createContents() {
        FlexLayout contents = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        FlexLayout packLayout = FlexLayout.vertical().spacing(SPACING);
        this.availablePacks.init(contents.addFlexChild(packLayout, true));
        this.currentPacks.init(contents.addFlexChild(packLayout.copyLayout(), true));
        this.currentPacks.getSearchField().addListener(this::recordState);
        this.availablePacks.getSearchField().addListener(this::recordState);
        return contents;
    }

    private FlexLayout createFooter() {
        final FlexLayout footer = FlexLayout.horizontal(this::getMaxWidth).spacing(SPACING);
        FlexLayout firstColumn = FlexLayout.horizontal().spacing(SPACING);
        FlexLayout secondColumn = firstColumn.copyLayout();

        firstColumn.addFlexChild(
                FidgetzButton.builder()
                        .setMessage(OPEN_FOLDER_TEXT)
                        .setTooltip(class_7919.method_47407(class_2561.method_43471("pack.folderInfo")))
                        .setOnPress(this.repository::openDir)
                        .build()
        );

        if (this.original.packType() == class_3264.field_14188) {
            secondColumn.addFlexChild(FidgetzButton.builder().setMessage(ResourceUtil.getText("apply")).setOnPress(this::commit).build());
        }

        secondColumn.addFlexChild(FidgetzButton.builder().setMessage(class_5244.field_24334).setOnPress(this::method_25419).build());

        footer.addFlexChild(firstColumn);
        footer.addFlexChild(secondColumn);
        return footer;
    }

    public int getMaxHeight() {
        return this.field_22790 - SPACING * 2;
    }

    public int getMaxWidth() {
        return this.field_22789 - SPACING * 2;
    }

    @Override
    protected void method_41843() {
        if (this.field_22787 == null) return;

        PackedPacksScreen screen;
        Profile profile = this.profiles.getProfile();

        if (profile != null) {
            profile.setPacks(this.currentPacks.list().copyPacks());
            screen = new PackedPacksScreen(this.field_22787, this.previous, this.original, profile);
        } else {
            PackGroup packs = PackGroup.of(this.currentPacks.list().copyPacks(), this.availablePacks.list().copyPacks());
            screen = new PackedPacksScreen(this.field_22787, this.previous, this.original, packs);
        }

        this.field_22787.method_1507(screen);
    }

    @Override
    public void method_29638(List<Path> packs) {
        if (this.field_22787 != null) {
            this.field_22787.method_1507(new class_410(
                    this.confirmFileDrop(packs),
                    class_2561.method_43471("pack.dropConfirm"),
                    class_2561.method_43470(joinPackNames(packs))
            ));
        }
    }

    private BooleanConsumer confirmFileDrop(List<Path> packs) {
        return confirmed -> {
            if (this.field_22787 == null) {
                return;
            }
            if (!confirmed) {
                this.field_22787.method_1507(this);
                return;
            }
            PathValidationResults results = validatePaths(packs);

            if (!results.symlinkWarnings().isEmpty()) {
                this.field_22787.method_1507(class_8669.method_52750(() -> this.field_22787.method_1507(this)));
                return;
            }
            if (!results.valid().isEmpty()) {
                class_5375.method_29669(this.field_22787, results.valid(), this.original.packDir());
                this.refreshPacks();
            }
            if (!results.rejected().isEmpty()) {
                String rejectedNames = joinPackNames(results.rejected());
                this.field_22787.method_1507(new class_403(
                        () -> this.field_22787.method_1507(this),
                        class_2561.method_43471("pack.dropRejected.title"),
                        class_2561.method_43469("pack.dropRejected.message", rejectedNames)
                ));
                return;
            }
            this.field_22787.method_1507(this);
        };
    }

    private void setOriginalScreen() {
        if (this.previous instanceof class_5375) {
            this.method_25419();
        } else if (this.field_22787 != null) {
            this.field_22787.method_1507(this.original.createScreen(this.previous));
        }
    }

    @Override
    public void method_25419() {
        if (this.field_22787 == null) return;

        String commitRequestor = ModAdditions.forceCommitOnClose(this.original.packType());
        if (commitRequestor != null) {
            this.commit();
            PackedPacks.LOGGER.info("[packed_packs] Commiting packs on close at the request of mod '{}'.", commitRequestor);
        } else if (!(this.options.getUserConfig() instanceof Config.ResourcePacks resourceConfig) || resourceConfig.isApplyOnClose()) {
            this.commit();
        }

        if (this.original.packType() == class_3264.field_14190 && !(this.previous instanceof class_5375)) {
            this.original.output().accept(this.repository.getRepository()); // validate datapacks
            return;
        }

        if (this.previous instanceof PackSelectionScreenAccessor packScreen) {
            ((PackSelectionModelAccessor) packScreen.getModel()).packed_packs$reset();
            packScreen.invokeReload();
        }

        this.field_22787.method_1507(this.previous);
    }

    @Override
    public void method_25393() {
        if (this.watcher != null) {
            this.watcher.poll();
        }
    }

    private void createWatcher() {
        if (this.watcher == null) {
            this.watcherFuture = CompletableFuture.supplyAsync(() -> {
                try {
                    List<Path> paths = new ObjectArrayList<>(this.additionalFolders.size() + 1);
                    paths.add(this.repository.getBaseDir());
                    paths.addAll(this.additionalFolders);
                    return new PackWatcher(this.original.packType(), paths, this::refreshPacks);
                } catch (Exception e) {
                    PackedPacks.LOGGER.error("[packed_packs] Failed to initialize pack directory watcher.", e);
                    return null;
                }
            }, class_156.method_18349()).thenAcceptAsync(watcher -> {
                if (watcher != null) {
                    this.watcher = watcher;
                } else {
                    this.closeWatcher();
                }
            }, this.field_22787);
        }
    }

    private void closeWatcher() {
        if (this.watcherFuture != null) {
            this.watcherFuture.cancel(true);
        }

        if (this.watcher != null) {
            this.watcher.close();
            this.watcher = null;
        }
    }

    private void initAdditionalFolders() {
        this.additionalFolders = CollectionsUtil.deduplicate(CollectionsUtil.addAll(
                mapValidDirectories(this.options.getUserConfig().getAdditionalFolders()),
                this.repository.getAdditionalDirs()
        ));
    }

    private void repositionLists() {
        this.availablePacks.setHeaderVisibility(this.showActionBar);
        this.currentPacks.setHeaderVisibility(this.showActionBar);
    }

    @Override
    protected void method_48640() {
        this.layout.method_48222();
        this.layout.method_48229(0, 0);
        this.dialogs.forEach(ToggleableDialog::repositionElements);
        this.contextMenu.setOpen(false);
        this.repositionLists();
    }

    public void toggleActionBar() {
        this.showActionBar = !this.showActionBar;
        Config.get().setShowActionBar(this.showActionBar);
        this.repositionLists();
    }

    public void commit() {
        this.currentPacks.getSearchField().method_1852("");
        this.syncProfile(this.profiles.getProfile());
        this.repository.selectPacks(this.currentPacks.list().copyPacks());

        if (this.original.packType() == class_3264.field_14188) {
            this.original.output().accept(this.repository.getRepository());
        }
    }

    private void replacePacks(PackList list, List<class_3288> packs) {
        list.captureState().replaceAll(packs).restore();
    }

    private void revalidateFolder() {
        if (this.folderDialog.isOpen()) {
            FolderPack folderPack = this.folderDialog.getFolderPack();
            if (folderPack == null || this.repository.getFolderConfig(folderPack) == null) {
                this.folderDialog.setOpen(false);
            } else {
                this.replacePacks(this.folderDialog.root(), ImmutableList.copyOf(this.repository.getNestedPacks(folderPack)));
            }
        }
    }

    public void revalidatePacks() {
        PackList availableList = this.availablePacks.list();
        PackList currentList = this.currentPacks.list();
        PackGroup packs = this.repository.validatePacks(availableList.copyPacks(), currentList.copyPacks());
        this.assetManager.clearIconCache();
        this.replacePacks(availableList, packs.unselected());
        this.replacePacks(currentList, packs.selected());
        this.revalidateFolder();
        this.clearHistory();
    }

    public void refreshPacks() {
        this.refreshFuture = CompletableFuture.runAsync(this.repository::refresh, class_156.method_18349())
                .thenRunAsync(this::revalidatePacks, this.field_22787);
    }

    public void useSelected() {
        PackGroup packs = this.repository.getPacksBySelected();
        this.availablePacks.list().reload(packs.unselected());
        this.currentPacks.list().reload(packs.selected());
        this.clearHistory();
    }

    public void onProfileChange(@Nullable Profile previous, @Nullable Profile current) {
        if (previous != null) {
            previous.setPacks(this.currentPacks.list().copyPacks());
            Profiles.save(this.original.packType(), previous);
        }

        boolean unlocked = current == null || !current.isLocked();
        this.availablePacks.getTransferButton().field_22763 = unlocked;
        this.currentPacks.getTransferButton().field_22763 = unlocked;
        this.availablePacks.getSearchField().setValueSilently("");
        this.currentPacks.getSearchField().setValueSilently("");
        this.availablePacks.list().search("");
        this.currentPacks.list().search("");

        if (current != null && !current.getPackIds().isEmpty()) {
            this.applyProfile(current);
        } else {
            this.useSelected();
        }
    }

    public void onProfileCopy(@Nullable Profile original, @NotNull Profile copy) {
        copy.setPacks(this.currentPacks.list().copyPacks());
    }

    private void applyProfile(@NotNull Profile profile) {
        List<class_3288> available = this.availablePacks.list().copyPacks();
        List<class_3288> current = this.repository.getPacksByFlattenedIds(profile.getPackIds());
        this.applyPacks(available, current);
    }

    private void applyPacks(List<class_3288> available, List<class_3288> current) {
        PackGroup packs = this.repository.validatePacks(available, current);
        this.availablePacks.list().reload(packs.unselected());
        this.currentPacks.list().reload(packs.selected());
        this.clearHistory();
    }

    public void syncProfile(@Nullable Profile profile) {
        if (profile != null) {
            profile.syncPacks(this.repository.getPacks(), this.currentPacks.list().copyPacks());
        }
    }

    @Override
    public boolean isUnlocked() {
        Profile profile = this.profiles.getProfile();
        return profile == null || !profile.isLocked();
    }

    @Override
    public @NotNull List<PackList> getPackLists() {
        return this.packLists;
    }

    @Override
    public @Nullable PackList getDestination(PackList source) {
        if (source == this.availablePacks.list()) {
            return this.currentPacks.list();
        } else if (source == this.currentPacks.list()) {
            return this.availablePacks.list();
        }
        return null;
    }

    @Override
    protected void transferFocus(PackList source, PackList destination) {
        super.transferFocus(source, destination);

        if (destination == currentPacks.list()) {
            currentPacks.list().scrollToLastSelected();
        }
    }

    private void onFolderOpen(FolderOpenEvent event) {
        this.folderDialog.root().reload(this.repository.getNestedPacks(event.opened()));
        this.folderDialog.updateFolder(event.target(), event.opened(), this.assetManager);
        this.folderDialog.setOpen(true);
    }

    private void onFolderClose(FolderCloseEvent event) {
        this.folderDialog.setOpen(false);

        FolderPack folderPack = event.folderPack();
        if (folderPack == null) return;

        Folder folder = this.repository.getFolderConfig(folderPack);
        if (folder != null && this.isUnlocked()) {
            if (folder.trySetPacks(this.repository.validateAndOrderNestedPacks(folderPack, event.target().copyPacks()))) {
                folderPack.saveConfig(folder);
            }
            this.focusList(ObjectsUtil.firstNonNullOrDefault(this.availablePacks.list(), this.folderDialog.getParent()));
        }
    }

    private void onFileRename(FileRenameEvent event) {
        if (this.folderDialog.isOpen()) {
            this.folderDialog.onRename(event.renamed(), event.newName());
        }
        this.refreshPacks();
    }

    @Override
    protected void handleMoveEvent(MoveEvent event) {
        if (event.target() != this.folderDialog.root()) {
            super.handleMoveEvent(event);
            return;
        }

        PackList.Entry entry = event.target().getEntry(event.trigger());
        if (entry != null) {
            this.focus(class_8016.method_48194(entry, event.target(), this.folderDialog, this));
        } else {
            this.focus(class_8016.method_48194(event.target(), this.folderDialog, this));
        }
    }

    private void onOpenAliases(PackAliasOpenEvent event) {
        Objects.requireNonNull(this.aliasModal, "aliasModal");
        this.aliasModal.clear();
        this.aliasModal.root().layout().editAliases(event.trigger(), this.aliasModal::closeModal);
        this.aliasModal.root().method_48206(this.aliasModal::addRenderableWidget);
        this.aliasModal.repositionElements();
        this.aliasModal.setOpen(true);
    }

    @Override
    public void onEvent(PackListEvent event) {
        super.onEvent(event);

        this.profiles.getSidebar().setOpen(false);
        this.contextMenu.setOpen(false);
        this.fileRenameModal.setOpen(false);
        ObjectsUtil.ifPresent(this.aliasModal, Modal::closeModal);

        boolean notFolderDialogEvent = event.target() != this.folderDialog.root();
        if (notFolderDialogEvent) this.folderDialog.setOpen(false);

        switch (event) {
            case FileDeleteEvent ignore -> this.revalidatePacks();
            case FileRenameOpenEvent(PackList target, Pack trigger) -> this.fileRenameModal.open(target, trigger);
            case FileRenameEvent e -> this.onFileRename(e);
            case FileRenameCloseEvent e -> this.focusList(e.target());
            case FolderOpenEvent e -> this.onFolderOpen(e);
            case FolderCloseEvent e -> this.onFolderClose(e);
            case PackAliasOpenEvent e -> this.onOpenAliases(e);
            default -> {
            }
        }

        if (this.isUnlocked() && event.pushToHistory() && notFolderDialogEvent) {
            this.history.push(this.captureState());
        }
    }

    public @Nullable PackLayout getLayoutFromSelectedList() {
        return ObjectsUtil.firstNonNull(
                ObjectsUtil.pick(this.availablePacks, this.currentPacks, pl -> pl.list() == this.method_25399()),
                ObjectsUtil.pick(this.availablePacks, this.currentPacks, pl -> pl.list().method_49606()),
                ObjectsUtil.pick(this.availablePacks, this.currentPacks, pl -> pl.list().method_25370())
        );
    }

    public ToggleableEditBox<Void> focusSearchField(@NotNull PackLayout packLayout) {
        if (!this.showActionBar) this.toggleActionBar();
        ToggleableEditBox<Void> searchField = packLayout.getSearchField();
        this.focus(searchField);
        return searchField;
    }

    @Override
    public boolean method_25400(char codePoint, int modifiers) {
        if (super.method_25400(codePoint, modifiers)) {
            return true;
        }
        if (CollectionsUtil.anyMatch(this.dialogs, ToggleableDialog::isOpen)) {
            return false;
        }
        if (codePoint != field_31947 && noModifiers(modifiers)) {
            PackLayout packLayout = this.getLayoutFromSelectedList();
            if (packLayout != null && !packLayout.getSearchField().method_25370()) {
                return this.focusSearchField(packLayout).method_25400(codePoint, modifiers);
            }
        }
        return false;
    }

    public void toggleDevMode() {
        Config.get().setDevMode(!Config.get().isDevMode());
        ToastUtil.onDevModeToggleToast(Config.get().isDevMode());
        this.method_41843();
    }

    public void switchDefaultProfile() {
        this.options.getDefaultProfile().ifPresent(profile -> {
            if (Objects.equals(this.profiles.getProfile(), profile)) {
                this.profiles.setProfile(null);
            } else {
                this.profiles.setProfile(profile);
            }
        });
    }

    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        this.contextMenu.setOpen(false);

        if (isDeveloperMode(keyCode, modifiers)) {
            this.toggleDevMode();
            return true;
        }
        if (isSwitchDefaultProfile(keyCode, modifiers)) {
            this.switchDefaultProfile();
            return true;
        }
        if (isRefresh(keyCode, modifiers) && (this.refreshFuture == null || this.refreshFuture.isDone())) {
            this.refreshPacks();
            return true;
        }
        if (isOpenProfiles(keyCode, modifiers)) {
            this.profiles.getSidebar().toggle();
            return true;
        }
        if (super.method_25404(keyCode, scanCode, modifiers)) {
            return true;
        }
        if (isRedo(keyCode, modifiers) && this.isUnlocked()) {
            return this.history.redo();
        }
        if (isUndo(keyCode, modifiers) && this.isUnlocked()) {
            return this.history.undo();
        }
        if (method_25439(keyCode)) {
            PackLayout packLayout = this.getLayoutFromSelectedList();
            if (packLayout != null) {
                packLayout.list().selectAll();
                this.onEvent(new SelectionEvent(packLayout.list()));
                return true;
            }
        }
        if (keyCode == field_31986) {
            PackLayout packLayout = this.getLayoutFromSelectedList();
            if (packLayout != null) {
                ToggleableEditBox<Void> searchField = packLayout.getSearchField();
                if (!searchField.method_25370() && !searchField.method_1882().isEmpty()) {
                    return this.focusSearchField(packLayout).method_25404(keyCode, scanCode, modifiers);
                }
            }
        }
        return false;
    }

    private boolean hasHeader(List<MenuItem> items) {
        return !items.isEmpty() && items.getFirst() instanceof PackMenuHeader;
    }

    private void openContextMenu(int mouseX, int mouseY) {
        if (this.contextMenu.method_25405(mouseX, mouseY)) return;

        this.buildItems(mouseX, mouseY)
                .when(Config.get().isDevMode())
                .ifTrue(dev -> dev.separatorIfNonEmpty()
                        .whenNonNull(this.profiles.getProfile())
                        .ifTrue((profile, b) -> b.
                                add(devItem(ResourceUtil.getText("profile.save"))
                                        .action(() -> profile.setPacks(this.currentPacks.list().copyPacks()))
                                        .build())
                                .separator())
                        .add(devItem(ResourceUtil.getText("preferences"))
                                .addChildren(ToggleableHelper.preferences(this.original.packType()))
                                .addChild(devItem(ResourceUtil.getText("preferences.reset"))
                                        .action(Preferences.INSTANCE::reset)
                                        .build())
                                .build())
                )
                .separatorIfNonEmpty()
                .simpleItem(ResourceUtil.getText("reset_enabled"), this::isUnlocked, this::useSelected)
                .simpleItem(ResourceUtil.getText("refresh"), this::canRefresh, this::refreshPacks)
                .when(this.additionalFolders, List::isEmpty)
                .ifTrue(b -> b.simpleItem(OPEN_FOLDER_TEXT, this.repository::openDir))
                .orElse((dirs, b) -> b
                        .parent(OPEN_FOLDER_TEXT, p -> p
                                .add(new DirectoryMenuItem(this.repository.getBaseDir()))
                                .separator()
                                .iterate(dirs)
                                .map(DirectoryMenuItem::new)))
                .peek(items -> {
                    int yOffset = this.hasHeader(items) ? this.contextMenu.getItemHeight() : 0;
                    this.contextMenu.open(mouseX, mouseY - yOffset, items);
                });
    }

    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        this.setDragged(null);
        if (isRightClick(button) && !this.optionsModal.method_25405(mouseX, mouseY)) {
            this.openContextMenu((int) mouseX, (int) mouseY);
            return true;
        }
        if (ToggleableDialogContainer.super.method_25402(mouseX, mouseY, button)) {
            return true;
        }
        if (isClickForward(button) && this.isUnlocked()) {
            return this.history.redo();
        }
        if (isClickBack(button) && this.isUnlocked()) {
            return this.history.undo();
        }
        if (isLeftClick(button) && !(this.method_25399() instanceof PackList)) {
            this.method_25395(this.method_25396().getFirst());
            this.layout.method_48206(w -> w.method_25365(false));
        }
        this.contextMenu.setOpen(false);
        return false;
    }

    @Override
    public List<ToggleableDialog<?>> getDialogs() {
        return this.dialogs;
    }

    @Override
    public @Nullable class_364 getHovered() {
        return this.hoveredElement;
    }

    @Override
    public void method_25394(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
        this.hoveredElement = this.findHovered(mouseX, mouseY);

        super.method_25394(guiGraphics, mouseX, mouseY, partialTick);

        if (Config.get().isDevMode()) {
            float scale = 0.5f;
            int y = (int) ((field_22790 - this.field_22793.field_2000 * scale) / scale);

            guiGraphics.method_51448().method_22903();
            guiGraphics.method_51448().method_22905(scale, scale, 0);
            guiGraphics.method_27535(this.field_22793, ResourceUtil.getText("dev_mode", DEV_MODE_SHORTCUT), 0, y, Theme.WHITE.getARGB());
            guiGraphics.method_51448().method_22909();
        }
    }

    public boolean canRefresh() {
        return this.refreshFuture == null || this.refreshFuture.isDone();
    }

    public void clearHistory() {
        this.history.reset(this.captureState());
    }

    public void recordState(String eventName) {
        this.history.push(this.captureState(eventName));
    }

    @Override
    public @NotNull io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot captureState(String eventName) {
        return new io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot(this, this.availablePacks.list().captureState(), this.currentPacks.list().captureState());
    }

    @Override
    public void replaceState(@NotNull io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot snapshot) {
        Set<class_3288> validPacks = new ObjectOpenHashSet<>(this.repository.getPacks());
        Query availablePacksQuery = snapshot.availablePacks.model().query();
        this.availablePacks.getSortButton().setValueSilently(availablePacksQuery.sort());
        this.availablePacks.getCompatButton().setValueSilently(availablePacksQuery.hideIncompatible());
        this.availablePacks.getSearchField().setValueSilently(availablePacksQuery.unmodifiedSearch());
        this.currentPacks.getSearchField().setValueSilently(snapshot.currentPacks().model().query().unmodifiedSearch());
        snapshot.availablePacks.retainAll(validPacks).restore();
        snapshot.currentPacks.retainAll(validPacks).restore();
        this.availablePacks.list().scrollToLastSelected();
        this.currentPacks.list().scrollToLastSelected();
    }

    public record Snapshot(
            PackedPacksScreen target,
            PackList.Snapshot availablePacks,
            PackList.Snapshot currentPacks
    ) implements Restorable.Snapshot<io.github.fishstiz.packed_packs.gui.screens.PackedPacksScreen.Snapshot> {
    }
}
