/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.cache;

import com.alan.VillagerTradeManager.cache.CacheMetrics;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public class AdaptiveCache<K, V> {
    private final CacheMetrics metrics;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Map<K, CacheEntry<V>> cache = new ConcurrentHashMap<K, CacheEntry<V>>();
    private final Map<K, Long> accessTimes = new ConcurrentHashMap<K, Long>();
    private final Map<K, AtomicInteger> accessFrequencies = new ConcurrentHashMap<K, AtomicInteger>();
    private volatile int minSize;
    private volatile int maxSize;
    private volatile int currentMaxSize;
    private final long sizeAdjustmentIntervalMs;
    private final double targetHitRate;
    private final double sizeAdjustmentFactor;
    private volatile long lastSizeAdjustment = System.currentTimeMillis();
    private volatile double currentHitRate = 0.0;
    private volatile boolean adaptiveSizingEnabled = true;
    private static final double FREQUENCY_WEIGHT = 0.6;
    private static final double RECENCY_WEIGHT = 0.4;
    private static final int MIN_SIZE_ADJUSTMENT_INTERVAL = 30000;
    private static final double HIT_RATE_IMPROVEMENT_THRESHOLD = 2.0;

    public AdaptiveCache(int initialMinSize, int initialMaxSize, double targetHitRate, long sizeAdjustmentIntervalMs, double sizeAdjustmentFactor) {
        this.minSize = initialMinSize;
        this.maxSize = initialMaxSize;
        this.currentMaxSize = initialMaxSize;
        this.targetHitRate = targetHitRate;
        this.sizeAdjustmentIntervalMs = Math.max(sizeAdjustmentIntervalMs, 30000L);
        this.sizeAdjustmentFactor = Math.max(0.1, Math.min(sizeAdjustmentFactor, 0.5));
        this.metrics = new CacheMetrics();
    }

    public AdaptiveCache(int initialMaxSize) {
        this(Math.max(10, initialMaxSize / 10), initialMaxSize, 85.0, 60000L, 0.2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V get(K key) {
        long startTime = System.nanoTime();
        this.lock.readLock().lock();
        try {
            CacheEntry<V> entry = this.cache.get(key);
            if (entry != null) {
                entry.recordAccess();
                this.accessTimes.put(key, entry.lastAccessTime);
                this.accessFrequencies.computeIfAbsent(key, k -> new AtomicInteger(0)).incrementAndGet();
                long accessTime = System.nanoTime() - startTime;
                this.metrics.recordHit(accessTime);
                Object v = entry.value;
                return v;
            }
            long accessTime = System.nanoTime() - startTime;
            this.metrics.recordMiss(accessTime);
            V v = null;
            return v;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void put(K key, V value) {
        this.lock.writeLock().lock();
        try {
            if (this.cache.size() >= this.currentMaxSize && !this.cache.containsKey(key)) {
                this.performEviction();
            }
            CacheEntry<V> entry = this.cache.computeIfAbsent(key, k -> new CacheEntry<Object>(value));
            if (entry.value != value) {
                entry = new CacheEntry<V>(value);
                this.cache.put(key, entry);
            }
            this.accessTimes.put(key, entry.lastAccessTime);
            this.accessFrequencies.computeIfAbsent(key, k -> new AtomicInteger(0)).set(entry.accessCount);
            this.metrics.updateSize(this.cache.size());
            this.updateMemoryUsage();
            this.checkAdaptiveSizing();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V remove(K key) {
        this.lock.writeLock().lock();
        try {
            CacheEntry<V> entry = this.cache.remove(key);
            if (entry != null) {
                this.accessTimes.remove(key);
                this.accessFrequencies.remove(key);
                this.metrics.updateSize(this.cache.size());
                this.updateMemoryUsage();
                Object v = entry.value;
                return v;
            }
            V v = null;
            return v;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public boolean containsKey(K key) {
        return this.cache.containsKey(key);
    }

    public int size() {
        return this.cache.size();
    }

    public int getMaxSize() {
        return this.currentMaxSize;
    }

    public CacheMetrics getMetrics() {
        return this.metrics;
    }

    public void clear() {
        this.lock.writeLock().lock();
        try {
            this.cache.clear();
            this.accessTimes.clear();
            this.accessFrequencies.clear();
            this.metrics.updateSize(0);
            this.updateMemoryUsage();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void performEviction() {
        long startTime = System.nanoTime();
        int evictionsNeeded = Math.max(1, this.cache.size() - this.currentMaxSize + 1);
        PriorityQueue<AbstractMap.SimpleEntry<K, Double>> evictionCandidates = new PriorityQueue<AbstractMap.SimpleEntry<K, Double>>(evictionsNeeded + 1, (e1, e2) -> Double.compare((Double)e2.getValue(), (Double)e1.getValue()));
        long currentTime = System.currentTimeMillis();
        int sampleSize = 0;
        int maxSamples = Math.min(this.cache.size(), Math.max(50, evictionsNeeded * 5));
        for (Map.Entry<K, CacheEntry<V>> entry : this.cache.entrySet()) {
            K key = entry.getKey();
            CacheEntry<V> entry2 = entry.getValue();
            long timeSinceAccess = currentTime - entry2.lastAccessTime;
            double recencyScore = Math.max(0.0, 1.0 - (double)timeSinceAccess / (double)this.sizeAdjustmentIntervalMs);
            int frequency = this.accessFrequencies.getOrDefault(key, new AtomicInteger(0)).get();
            double frequencyScore = Math.min(1.0, (double)frequency / 100.0);
            double combinedScore = 0.6 * frequencyScore + 0.4 * recencyScore;
            evictionCandidates.offer(new AbstractMap.SimpleEntry<K, Double>(key, combinedScore));
            if (evictionCandidates.size() > evictionsNeeded) {
                evictionCandidates.poll();
            }
            if (++sampleSize < maxSamples) continue;
            break;
        }
        for (Map.Entry<K, CacheEntry<V>> entry : evictionCandidates) {
            K keyToEvict = entry.getKey();
            this.cache.remove(keyToEvict);
            this.accessTimes.remove(keyToEvict);
            this.accessFrequencies.remove(keyToEvict);
        }
        long evictionTime = System.nanoTime() - startTime;
        this.metrics.recordEviction(evictionTime);
        this.metrics.updateSize(this.cache.size());
    }

    private void checkAdaptiveSizing() {
        if (!this.adaptiveSizingEnabled) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        if (currentTime - this.lastSizeAdjustment < this.sizeAdjustmentIntervalMs) {
            return;
        }
        double newHitRate = this.metrics.getHitRate();
        double hitRateChange = newHitRate - this.currentHitRate;
        if (this.metrics.getSnapshot().totalRequests < 100L) {
            return;
        }
        int newMaxSize = this.currentMaxSize;
        if (hitRateChange > 2.0) {
            if (newHitRate >= this.targetHitRate) {
                newMaxSize = (int)Math.min((double)this.maxSize, (double)this.currentMaxSize * (1.0 + this.sizeAdjustmentFactor));
            }
        } else if (hitRateChange < -2.0 && newHitRate < this.targetHitRate) {
            newMaxSize = (int)Math.max((double)this.minSize, (double)this.currentMaxSize * (1.0 - this.sizeAdjustmentFactor));
        }
        if (newMaxSize != this.currentMaxSize) {
            this.currentMaxSize = newMaxSize;
            this.lastSizeAdjustment = currentTime;
            this.metrics.recordSizeAdjustment();
            if (this.cache.size() > this.currentMaxSize) {
                this.performEviction();
            }
        }
        this.currentHitRate = newHitRate;
    }

    private void updateMemoryUsage() {
        long estimatedBytes = (long)this.cache.size() * 1024L;
        this.metrics.updateMemoryUsage(estimatedBytes);
    }

    public void setAdaptiveSizingEnabled(boolean enabled) {
        this.adaptiveSizingEnabled = enabled;
    }

    public void setSizeBounds(int minSize, int maxSize) {
        this.minSize = Math.max(1, minSize);
        this.maxSize = Math.max(minSize, maxSize);
        this.currentMaxSize = Math.min(Math.max(this.currentMaxSize, minSize), maxSize);
    }

    public Map<String, Object> getStatistics() {
        CacheMetrics.CacheMetricsSnapshot snapshot = this.metrics.getSnapshot();
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("currentSize", this.cache.size());
        stats.put("maxSize", this.currentMaxSize);
        stats.put("minSize", this.minSize);
        stats.put("absoluteMaxSize", this.maxSize);
        stats.put("hitRate", snapshot.hitRate);
        stats.put("missRate", this.metrics.getMissRate());
        stats.put("evictionRate", this.metrics.getEvictionRate());
        stats.put("averageHitTime", snapshot.averageHitTime);
        stats.put("averageMissTime", snapshot.averageMissTime);
        stats.put("totalRequests", snapshot.totalRequests);
        stats.put("totalEvictions", snapshot.evictions);
        stats.put("estimatedMemoryUsage", snapshot.estimatedMemoryUsage);
        stats.put("sizeAdjustments", snapshot.sizeAdjustments);
        stats.put("adaptiveSizingEnabled", this.adaptiveSizingEnabled);
        stats.put("targetHitRate", this.targetHitRate);
        return stats;
    }

    private static class CacheEntry<V> {
        final V value;
        final long creationTime;
        volatile long lastAccessTime;
        volatile int accessCount;

        CacheEntry(V value) {
            this.value = value;
            this.lastAccessTime = this.creationTime = System.currentTimeMillis();
            this.accessCount = 1;
        }

        void recordAccess() {
            this.lastAccessTime = System.currentTimeMillis();
            ++this.accessCount;
        }
    }
}

