/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.metrics;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.database.DatabaseManager;
import com.alan.VillagerTradeManager.metrics.MetricsService;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import org.bukkit.Server;

public class HealthCheckService {
    private final VillagerTradeManager plugin;
    private final MetricsService metricsService;
    private final AtomicReference<HealthStatus> lastHealthStatus;
    private volatile Instant lastHealthCheck;

    public HealthCheckService(VillagerTradeManager plugin, MetricsService metricsService) {
        this.plugin = plugin;
        this.metricsService = metricsService;
        this.lastHealthStatus = new AtomicReference();
        this.lastHealthCheck = Instant.now();
    }

    public CompletableFuture<HealthStatus> performHealthCheck() {
        return CompletableFuture.supplyAsync(this::executeHealthCheck).thenApply(status -> {
            this.lastHealthStatus.set((HealthStatus)status);
            this.lastHealthCheck = Instant.now();
            return status;
        });
    }

    private HealthStatus executeHealthCheck() {
        HealthStatus.Builder builder = HealthStatus.builder().timestamp(Instant.now()).overallStatus(HealthStatus.Status.UP);
        this.checkPluginHealth(builder);
        this.checkDatabaseHealth(builder);
        this.checkServerHealth(builder);
        this.checkMemoryHealth(builder);
        this.checkPerformanceHealth(builder);
        return builder.build();
    }

    private void checkPluginHealth(HealthStatus.Builder builder) {
        try {
            boolean pluginEnabled = this.plugin.isEnabled();
            boolean pluginValid = this.plugin.getPluginMeta() != null && this.plugin.getPluginMeta().getName() != null;
            builder.component("plugin", HealthComponent.builder().status(pluginEnabled && pluginValid ? HealthStatus.Status.UP : HealthStatus.Status.DOWN).details(Map.of("enabled", pluginEnabled, "valid", pluginValid, "version", this.plugin.getPluginMeta().getVersion())).build());
            if (!pluginEnabled || !pluginValid) {
                builder.overallStatus(HealthStatus.Status.DOWN);
            } else {
                builder.overallStatus(HealthStatus.Status.UP);
            }
        }
        catch (Exception e) {
            builder.component("plugin", HealthComponent.builder().status(HealthStatus.Status.DOWN).details(Map.of("error", e.getMessage())).build());
            builder.overallStatus(HealthStatus.Status.DOWN);
        }
    }

    private void checkDatabaseHealth(HealthStatus.Builder builder) {
        try {
            HealthStatus.Status status;
            DatabaseManager dbManager = this.plugin.getDatabaseManager();
            boolean connected = dbManager.isConnected();
            HashMap<String, Object> details = new HashMap<String, Object>();
            details.put("connected", connected);
            details.put("type", dbManager.getDatabaseType().toString());
            Instant start = Instant.now();
            boolean responsive = dbManager.testConnection().get();
            Duration responseTime = Duration.between(start, Instant.now());
            details.put("responsive", responsive);
            details.put("responseTimeMs", responseTime.toMillis());
            double successRate = this.metricsService.getQuerySuccessRate();
            details.put("querySuccessRate", successRate);
            HealthStatus.Status status2 = status = connected && responsive && successRate > 0.95 ? HealthStatus.Status.UP : HealthStatus.Status.DEGRADED;
            if (!connected || !responsive) {
                status = HealthStatus.Status.DOWN;
                builder.overallStatus(HealthStatus.Status.DOWN);
            }
            builder.component("database", HealthComponent.builder().status(status).details(details).build());
        }
        catch (Exception e) {
            builder.component("database", HealthComponent.builder().status(HealthStatus.Status.DOWN).details(Map.of("error", e.getMessage())).build());
            builder.overallStatus(HealthStatus.Status.DOWN);
        }
    }

    private void checkServerHealth(HealthStatus.Builder builder) {
        try {
            HealthStatus.Status status;
            double[] tps;
            Server server = this.plugin.getServer();
            boolean running = !server.isStopping();
            HashMap<String, Object> details = new HashMap<String, Object>();
            details.put("running", running);
            details.put("version", server.getVersion());
            details.put("bukkitVersion", server.getBukkitVersion());
            details.put("onlinePlayers", server.getOnlinePlayers().size());
            details.put("maxPlayers", server.getMaxPlayers());
            details.put("tps", server.getTPS());
            double avgTps = 0.0;
            for (double t : tps = server.getTPS()) {
                avgTps += t;
            }
            details.put("averageTps", avgTps /= (double)tps.length);
            HealthStatus.Status status2 = status = running && avgTps > 15.0 ? HealthStatus.Status.UP : HealthStatus.Status.DEGRADED;
            if (!running) {
                status = HealthStatus.Status.DOWN;
                builder.overallStatus(HealthStatus.Status.DOWN);
            }
            builder.component("server", HealthComponent.builder().status(status).details(details).build());
        }
        catch (Exception e) {
            builder.component("server", HealthComponent.builder().status(HealthStatus.Status.DOWN).details(Map.of("error", e.getMessage())).build());
            builder.overallStatus(HealthStatus.Status.DOWN);
        }
    }

    private void checkMemoryHealth(HealthStatus.Builder builder) {
        try {
            HealthStatus.Status status;
            Runtime runtime = Runtime.getRuntime();
            long totalMemory = runtime.totalMemory();
            long freeMemory = runtime.freeMemory();
            long usedMemory = totalMemory - freeMemory;
            long maxMemory = runtime.maxMemory();
            double memoryUsagePercent = (double)usedMemory / (double)maxMemory * 100.0;
            HashMap<String, Object> details = new HashMap<String, Object>();
            details.put("totalMemory", totalMemory);
            details.put("freeMemory", freeMemory);
            details.put("usedMemory", usedMemory);
            details.put("maxMemory", maxMemory);
            details.put("memoryUsagePercent", memoryUsagePercent);
            if (memoryUsagePercent > 95.0) {
                status = HealthStatus.Status.DOWN;
                builder.overallStatus(HealthStatus.Status.DOWN);
            } else if (memoryUsagePercent > 90.0) {
                status = HealthStatus.Status.DEGRADED;
                builder.overallStatus(HealthStatus.Status.DEGRADED);
            } else {
                status = HealthStatus.Status.UP;
            }
            builder.component("memory", HealthComponent.builder().status(status).details(details).build());
        }
        catch (Exception e) {
            builder.component("memory", HealthComponent.builder().status(HealthStatus.Status.DOWN).details(Map.of("error", e.getMessage())).build());
            builder.overallStatus(HealthStatus.Status.DOWN);
        }
    }

    private void checkPerformanceHealth(HealthStatus.Builder builder) {
        try {
            HashMap<String, Object> details = new HashMap<String, Object>();
            double cacheHitRatio = this.metricsService.getCacheHitRatio();
            details.put("cacheHitRatio", cacheHitRatio);
            double querySuccessRate = this.metricsService.getQuerySuccessRate();
            details.put("querySuccessRate", querySuccessRate);
            double performanceScore = (cacheHitRatio + querySuccessRate) / 2.0;
            details.put("performanceScore", performanceScore);
            HealthStatus.Status status = performanceScore > 0.8 ? HealthStatus.Status.UP : HealthStatus.Status.DEGRADED;
            builder.component("performance", HealthComponent.builder().status(status).details(details).build());
            if (performanceScore < 0.5) {
                builder.overallStatus(HealthStatus.Status.DEGRADED);
            }
        }
        catch (Exception e) {
            builder.component("performance", HealthComponent.builder().status(HealthStatus.Status.DOWN).details(Map.of("error", e.getMessage())).build());
        }
    }

    public HealthStatus getLastHealthStatus() {
        return this.lastHealthStatus.get();
    }

    public Duration getTimeSinceLastCheck() {
        return Duration.between(this.lastHealthCheck, Instant.now());
    }

    public boolean isHealthStatusCurrent() {
        return this.getTimeSinceLastCheck().getSeconds() < 30L;
    }

    public static class HealthStatus {
        private final Status overallStatus;
        private final Instant timestamp;
        private final Map<String, HealthComponent> components;

        private HealthStatus(Status overallStatus, Instant timestamp, Map<String, HealthComponent> components) {
            this.overallStatus = overallStatus;
            this.timestamp = timestamp;
            this.components = components;
        }

        public Status getOverallStatus() {
            return this.overallStatus;
        }

        public Instant getTimestamp() {
            return this.timestamp;
        }

        public Map<String, HealthComponent> getComponents() {
            return this.components;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static enum Status {
            UP,
            DOWN,
            DEGRADED;

        }

        public static class Builder {
            private Status overallStatus = Status.UP;
            private Instant timestamp;
            private final Map<String, HealthComponent> components = new HashMap<String, HealthComponent>();

            public Builder overallStatus(Status status) {
                this.overallStatus = status;
                return this;
            }

            public Builder timestamp(Instant timestamp) {
                this.timestamp = timestamp;
                return this;
            }

            public Builder component(String name, HealthComponent component) {
                this.components.put(name, component);
                return this;
            }

            public HealthStatus build() {
                return new HealthStatus(this.overallStatus, this.timestamp, Map.copyOf(this.components));
            }
        }
    }

    public static class HealthComponent {
        private final HealthStatus.Status status;
        private final Map<String, Object> details;

        private HealthComponent(HealthStatus.Status status, Map<String, Object> details) {
            this.status = status;
            this.details = details;
        }

        public HealthStatus.Status getStatus() {
            return this.status;
        }

        public Map<String, Object> getDetails() {
            return this.details;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private HealthStatus.Status status;
            private final Map<String, Object> details = new HashMap<String, Object>();

            public Builder status(HealthStatus.Status status) {
                this.status = status;
                return this;
            }

            public Builder details(Map<String, Object> details) {
                this.details.putAll(details);
                return this;
            }

            public Builder detail(String key, Object value) {
                this.details.put(key, value);
                return this;
            }

            public HealthComponent build() {
                return new HealthComponent(this.status, Map.copyOf(this.details));
            }
        }
    }
}

