/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.exception.VillagerTradeException;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.lang.constant.Constable;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.bukkit.configuration.ConfigurationSection;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;

public class ConfigurationService {
    private final VillagerTradeManager plugin;
    private final Yaml yaml;

    public ConfigurationService(VillagerTradeManager plugin) {
        this.plugin = plugin;
        DumperOptions options = new DumperOptions();
        options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
        options.setPrettyFlow(true);
        this.yaml = new Yaml(options);
    }

    public ValidationResult validateConfiguration() {
        ValidationResult result = new ValidationResult();
        try {
            this.validateCoreSettings(result);
            this.validateProfessionSettings(result);
            this.validateEconomySettings(result);
            this.validateItemSettings(result);
            this.validateAdvancedFeatures(result);
            this.validateSystemSettings(result);
        }
        catch (Exception e) {
            result.addError("general", "Configuration validation failed: " + e.getMessage());
        }
        return result;
    }

    public Map<String, Object> getDefaultConfiguration() {
        HashMap<String, Object> defaults = new HashMap<String, Object>();
        HashMap<String, Object> coreDefaults = new HashMap<String, Object>();
        coreDefaults.put("trade_management_enabled", true);
        coreDefaults.put("operation_mode", "normal");
        defaults.put("core", coreDefaults);
        HashMap professionDefaults = new HashMap();
        for (VillagerProfession profession : VillagerProfession.values()) {
            HashMap<String, Object> profConfig = new HashMap<String, Object>();
            profConfig.put("enabled", true);
            profConfig.put("description", profession.getDisplayName() + " trades");
            professionDefaults.put(profession.getKey(), profConfig);
        }
        defaults.put("professions", professionDefaults);
        HashMap<String, Comparable<Boolean>> economyDefaults = new HashMap<String, Comparable<Boolean>>();
        economyDefaults.put("custom_restock_enabled", Boolean.valueOf(false));
        economyDefaults.put("restock_interval_minutes", Integer.valueOf(30));
        economyDefaults.put("restock_limit_per_day", Integer.valueOf(3));
        economyDefaults.put("unlimited_restock", Boolean.valueOf(false));
        economyDefaults.put("instant_restock_enabled", Boolean.valueOf(false));
        economyDefaults.put("stock_management_enabled", Boolean.valueOf(false));
        economyDefaults.put("default_max_stock", Integer.valueOf(10));
        economyDefaults.put("stock_restock_interval_minutes", Integer.valueOf(60));
        economyDefaults.put("job_site_validation_enabled", Boolean.valueOf(true));
        economyDefaults.put("automatic_restocking_enabled", Boolean.valueOf(true));
        defaults.put("economy", economyDefaults);
        HashMap<String, Object> itemDefaults = new HashMap<String, Object>();
        itemDefaults.put("banned_items", Arrays.asList("diamond_sword", "enchanted_golden_apple"));
        itemDefaults.put("allow_enchanted_items", true);
        itemDefaults.put("allow_rare_items", true);
        itemDefaults.put("filter_custom_items", false);
        defaults.put("items", itemDefaults);
        HashMap<String, Boolean> advancedDefaults = new HashMap<String, Boolean>();
        advancedDefaults.put("dynamic_trades.enabled", true);
        advancedDefaults.put("level_progression.enabled", true);
        advancedDefaults.put("price_dynamics.enabled", true);
        advancedDefaults.put("wandering_traders.enabled", true);
        advancedDefaults.put("trade_enhancements.enabled", true);
        defaults.put("advanced_features", advancedDefaults);
        HashMap<String, Constable> systemDefaults = new HashMap<String, Constable>();
        systemDefaults.put("async_timeout_seconds", Integer.valueOf(30));
        systemDefaults.put("folia_optimizations", Boolean.valueOf(true));
        systemDefaults.put("paper_optimizations", Boolean.valueOf(true));
        systemDefaults.put("spigot_compatibility_mode", Boolean.valueOf(false));
        systemDefaults.put("config_backup_enabled", Boolean.valueOf(true));
        systemDefaults.put("config_backup_interval_hours", Integer.valueOf(24));
        systemDefaults.put("max_config_backups", Integer.valueOf(10));
        defaults.put("system", systemDefaults);
        return defaults;
    }

    public boolean isProfessionEnabled(VillagerProfession profession) {
        return this.plugin.getConfig().getBoolean("professions." + profession.getKey() + ".enabled", true);
    }

    public Set<VillagerProfession> getEnabledProfessions() {
        return Arrays.stream(VillagerProfession.values()).filter(this::isProfessionEnabled).collect(Collectors.toSet());
    }

    public void exportConfiguration(File exportFile) throws VillagerTradeException {
        try {
            HashMap<String, Map> configData = new HashMap<String, Map>();
            for (ConfigCategory category : ConfigCategory.values()) {
                ConfigurationSection section = this.plugin.getConfig().getConfigurationSection(category.getKey());
                if (section == null) continue;
                configData.put(category.getKey(), section.getValues(true));
            }
            try (FileWriter writer = new FileWriter(exportFile);){
                this.yaml.dump(configData, (Writer)writer);
            }
            this.plugin.getLogger().info("Configuration exported to: " + exportFile.getAbsolutePath());
        }
        catch (IOException e) {
            throw new VillagerTradeException("Failed to export configuration: " + e.getMessage(), e);
        }
    }

    public ValidationResult importConfiguration(File importFile) throws VillagerTradeException {
        try {
            Object loadedData;
            try (FileReader reader = new FileReader(importFile);){
                loadedData = this.yaml.load((Reader)reader);
            }
            if (!(loadedData instanceof Map)) {
                throw new VillagerTradeException("Invalid YAML format: expected configuration object, got " + (loadedData != null ? loadedData.getClass().getSimpleName() : "null"));
            }
            Map importedData = (Map)loadedData;
            ValidationResult validation = this.validateImportedData(importedData);
            if (!validation.isValid()) {
                return validation;
            }
            this.createConfigBackup();
            this.applyImportedConfiguration(importedData);
            this.plugin.reloadConfig();
            this.plugin.getLogger().info("Configuration imported from: " + importFile.getAbsolutePath());
            return validation;
        }
        catch (IOException e) {
            throw new VillagerTradeException("Failed to import configuration: " + e.getMessage(), e);
        }
    }

    public Map<ConfigCategory, Map<String, Object>> getCategorizedSettings() {
        LinkedHashMap<ConfigCategory, Map<String, Object>> categorized = new LinkedHashMap<ConfigCategory, Map<String, Object>>();
        for (ConfigCategory category : ConfigCategory.values()) {
            ConfigurationSection section = this.plugin.getConfig().getConfigurationSection(category.getKey());
            if (section != null) {
                categorized.put(category, section.getValues(true));
                continue;
            }
            categorized.put(category, new HashMap());
        }
        return categorized;
    }

    public Map<String, Object> getFlattenedSettings() {
        HashMap<String, Object> flattened = new HashMap<String, Object>();
        for (ConfigCategory category : ConfigCategory.values()) {
            ConfigurationSection section = this.plugin.getConfig().getConfigurationSection(category.getKey());
            if (section == null) continue;
            this.flattenSection(section, category.getKey(), flattened);
        }
        return flattened;
    }

    private void validateCoreSettings(ValidationResult result) {
        boolean tradeEnabled = this.plugin.getConfig().getBoolean("core.trade_management_enabled", true);
        String operationMode = this.plugin.getConfig().getString("core.operation_mode", "normal");
        if (!Arrays.asList("normal", "maintenance", "debug").contains(operationMode)) {
            result.addError("core.operation_mode", "Invalid operation mode: " + operationMode);
        }
    }

    private void validateProfessionSettings(ValidationResult result) {
        for (VillagerProfession profession : VillagerProfession.values()) {
            String path = "professions." + profession.getKey() + ".enabled";
            if (this.plugin.getConfig().isBoolean(path)) continue;
            result.addWarning(path, "Profession toggle should be boolean, using default true");
        }
    }

    private void validateEconomySettings(ValidationResult result) {
        int restockLimit;
        int restockInterval = this.plugin.getConfig().getInt("economy.restock_interval_minutes", 30);
        if (restockInterval < 1 || restockInterval > 1440) {
            result.addError("economy.restock_interval_minutes", "Restock interval must be between 1-1440 minutes");
        }
        if ((restockLimit = this.plugin.getConfig().getInt("economy.restock_limit_per_day", 3)) < 0) {
            result.addError("economy.restock_limit_per_day", "Restock limit cannot be negative");
        }
    }

    private void validateItemSettings(ValidationResult result) {
        List bannedItems = this.plugin.getConfig().getStringList("items.banned_items");
        for (String item : bannedItems) {
            if (item != null && !item.trim().isEmpty()) continue;
            result.addWarning("items.banned_items", "Empty or null item in banned items list");
        }
    }

    private void validateAdvancedFeatures(ValidationResult result) {
    }

    private void validateSystemSettings(ValidationResult result) {
        int asyncTimeout = this.plugin.getConfig().getInt("system.async_timeout_seconds", 30);
        if (asyncTimeout < 5 || asyncTimeout > 300) {
            result.addError("system.async_timeout_seconds", "Async timeout must be between 5-300 seconds");
        }
    }

    private ValidationResult validateImportedData(Map<String, Object> data) {
        ValidationResult result = new ValidationResult();
        if (data == null) {
            result.addError("general", "Imported data is null");
            return result;
        }
        Map metadata = (Map)data.get("metadata");
        if (metadata == null) {
            result.addWarning("metadata", "No metadata found in imported configuration");
        }
        for (ConfigCategory category : ConfigCategory.values()) {
            if (data.containsKey(category.getKey())) continue;
            result.addWarning(category.getKey(), "Category missing from imported configuration");
        }
        return result;
    }

    private void createConfigBackup() {
        try {
            File configFile = new File(this.plugin.getDataFolder(), "config.yml");
            if (configFile.exists()) {
                File backupDir = new File(this.plugin.getDataFolder(), "config_backups");
                if (!backupDir.exists()) {
                    backupDir.mkdirs();
                }
                String timestamp = String.valueOf(System.currentTimeMillis());
                File backupFile = new File(backupDir, "config_backup_" + timestamp + ".yml");
                Files.copy(configFile.toPath(), backupFile.toPath(), new CopyOption[0]);
                this.plugin.getLogger().info("Configuration backup created: " + backupFile.getName());
            }
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to create configuration backup: " + e.getMessage());
        }
    }

    private void applyImportedConfiguration(Map<String, Object> data) {
        data.remove("metadata");
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String category = entry.getKey();
            Map categoryData = (Map)entry.getValue();
            this.applyCategoryConfiguration(category, categoryData);
        }
    }

    private void applyCategoryConfiguration(String category, Map<String, Object> data) {
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String key = category + "." + entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Map) {
                this.applyCategoryConfiguration(key, (Map)value);
                continue;
            }
            this.plugin.getConfig().set(key, value);
        }
    }

    private void flattenSection(ConfigurationSection section, String prefix, Map<String, Object> result) {
        for (String key : section.getKeys(false)) {
            String fullKey = prefix + "." + key;
            Object value = section.get(key);
            if (value instanceof ConfigurationSection) {
                this.flattenSection((ConfigurationSection)value, fullKey, result);
                continue;
            }
            result.put(fullKey, value);
        }
    }

    public static class ValidationResult {
        private final List<String> errors = new ArrayList<String>();
        private final List<String> warnings = new ArrayList<String>();
        private final Map<String, String> fieldErrors = new HashMap<String, String>();
        private final Map<String, String> fieldWarnings = new HashMap<String, String>();

        public void addError(String field, String message) {
            this.errors.add(message);
            this.fieldErrors.put(field, message);
        }

        public void addWarning(String field, String message) {
            this.warnings.add(message);
            this.fieldWarnings.put(field, message);
        }

        public boolean isValid() {
            return this.errors.isEmpty();
        }

        public List<String> getErrors() {
            return new ArrayList<String>(this.errors);
        }

        public List<String> getWarnings() {
            return new ArrayList<String>(this.warnings);
        }

        public Map<String, String> getFieldErrors() {
            return new HashMap<String, String>(this.fieldErrors);
        }

        public Map<String, String> getFieldWarnings() {
            return new HashMap<String, String>(this.fieldWarnings);
        }

        public String getSummary() {
            StringBuilder sb = new StringBuilder();
            if (!this.errors.isEmpty()) {
                sb.append("Errors: ").append(this.errors.size()).append(", ");
            }
            if (!this.warnings.isEmpty()) {
                sb.append("Warnings: ").append(this.warnings.size());
            }
            return sb.toString();
        }
    }

    public static enum VillagerProfession {
        ARMORER("armorer", "Armorer"),
        BUTCHER("butcher", "Butcher"),
        CARTOGRAPHER("cartographer", "Cartographer"),
        CLERIC("cleric", "Cleric"),
        FARMER("farmer", "Farmer"),
        FISHERMAN("fisherman", "Fisherman"),
        FLETCHER("fletcher", "Fletcher"),
        LEATHERWORKER("leatherworker", "Leatherworker"),
        LIBRARIAN("librarian", "Librarian"),
        MASON("mason", "Mason"),
        SHEPHERD("shepherd", "Shepherd"),
        TOOLSMITH("toolsmith", "Toolsmith"),
        WEAPONSMITH("weaponsmith", "Weaponsmith");

        private final String key;
        private final String displayName;

        private VillagerProfession(String key, String displayName) {
            this.key = key;
            this.displayName = displayName;
        }

        public String getKey() {
            return this.key;
        }

        public String getDisplayName() {
            return this.displayName;
        }

        public static VillagerProfession fromString(String key) {
            for (VillagerProfession profession : VillagerProfession.values()) {
                if (!profession.key.equalsIgnoreCase(key)) continue;
                return profession;
            }
            return null;
        }
    }

    public static enum ConfigCategory {
        CORE("core", "Core System Settings"),
        PROFESSIONS("professions", "Profession Trade Toggles"),
        ECONOMY("economy", "Restock & Economy Settings"),
        ITEMS("items", "Item Control Settings"),
        ADVANCED_FEATURES("advanced_features", "Advanced Features Settings"),
        SYSTEM("system", "System Configuration");

        private final String key;
        private final String displayName;

        private ConfigCategory(String key, String displayName) {
            this.key = key;
            this.displayName = displayName;
        }

        public String getKey() {
            return this.key;
        }

        public String getDisplayName() {
            return this.displayName;
        }
    }
}

