/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.database.DatabaseProvider;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;

public class DatabaseService {
    private final VillagerTradeManager plugin;
    private final DatabaseProvider databaseProvider;
    private final ExecutorService executor;

    public DatabaseService(VillagerTradeManager plugin, DatabaseProvider databaseProvider) {
        this.plugin = plugin;
        this.databaseProvider = databaseProvider;
        this.executor = Executors.newFixedThreadPool(4);
    }

    public <T> CompletableFuture<T> executeAsync(Supplier<T> operation, String operationName) {
        CompletableFuture future = new CompletableFuture();
        this.executor.submit(() -> {
            try {
                Object result = operation.get();
                future.complete(result);
            }
            catch (Exception e) {
                if (e instanceof DatabaseException) {
                    future.completeExceptionally(e);
                }
                if (e instanceof RuntimeException && e.getCause() instanceof DatabaseException) {
                    future.completeExceptionally((DatabaseException)e.getCause());
                }
                this.logDatabaseException("Async operation failed: " + operationName, e, Level.SEVERE);
                future.completeExceptionally(new DatabaseException("Async operation failed: " + operationName, "DatabaseService", operationName, e));
            }
        });
        return future;
    }

    public void executeAsync(Runnable operation, String operationName) {
        this.executor.submit(() -> {
            try {
                operation.run();
            }
            catch (Exception e) {
                this.logDatabaseException("Async operation failed: " + operationName, e, Level.SEVERE);
            }
        });
    }

    public Connection getConnection() throws DatabaseException {
        try {
            return this.databaseProvider.getConnection().get();
        }
        catch (Exception e) {
            throw new DatabaseException("Failed to get database connection", "DatabaseService", "getConnection", e);
        }
    }

    public void executeInTransaction(TransactionCallback callback, String operationName) throws DatabaseException {
        Connection conn = null;
        try {
            conn = this.getConnection();
            boolean originalAutoCommit = conn.getAutoCommit();
            conn.setAutoCommit(false);
            try {
                callback.execute(conn);
                conn.commit();
            }
            catch (Exception e) {
                conn.rollback();
                throw new DatabaseException("Transaction failed: " + operationName, "DatabaseService", operationName, e);
            }
            finally {
                conn.setAutoCommit(originalAutoCommit);
            }
        }
        catch (SQLException e) {
            throw new DatabaseException("Transaction setup failed: " + operationName, "DatabaseService", operationName, e);
        }
        finally {
            if (conn != null) {
                try {
                    conn.close();
                }
                catch (SQLException e) {
                    this.plugin.getLogger().warning("Failed to close connection: " + e.getMessage());
                }
            }
        }
    }

    public <T> T executeQuery(String sql, QueryCallback<T> callback, String operationName) throws DatabaseException {
        T t;
        Connection conn = null;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            conn = this.getConnection();
            pstmt = conn.prepareStatement(sql);
            rs = pstmt.executeQuery();
            t = callback.process(rs);
            this.closeResources(rs, pstmt, conn);
        }
        catch (SQLException e) {
            try {
                this.plugin.getLogger().severe("SQL Error in " + operationName + ": " + e.getMessage());
                throw new DatabaseException("Query execution failed: " + operationName + " | " + e.getMessage(), "DatabaseService", operationName, e);
            }
            catch (Throwable throwable) {
                this.closeResources(rs, pstmt, conn);
                throw throwable;
            }
        }
        return t;
    }

    public int executeUpdate(String sql, UpdateCallback callback, String operationName) throws DatabaseException {
        Connection conn = null;
        PreparedStatement pstmt = null;
        try {
            conn = this.getConnection();
            pstmt = conn.prepareStatement(sql);
            if (callback != null) {
                callback.setParameters(pstmt);
            }
            int n = pstmt.executeUpdate();
            return n;
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("SQL Error in " + operationName + ": " + e.getMessage());
            throw new DatabaseException("Update execution failed: " + operationName + " | " + e.getMessage(), "DatabaseService", operationName, e);
        }
        finally {
            this.closeResources(null, pstmt, conn);
        }
    }

    private void closeResources(ResultSet rs, PreparedStatement pstmt, Connection conn) {
        if (rs != null) {
            try {
                rs.close();
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to close ResultSet: " + e.getMessage());
            }
        }
        if (pstmt != null) {
            try {
                pstmt.close();
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to close PreparedStatement: " + e.getMessage());
            }
        }
        if (conn != null) {
            try {
                conn.close();
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to close Connection: " + e.getMessage());
            }
        }
    }

    private void logDatabaseException(String context, Exception e, Level level) {
        DatabaseException dbException = new DatabaseException(context + ": " + e.getMessage(), "DatabaseService", "database_operation", e);
        this.plugin.getLogger().log(level, dbException.getMessage());
    }

    public void shutdown() {
        if (this.executor != null && !this.executor.isShutdown()) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(30L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                    if (!this.executor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        this.plugin.getLogger().warning("DatabaseService executor did not terminate cleanly");
                    }
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    @FunctionalInterface
    public static interface TransactionCallback {
        public void execute(Connection var1) throws SQLException;
    }

    @FunctionalInterface
    public static interface QueryCallback<T> {
        public T process(ResultSet var1) throws SQLException;
    }

    @FunctionalInterface
    public static interface UpdateCallback {
        public void setParameters(PreparedStatement var1) throws SQLException;
    }
}

