/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import com.alan.VillagerTradeManager.platform.PlatformService;
import com.alan.VillagerTradeManager.scheduler.TaskScheduler;
import com.alan.VillagerTradeManager.services.DatabaseService;
import com.alan.VillagerTradeManager.services.SettingsService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.entity.Villager;
import org.bukkit.inventory.MerchantRecipe;

public class RestockService {
    private final VillagerTradeManager plugin;
    private final DatabaseService databaseService;
    private final SettingsService settingsService;
    private boolean customRestockEnabled;
    private int restockIntervalMinutes;
    private int restockLimitPerDay;
    private boolean unlimitedRestock;
    private boolean instantRestockEnabled;
    private Map<UUID, RestockData> villagerRestockData = new ConcurrentHashMap<UUID, RestockData>();
    private static final int MAX_RESTOCK_CACHE_SIZE = 5000;

    public RestockService(VillagerTradeManager plugin, DatabaseService databaseService, SettingsService settingsService) {
        this.plugin = plugin;
        this.databaseService = databaseService;
        this.settingsService = settingsService;
        this.createRestockTable();
        this.loadSettings();
    }

    private void loadSettings() {
        this.customRestockEnabled = this.settingsService.isCustomRestockEnabled();
        this.restockIntervalMinutes = this.settingsService.getRestockIntervalMinutes();
        this.restockLimitPerDay = this.settingsService.getRestockLimitPerDay();
        this.unlimitedRestock = this.settingsService.isUnlimitedRestock();
        this.instantRestockEnabled = this.settingsService.isInstantRestockEnabled();
    }

    private void createRestockTable() {
        try {
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS villager_restock_data (villager_uuid TEXT PRIMARY KEY, last_restock_time INTEGER, restock_count_today INTEGER, last_reset_day INTEGER)", null, "createRestockTable");
        }
        catch (DatabaseException e) {
            this.logDatabaseException("Could not create restock table", e, Level.SEVERE);
        }
    }

    public void processVillagerRestock(Villager villager) {
        if (this.instantRestockEnabled) {
            return;
        }
        if (!this.customRestockEnabled) {
            return;
        }
        UUID villagerUUID = villager.getUniqueId();
        long currentTime = System.currentTimeMillis();
        long currentDay = currentTime / 86400000L;
        RestockData restockData = this.loadVillagerRestockData(villagerUUID);
        if (restockData.lastResetDay != currentDay) {
            restockData.restockCountToday = 0;
            restockData.lastResetDay = currentDay;
        }
        if (!this.unlimitedRestock && restockData.restockCountToday >= this.restockLimitPerDay) {
            return;
        }
        if (currentTime - restockData.lastRestockTime < (long)(this.restockIntervalMinutes * 60) * 1000L) {
            return;
        }
        this.restockVillagerTrades(villager);
        restockData.lastRestockTime = currentTime;
        ++restockData.restockCountToday;
        this.saveVillagerRestockData(villagerUUID, restockData);
    }

    public void restockVillagerTrades(Villager villager) {
        ArrayList<MerchantRecipe> recipes = new ArrayList<MerchantRecipe>();
        for (MerchantRecipe recipe : villager.getRecipes()) {
            MerchantRecipe newRecipe = new MerchantRecipe(recipe.getResult(), 0, recipe.getMaxUses(), recipe.hasExperienceReward(), recipe.getVillagerExperience(), recipe.getPriceMultiplier());
            newRecipe.setIngredients(new ArrayList(recipe.getIngredients()));
            newRecipe.setDemand(recipe.getDemand());
            newRecipe.setSpecialPrice(recipe.getSpecialPrice());
            recipes.add(newRecipe);
        }
        villager.setRecipes(recipes);
    }

    private RestockData loadVillagerRestockData(UUID villagerUUID) {
        RestockData data = this.villagerRestockData.get(villagerUUID);
        if (data != null) {
            return data;
        }
        data = new RestockData(0L, 0, 0L);
        this.villagerRestockData.put(villagerUUID, data);
        ((CompletableFuture)this.databaseService.executeAsync(() -> {
            try {
                return this.databaseService.executeQuery("SELECT last_restock_time, restock_count_today, last_reset_day FROM villager_restock_data WHERE villager_uuid = ?", rs -> {
                    if (rs.next()) {
                        return new RestockData(rs.getLong("last_restock_time"), rs.getInt("restock_count_today"), rs.getLong("last_reset_day"));
                    }
                    return new RestockData(0L, 0, 0L);
                }, "loadVillagerRestockData");
            }
            catch (DatabaseException e) {
                this.logDatabaseException("Could not load villager restock data", e, Level.SEVERE);
                return new RestockData(0L, 0, 0L);
            }
        }, "loadVillagerRestockData").thenAccept(loadedData -> {
            if (loadedData != null) {
                this.villagerRestockData.put(villagerUUID, (RestockData)loadedData);
                this.enforceCacheSizeLimit();
            }
        })).exceptionally(throwable -> {
            this.plugin.getLogger().warning("Failed to load restock data for villager " + String.valueOf(villagerUUID) + ": " + throwable.getMessage());
            return null;
        });
        this.enforceCacheSizeLimit();
        return data;
    }

    private void enforceCacheSizeLimit() {
        if (this.villagerRestockData.size() >= 5000) {
            Iterator<Map.Entry<UUID, RestockData>> iterator = this.villagerRestockData.entrySet().iterator();
            int CACHE_CLEANUP_BATCH_SIZE = 100;
            int toRemove = Math.max(100, this.villagerRestockData.size() - 5000 + 100);
            for (int i = 0; i < toRemove && iterator.hasNext(); ++i) {
                iterator.next();
                iterator.remove();
            }
            this.plugin.getLogger().fine("Cleaned up " + toRemove + " old restock cache entries");
        }
    }

    private void saveVillagerRestockData(UUID villagerUUID, RestockData data) {
        this.databaseService.executeAsync(() -> {
            try {
                this.databaseService.executeUpdate("INSERT OR REPLACE INTO villager_restock_data (villager_uuid, last_restock_time, restock_count_today, last_reset_day) VALUES (?, ?, ?, ?)", pstmt -> {
                    pstmt.setString(1, villagerUUID.toString());
                    pstmt.setLong(2, data.lastRestockTime);
                    pstmt.setInt(3, data.restockCountToday);
                    pstmt.setLong(4, data.lastResetDay);
                }, "saveVillagerRestockData");
                return null;
            }
            catch (DatabaseException e) {
                this.logDatabaseException("Could not save villager restock data", e, Level.SEVERE);
                return null;
            }
        }, "saveVillagerRestockData");
    }

    public void handleInstantRestock(Villager villager, PlatformService platformService, TaskScheduler taskScheduler) {
        if (!this.instantRestockEnabled) {
            return;
        }
        platformService.executeRegionSafeForEntity(taskScheduler, (Entity)villager, () -> this.restockVillagerTrades(villager));
    }

    public void startRestockMonitor(PlatformService platformService, TaskScheduler taskScheduler) {
        long tradeFreq = this.plugin.getConfig().getLong("performance.trade_monitor_frequency", 120L);
        long restockFreq = Math.max(tradeFreq * 20L, 2400L);
        long initialDelay = 80L;
        taskScheduler.runSyncRepeating(() -> {
            if (!this.customRestockEnabled) {
                return;
            }
            for (Player player : this.plugin.getServer().getOnlinePlayers()) {
                platformService.executeRegionSafe(taskScheduler, player.getLocation(), () -> this.processPlayerRestock(player));
            }
        }, initialDelay, restockFreq);
    }

    private void processPlayerRestock(Player player) {
        int scanRadius = this.plugin.getConfig().getInt("performance.villager_scan_radius", 32);
        try {
            player.getNearbyEntities((double)scanRadius, (double)scanRadius, (double)scanRadius).stream().filter(entity -> entity instanceof Villager).map(entity -> (Villager)entity).forEach(villager -> this.processVillagerRestock((Villager)villager));
        }
        catch (NoSuchMethodError e) {
            for (Entity entity2 : player.getWorld().getEntities()) {
                if (!(entity2 instanceof Villager) || !(entity2.getLocation().distance(player.getLocation()) <= (double)scanRadius)) continue;
                Villager villager2 = (Villager)entity2;
                this.processVillagerRestock(villager2);
            }
        }
    }

    public boolean isCustomRestockEnabled() {
        return this.customRestockEnabled;
    }

    public void setCustomRestockEnabled(boolean enabled) {
        this.customRestockEnabled = enabled;
    }

    public int getRestockIntervalMinutes() {
        return this.restockIntervalMinutes;
    }

    public void setRestockIntervalMinutes(int minutes) {
        this.restockIntervalMinutes = minutes;
    }

    public int getRestockLimitPerDay() {
        return this.restockLimitPerDay;
    }

    public void setRestockLimitPerDay(int limit) {
        this.restockLimitPerDay = limit;
    }

    public boolean isUnlimitedRestock() {
        return this.unlimitedRestock;
    }

    public void setUnlimitedRestock(boolean unlimited) {
        this.unlimitedRestock = unlimited;
    }

    public boolean isInstantRestockEnabled() {
        return this.instantRestockEnabled;
    }

    public void setInstantRestockEnabled(boolean enabled) {
        this.instantRestockEnabled = enabled;
    }

    private void logDatabaseException(String context, Exception e, Level level) {
        DatabaseException dbException = new DatabaseException(context + ": " + e.getMessage(), "RestockService", "database_operation", e);
        this.plugin.getLogger().log(level, dbException.getMessage());
    }

    public void shutdown() {
        this.villagerRestockData.clear();
        this.plugin.getLogger().info("RestockService cache cleared during shutdown");
    }

    public Map<String, Object> getCacheStatistics() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("restockCacheSize", this.villagerRestockData.size());
        stats.put("restockCacheMax", 5000);
        return stats;
    }

    private static class RestockData {
        long lastRestockTime;
        int restockCountToday;
        long lastResetDay;

        RestockData(long lastRestockTime, int restockCountToday, long lastResetDay) {
            this.lastRestockTime = lastRestockTime;
            this.restockCountToday = restockCountToday;
            this.lastResetDay = lastResetDay;
        }
    }
}

