/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import com.alan.VillagerTradeManager.platform.PlatformService;
import com.alan.VillagerTradeManager.scheduler.TaskScheduler;
import com.alan.VillagerTradeManager.services.DatabaseService;
import com.alan.VillagerTradeManager.services.SettingsService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.block.Block;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.entity.Villager;

public class StockService {
    private final VillagerTradeManager plugin;
    private final DatabaseService databaseService;
    private final SettingsService settingsService;
    private boolean stockManagementEnabled;
    private int defaultMaxStock;
    private int restockIntervalMinutes;
    private boolean jobSiteValidationEnabled;
    private boolean automaticRestockingEnabled;
    private final Map<String, StockData> stockCache = new ConcurrentHashMap<String, StockData>();
    private static final int MAX_STOCK_CACHE_SIZE = 10000;
    private static final Map<String, Set<Material>> JOB_SITE_MATERIALS = new HashMap<String, Set<Material>>();

    public StockService(VillagerTradeManager plugin, DatabaseService databaseService, SettingsService settingsService) {
        this.plugin = plugin;
        this.databaseService = databaseService;
        this.settingsService = settingsService;
        this.createStockTable();
        this.loadSettings();
    }

    private void loadSettings() {
        this.stockManagementEnabled = this.settingsService.getBooleanSetting("stock_management_enabled");
        this.defaultMaxStock = this.settingsService.getIntSetting("default_max_stock");
        this.restockIntervalMinutes = this.settingsService.getIntSetting("stock_restock_interval_minutes");
        this.jobSiteValidationEnabled = this.settingsService.getBooleanSetting("job_site_validation_enabled");
        this.automaticRestockingEnabled = this.settingsService.getBooleanSetting("automatic_restocking_enabled");
    }

    private void createStockTable() {
        try {
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS villager_trade_stock (villager_uuid TEXT NOT NULL, profession TEXT NOT NULL, trade_id TEXT NOT NULL, current_stock INTEGER NOT NULL DEFAULT 0, max_stock INTEGER NOT NULL DEFAULT 10, last_used INTEGER NOT NULL DEFAULT 0, last_restocked INTEGER NOT NULL DEFAULT 0, total_used INTEGER NOT NULL DEFAULT 0, PRIMARY KEY (villager_uuid, profession, trade_id))", null, "createStockTable");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_villager_stock_uuid ON villager_trade_stock(villager_uuid)", null, "createStockIndex");
        }
        catch (DatabaseException e) {
            this.logDatabaseException("Could not create stock table", e, Level.SEVERE);
        }
    }

    public boolean isTradeAvailable(Villager villager, String profession, String tradeId) {
        if (!this.stockManagementEnabled) {
            return true;
        }
        StockData stockData = this.getStockData(villager.getUniqueId().toString(), profession, tradeId);
        return stockData.currentStock > 0;
    }

    public void recordTradeUsage(Villager villager, String profession, String tradeId) {
        if (!this.stockManagementEnabled) {
            return;
        }
        String villagerUuid = villager.getUniqueId().toString();
        StockData stockData = this.getStockData(villagerUuid, profession, tradeId);
        if (stockData.currentStock > 0) {
            --stockData.currentStock;
            stockData.lastUsed = System.currentTimeMillis();
            ++stockData.totalUsed;
            this.saveStockData(villagerUuid, profession, tradeId, stockData);
        }
    }

    public int getCurrentStock(Villager villager, String profession, String tradeId) {
        if (!this.stockManagementEnabled) {
            return -1;
        }
        StockData stockData = this.getStockData(villager.getUniqueId().toString(), profession, tradeId);
        return stockData.currentStock;
    }

    public int getMaxStock(Villager villager, String profession, String tradeId) {
        if (!this.stockManagementEnabled) {
            return -1;
        }
        StockData stockData = this.getStockData(villager.getUniqueId().toString(), profession, tradeId);
        return stockData.maxStock;
    }

    public void setMaxStock(Villager villager, String profession, String tradeId, int maxStock) {
        if (!this.stockManagementEnabled) {
            return;
        }
        String villagerUuid = villager.getUniqueId().toString();
        StockData stockData = this.getStockData(villagerUuid, profession, tradeId);
        stockData.maxStock = Math.max(0, maxStock);
        if (stockData.currentStock > stockData.maxStock) {
            stockData.currentStock = stockData.maxStock;
        }
        this.saveStockData(villagerUuid, profession, tradeId, stockData);
    }

    public void restockTrade(Villager villager, String profession, String tradeId) {
        if (!this.stockManagementEnabled) {
            return;
        }
        String villagerUuid = villager.getUniqueId().toString();
        StockData stockData = this.getStockData(villagerUuid, profession, tradeId);
        if (!this.jobSiteValidationEnabled || this.validateJobSite(villager, profession)) {
            stockData.currentStock = stockData.maxStock;
            stockData.lastRestocked = System.currentTimeMillis();
            this.saveStockData(villagerUuid, profession, tradeId, stockData);
        }
    }

    public void processAutomaticRestock(Villager villager, String profession) {
        if (!this.stockManagementEnabled || !this.automaticRestockingEnabled) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        long restockIntervalMs = TimeUnit.MINUTES.toMillis(this.restockIntervalMinutes);
        if (!this.jobSiteValidationEnabled || this.validateJobSite(villager, profession)) {
            String villagerUuid = villager.getUniqueId().toString();
            List<String> tradeIds = this.getTradeIdsForVillager(villagerUuid, profession);
            for (String tradeId : tradeIds) {
                StockData stockData = this.getStockData(villagerUuid, profession, tradeId);
                if (currentTime - stockData.lastRestocked < restockIntervalMs) continue;
                stockData.currentStock = Math.min(stockData.currentStock + 1, stockData.maxStock);
                stockData.lastRestocked = currentTime;
                this.saveStockData(villagerUuid, profession, tradeId, stockData);
            }
        }
    }

    private boolean validateJobSite(Villager villager, String profession) {
        if (!this.jobSiteValidationEnabled) {
            return true;
        }
        Set<Material> requiredMaterials = JOB_SITE_MATERIALS.get(profession);
        if (requiredMaterials == null || requiredMaterials.isEmpty()) {
            return true;
        }
        Location villagerLocation = villager.getLocation();
        int searchRadius = 10;
        for (int x = -searchRadius; x <= searchRadius; ++x) {
            for (int y = -searchRadius; y <= searchRadius; ++y) {
                for (int z = -searchRadius; z <= searchRadius; ++z) {
                    Location checkLocation = villagerLocation.clone().add((double)x, (double)y, (double)z);
                    Block block = checkLocation.getBlock();
                    if (!requiredMaterials.contains(block.getType())) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private List<String> getTradeIdsForVillager(String villagerUuid, String profession) {
        ArrayList<String> tradeIds = new ArrayList();
        try {
            tradeIds = this.databaseService.executeQuery("SELECT trade_id FROM villager_trade_stock WHERE villager_uuid = ? AND profession = ?", rs -> {
                ArrayList<String> ids = new ArrayList<String>();
                while (rs.next()) {
                    ids.add(rs.getString("trade_id"));
                }
                return ids;
            }, "getTradeIdsForVillager");
        }
        catch (DatabaseException e) {
            this.logDatabaseException("Could not get trade IDs for villager", e, Level.WARNING);
        }
        return tradeIds;
    }

    private StockData getStockData(String villagerUuid, String profession, String tradeId) {
        String cacheKey = villagerUuid + ":" + profession + ":" + tradeId;
        StockData data = this.stockCache.get(cacheKey);
        if (data != null) {
            return data;
        }
        data = this.loadStockData(villagerUuid, profession, tradeId);
        this.stockCache.put(cacheKey, data);
        this.enforceCacheSizeLimit();
        return data;
    }

    private StockData loadStockData(String villagerUuid, String profession, String tradeId) {
        try {
            return this.databaseService.executeQuery("SELECT current_stock, max_stock, last_used, last_restocked, total_used FROM villager_trade_stock WHERE villager_uuid = ? AND profession = ? AND trade_id = ?", rs -> {
                if (rs.next()) {
                    return new StockData(rs.getInt("current_stock"), rs.getInt("max_stock"), rs.getLong("last_used"), rs.getLong("last_restocked"), rs.getInt("total_used"));
                }
                return new StockData(this.defaultMaxStock, this.defaultMaxStock, 0L, 0L, 0);
            }, "loadStockData");
        }
        catch (DatabaseException e) {
            this.logDatabaseException("Could not load stock data", e, Level.WARNING);
            return new StockData(this.defaultMaxStock, this.defaultMaxStock, 0L, 0L, 0);
        }
    }

    private void saveStockData(String villagerUuid, String profession, String tradeId, StockData data) {
        String cacheKey = villagerUuid + ":" + profession + ":" + tradeId;
        this.stockCache.put(cacheKey, data);
        this.databaseService.executeAsync(() -> {
            try {
                this.databaseService.executeUpdate("INSERT OR REPLACE INTO villager_trade_stock (villager_uuid, profession, trade_id, current_stock, max_stock, last_used, last_restocked, total_used) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                    pstmt.setString(1, villagerUuid);
                    pstmt.setString(2, profession);
                    pstmt.setString(3, tradeId);
                    pstmt.setInt(4, data.currentStock);
                    pstmt.setInt(5, data.maxStock);
                    pstmt.setLong(6, data.lastUsed);
                    pstmt.setLong(7, data.lastRestocked);
                    pstmt.setInt(8, data.totalUsed);
                }, "saveStockData");
                return null;
            }
            catch (DatabaseException e) {
                this.logDatabaseException("Could not save stock data", e, Level.SEVERE);
                return null;
            }
        }, "saveStockData");
    }

    private void enforceCacheSizeLimit() {
        if (this.stockCache.size() >= 10000) {
            Iterator<Map.Entry<String, StockData>> iterator = this.stockCache.entrySet().iterator();
            int toRemove = Math.max(500, this.stockCache.size() - 10000 + 500);
            for (int i = 0; i < toRemove && iterator.hasNext(); ++i) {
                iterator.next();
                iterator.remove();
            }
            this.plugin.getLogger().fine("Cleaned up " + toRemove + " old stock cache entries");
        }
    }

    public Map<String, Object> getVillagerStockStats(Villager villager) {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        String villagerUuid = villager.getUniqueId().toString();
        try {
            List tradeStocks = this.databaseService.executeQuery("SELECT profession, trade_id, current_stock, max_stock, total_used FROM villager_trade_stock WHERE villager_uuid = ?", rs -> {
                ArrayList stocks = new ArrayList();
                while (rs.next()) {
                    HashMap<String, Object> stock = new HashMap<String, Object>();
                    stock.put("profession", rs.getString("profession"));
                    stock.put("tradeId", rs.getString("trade_id"));
                    stock.put("currentStock", rs.getInt("current_stock"));
                    stock.put("maxStock", rs.getInt("max_stock"));
                    stock.put("totalUsed", rs.getInt("total_used"));
                    stocks.add(stock);
                }
                return stocks;
            }, "getVillagerStockStats");
            stats.put("tradeStocks", tradeStocks);
            stats.put("totalTrades", tradeStocks.size());
        }
        catch (DatabaseException e) {
            this.logDatabaseException("Could not get villager stock stats", e, Level.WARNING);
            stats.put("error", "Failed to load stock statistics");
        }
        return stats;
    }

    public void startRestockScheduler(PlatformService platformService, TaskScheduler taskScheduler) {
        if (!this.stockManagementEnabled || !this.automaticRestockingEnabled) {
            return;
        }
        long frequencyTicks = this.plugin.getConfig().getLong("performance.stock_restock_frequency", 600L);
        long initialDelay = 200L;
        taskScheduler.runSyncRepeating(() -> {
            for (Player player : this.plugin.getServer().getOnlinePlayers()) {
                platformService.executeRegionSafe(taskScheduler, player.getLocation(), () -> this.processPlayerRestock(player));
            }
        }, initialDelay, frequencyTicks);
    }

    private void processPlayerRestock(Player player) {
        int scanRadius = this.plugin.getConfig().getInt("performance.villager_scan_radius", 32);
        try {
            player.getNearbyEntities((double)scanRadius, (double)scanRadius, (double)scanRadius).stream().filter(entity -> entity instanceof Villager).map(entity -> (Villager)entity).forEach(villager -> {
                String profession = this.getVillagerProfession((Villager)villager);
                if (profession != null) {
                    this.processAutomaticRestock((Villager)villager, profession);
                }
            });
        }
        catch (NoSuchMethodError e) {
            for (Entity entity2 : player.getWorld().getEntities()) {
                Villager villager2;
                String profession;
                if (!(entity2 instanceof Villager) || !(entity2.getLocation().distance(player.getLocation()) <= (double)scanRadius) || (profession = this.getVillagerProfession(villager2 = (Villager)entity2)) == null) continue;
                this.processAutomaticRestock(villager2, profession);
            }
        }
    }

    private String getVillagerProfession(Villager villager) {
        if (villager.getProfession() == null) {
            return "none";
        }
        try {
            return villager.getProfession().getKey().getKey().toLowerCase();
        }
        catch (Exception e) {
            try {
                return villager.getProfession().toString().toLowerCase();
            }
            catch (Exception ex) {
                return "none";
            }
        }
    }

    private void logDatabaseException(String context, Exception e, Level level) {
        DatabaseException dbException = new DatabaseException(context + ": " + e.getMessage(), "StockService", "database_operation", e);
        this.plugin.getLogger().log(level, dbException.getMessage());
    }

    public void shutdown() {
        this.stockCache.clear();
        this.plugin.getLogger().info("StockService cache cleared during shutdown");
    }

    public Map<String, Object> getCacheStatistics() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("stockCacheSize", this.stockCache.size());
        stats.put("stockCacheMax", 10000);
        return stats;
    }

    public boolean isStockManagementEnabled() {
        return this.stockManagementEnabled;
    }

    public void setStockManagementEnabled(boolean enabled) {
        this.stockManagementEnabled = enabled;
    }

    public int getDefaultMaxStock() {
        return this.defaultMaxStock;
    }

    public void setDefaultMaxStock(int maxStock) {
        this.defaultMaxStock = maxStock;
    }

    public int getRestockIntervalMinutes() {
        return this.restockIntervalMinutes;
    }

    public void setRestockIntervalMinutes(int minutes) {
        this.restockIntervalMinutes = minutes;
    }

    public boolean isJobSiteValidationEnabled() {
        return this.jobSiteValidationEnabled;
    }

    public void setJobSiteValidationEnabled(boolean enabled) {
        this.jobSiteValidationEnabled = enabled;
    }

    public boolean isAutomaticRestockingEnabled() {
        return this.automaticRestockingEnabled;
    }

    public void setAutomaticRestockingEnabled(boolean enabled) {
        this.automaticRestockingEnabled = enabled;
    }

    static {
        JOB_SITE_MATERIALS.put("armorer", Set.of(Material.BLAST_FURNACE));
        JOB_SITE_MATERIALS.put("butcher", Set.of(Material.SMOKER));
        JOB_SITE_MATERIALS.put("cartographer", Set.of(Material.CARTOGRAPHY_TABLE));
        JOB_SITE_MATERIALS.put("cleric", Set.of(Material.BREWING_STAND));
        JOB_SITE_MATERIALS.put("farmer", Set.of(Material.COMPOSTER));
        JOB_SITE_MATERIALS.put("fisherman", Set.of(Material.BARREL));
        JOB_SITE_MATERIALS.put("fletcher", Set.of(Material.FLETCHING_TABLE));
        JOB_SITE_MATERIALS.put("leatherworker", Set.of(Material.CAULDRON));
        JOB_SITE_MATERIALS.put("librarian", Set.of(Material.LECTERN));
        JOB_SITE_MATERIALS.put("mason", Set.of(Material.STONECUTTER));
        JOB_SITE_MATERIALS.put("nitwit", Set.of());
        JOB_SITE_MATERIALS.put("shepherd", Set.of(Material.LOOM));
        JOB_SITE_MATERIALS.put("toolsmith", Set.of(Material.SMITHING_TABLE));
        JOB_SITE_MATERIALS.put("weaponsmith", Set.of(Material.GRINDSTONE));
        JOB_SITE_MATERIALS.put("unemployed", Set.of());
    }

    private static class StockData {
        int currentStock;
        int maxStock;
        long lastUsed;
        long lastRestocked;
        int totalUsed;

        StockData(int currentStock, int maxStock, long lastUsed, long lastRestocked, int totalUsed) {
            this.currentStock = currentStock;
            this.maxStock = maxStock;
            this.lastUsed = lastUsed;
            this.lastRestocked = lastRestocked;
            this.totalUsed = totalUsed;
        }
    }
}

