/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.cache.AdaptiveCache;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import com.alan.VillagerTradeManager.services.DatabaseService;
import com.alan.VillagerTradeManager.services.ReputationService;
import com.alan.VillagerTradeManager.services.SettingsService;
import com.alan.VillagerTradeManager.services.TradeRegistry;
import com.alan.VillagerTradeManager.services.TradeService;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import org.bukkit.Material;
import org.bukkit.Statistic;
import org.bukkit.World;
import org.bukkit.block.Biome;
import org.bukkit.entity.Player;
import org.bukkit.entity.Villager;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.MerchantRecipe;

public class TradeEnhancementService {
    private final VillagerTradeManager plugin;
    private final DatabaseService databaseService;
    private final SettingsService settingsService;
    private final ReputationService reputationService;
    private final TradeService tradeService;
    private boolean enhancementsEnabled = true;
    private boolean prerequisitesEnabled = true;
    private boolean randomizationEnabled = true;
    private boolean discoveryEnabled = true;
    private boolean tradeChainsEnabled = true;
    private boolean worldVariationsEnabled = true;
    private boolean progressiveUnlockingEnabled = true;
    private double tradeAvailabilityRandomness = 0.15;
    private double priceRandomnessFactor = 0.2;
    private int maxRandomTradesPerVillager = 2;
    private int discoveryHintsPerTrade = 3;
    private double discoveryRevealChance = 0.25;
    private long hintCooldownMinutes = 30L;
    private int maxChainDepth = 5;
    private boolean allowCircularChains = false;
    private Map<String, Double> biomeTradeModifiers = new ConcurrentHashMap<String, Double>();
    private Map<String, Double> dimensionTradeModifiers = new ConcurrentHashMap<String, Double>();
    private final AdaptiveCache<String, TradePrerequisite> prerequisiteCache = new AdaptiveCache(200, 500, 80.0, 1800000L, 0.15);
    private final AdaptiveCache<String, TradeChain> tradeChainCache = new AdaptiveCache(100, 300, 85.0, 3600000L, 0.1);
    private final AdaptiveCache<String, List<String>> playerDiscoveryCache = new AdaptiveCache(150, 400, 75.0, 0x6DDD00L, 0.12);
    private static final String TABLE_TRADE_PREREQUISITES = "trade_prerequisites";
    private static final String TABLE_TRADE_CHAINS = "trade_chains";
    private static final String TABLE_PLAYER_DISCOVERIES = "player_trade_discoveries";
    private static final String TABLE_WORLD_VARIATIONS = "world_trade_variations";
    private static final String TABLE_UNLOCK_PROGRESS = "trade_unlock_progress";

    public TradeEnhancementService(VillagerTradeManager plugin, DatabaseService databaseService, SettingsService settingsService, ReputationService reputationService, TradeService tradeService) {
        this.plugin = plugin;
        this.databaseService = databaseService;
        this.settingsService = settingsService;
        this.reputationService = reputationService;
        this.tradeService = tradeService;
        this.initializeConfiguration();
        this.initializeDatabaseSchema();
        this.loadConfiguration();
        this.initializeDefaultVariations();
    }

    private void initializeDatabaseSchema() {
        try {
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS trade_prerequisites (prerequisite_id TEXT PRIMARY KEY, trade_id TEXT NOT NULL, prerequisite_type TEXT NOT NULL, required_value TEXT, description TEXT, created_at INTEGER NOT NULL)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_prereq_trade ON trade_prerequisites (trade_id)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS trade_chains (chain_id TEXT PRIMARY KEY, parent_trade_id TEXT NOT NULL, child_trade_id TEXT NOT NULL, unlock_order INTEGER NOT NULL, required_completions INTEGER DEFAULT 1, created_at INTEGER NOT NULL)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE UNIQUE INDEX IF NOT EXISTS idx_chain_unique ON trade_chains (parent_trade_id, child_trade_id)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_chain_parent ON trade_chains (parent_trade_id)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_chain_child ON trade_chains (child_trade_id)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS player_trade_discoveries (player_uuid TEXT NOT NULL, trade_id TEXT NOT NULL, discovery_level INTEGER DEFAULT 0, hints_received INTEGER DEFAULT 0, last_hint_time INTEGER, fully_discovered BOOLEAN DEFAULT FALSE, discovered_at INTEGER, PRIMARY KEY (player_uuid, trade_id))", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_discovery_player ON player_trade_discoveries (player_uuid)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_discovery_trade ON player_trade_discoveries (trade_id)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS world_trade_variations (variation_id TEXT PRIMARY KEY, world_name TEXT, biome_name TEXT, dimension_type TEXT, trade_modifier REAL DEFAULT 1.0, availability_modifier REAL DEFAULT 1.0, special_trades TEXT, created_at INTEGER NOT NULL)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_variation_world ON world_trade_variations (world_name)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE INDEX IF NOT EXISTS idx_variation_biome ON world_trade_variations (biome_name)", null, "initializeDatabaseSchema");
            this.databaseService.executeUpdate("CREATE TABLE IF NOT EXISTS trade_unlock_progress (player_uuid TEXT NOT NULL, trade_id TEXT NOT NULL, progress_type TEXT NOT NULL, progress_value INTEGER DEFAULT 0, last_updated INTEGER NOT NULL, PRIMARY KEY (player_uuid, trade_id, progress_type))", null, "initializeDatabaseSchema");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().severe("Failed to initialize trade enhancement database schema: " + e.getMessage());
        }
    }

    private void initializeConfiguration() {
        this.enhancementsEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.enabled", true);
        this.prerequisitesEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.prerequisites.enabled", true);
        this.randomizationEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.randomization.enabled", true);
        this.discoveryEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.discovery.enabled", true);
        this.tradeChainsEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.chains.enabled", true);
        this.worldVariationsEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.world_variations.enabled", true);
        this.progressiveUnlockingEnabled = this.plugin.getConfig().getBoolean("trade_enhancements.progressive_unlocking.enabled", true);
        this.tradeAvailabilityRandomness = this.plugin.getConfig().getDouble("trade_enhancements.randomization.availability_factor", 0.15);
        this.priceRandomnessFactor = this.plugin.getConfig().getDouble("trade_enhancements.randomization.price_factor", 0.2);
        this.maxRandomTradesPerVillager = this.plugin.getConfig().getInt("trade_enhancements.randomization.max_random_trades", 2);
        this.discoveryHintsPerTrade = this.plugin.getConfig().getInt("trade_enhancements.discovery.hints_per_trade", 3);
        this.discoveryRevealChance = this.plugin.getConfig().getDouble("trade_enhancements.discovery.reveal_chance", 0.25);
        this.hintCooldownMinutes = this.plugin.getConfig().getLong("trade_enhancements.discovery.hint_cooldown_minutes", 30L);
        this.maxChainDepth = this.plugin.getConfig().getInt("trade_enhancements.chains.max_depth", 5);
        this.allowCircularChains = this.plugin.getConfig().getBoolean("trade_enhancements.chains.allow_circular", false);
    }

    private void loadConfiguration() {
        try {
            this.databaseService.executeQuery("SELECT * FROM world_trade_variations", rs -> {
                while (rs.next()) {
                    String worldName = rs.getString("world_name");
                    String biomeName = rs.getString("biome_name");
                    String dimensionType = rs.getString("dimension_type");
                    double tradeModifier = rs.getDouble("trade_modifier");
                    if (worldName != null) {
                        this.dimensionTradeModifiers.put(worldName, tradeModifier);
                    }
                    if (biomeName != null) {
                        this.biomeTradeModifiers.put(biomeName, tradeModifier);
                    }
                    if (dimensionType == null) continue;
                    this.dimensionTradeModifiers.put("dimension_" + dimensionType, tradeModifier);
                }
                return null;
            }, "loadConfiguration");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to load trade enhancement configuration: " + e.getMessage());
        }
    }

    private void initializeDefaultVariations() {
        this.biomeTradeModifiers.put("DESERT", 1.2);
        this.biomeTradeModifiers.put("SAVANNA", 0.9);
        this.biomeTradeModifiers.put("TAIGA", 1.1);
        this.biomeTradeModifiers.put("SNOWY_TUNDRA", 1.3);
        this.biomeTradeModifiers.put("MUSHROOM_FIELDS", 0.7);
        this.dimensionTradeModifiers.put("NORMAL", 1.0);
        this.dimensionTradeModifiers.put("NETHER", 1.5);
        this.dimensionTradeModifiers.put("THE_END", 2.0);
        this.saveDefaultVariations();
    }

    private void saveDefaultVariations() {
        long currentTime = System.currentTimeMillis();
        this.biomeTradeModifiers.forEach((biome, modifier) -> {
            try {
                this.databaseService.executeUpdate("INSERT OR REPLACE INTO world_trade_variations (variation_id, world_name, biome_name, dimension_type, trade_modifier, availability_modifier, special_trades, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                    pstmt.setString(1, "biome_" + biome.toLowerCase());
                    pstmt.setString(2, null);
                    pstmt.setString(3, (String)biome);
                    pstmt.setString(4, null);
                    pstmt.setDouble(5, (double)modifier);
                    pstmt.setDouble(6, 1.0);
                    pstmt.setString(7, null);
                    pstmt.setLong(8, currentTime);
                }, "saveDefaultVariations");
            }
            catch (DatabaseException e) {
                this.plugin.getLogger().warning("Failed to save biome variation: " + biome);
            }
        });
        this.dimensionTradeModifiers.forEach((dimension, modifier) -> {
            try {
                this.databaseService.executeUpdate("INSERT OR REPLACE INTO world_trade_variations (variation_id, world_name, biome_name, dimension_type, trade_modifier, availability_modifier, special_trades, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                    pstmt.setString(1, "dimension_" + dimension.toLowerCase());
                    pstmt.setString(2, null);
                    pstmt.setString(3, null);
                    pstmt.setString(4, (String)dimension);
                    pstmt.setDouble(5, (double)modifier);
                    pstmt.setDouble(6, 1.0);
                    pstmt.setString(7, null);
                    pstmt.setLong(8, currentTime);
                }, "saveDefaultVariations");
            }
            catch (DatabaseException e) {
                this.plugin.getLogger().warning("Failed to save dimension variation: " + dimension);
            }
        });
    }

    public boolean isEnhancementsEnabled() {
        return this.enhancementsEnabled;
    }

    public void setEnhancementsEnabled(boolean enabled) {
        this.enhancementsEnabled = enabled;
        this.settingsService.saveSettingToDatabase("trade_enhancements_enabled", String.valueOf(enabled));
    }

    public Map<String, Object> getCacheStatistics() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("prerequisiteCacheSize", this.prerequisiteCache.size());
        stats.put("prerequisiteCacheMax", this.prerequisiteCache.getMaxSize());
        stats.put("tradeChainCacheSize", this.tradeChainCache.size());
        stats.put("tradeChainCacheMax", this.tradeChainCache.getMaxSize());
        stats.put("playerDiscoveryCacheSize", this.playerDiscoveryCache.size());
        stats.put("playerDiscoveryCacheMax", this.playerDiscoveryCache.getMaxSize());
        Map<String, Object> prereqStats = this.prerequisiteCache.getStatistics();
        Map<String, Object> chainStats = this.tradeChainCache.getStatistics();
        Map<String, Object> discoveryStats = this.playerDiscoveryCache.getStatistics();
        stats.put("prerequisiteCacheHitRate", prereqStats.get("hitRate"));
        stats.put("tradeChainCacheHitRate", chainStats.get("hitRate"));
        stats.put("playerDiscoveryCacheHitRate", discoveryStats.get("hitRate"));
        return stats;
    }

    public void shutdown() {
        this.prerequisiteCache.clear();
        this.tradeChainCache.clear();
        this.playerDiscoveryCache.clear();
        this.biomeTradeModifiers.clear();
        this.dimensionTradeModifiers.clear();
        this.plugin.getLogger().info("TradeEnhancementService caches cleared during shutdown");
    }

    public void addTradePrerequisite(String tradeId, PrerequisiteType type, String requiredValue, String description) {
        if (!this.prerequisitesEnabled) {
            return;
        }
        String prerequisiteId = this.generatePrerequisiteId(tradeId, type);
        TradePrerequisite prerequisite = new TradePrerequisite(prerequisiteId, tradeId, type, requiredValue, description);
        try {
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO trade_prerequisites (prerequisite_id, trade_id, prerequisite_type, required_value, description, created_at) VALUES (?, ?, ?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, prerequisiteId);
                pstmt.setString(2, tradeId);
                pstmt.setString(3, type.name());
                pstmt.setString(4, requiredValue);
                pstmt.setString(5, description);
                pstmt.setLong(6, prerequisite.getCreatedAt());
            }, "addTradePrerequisite");
            this.prerequisiteCache.put(prerequisiteId, prerequisite);
            if (this.plugin.getConfig().getBoolean("logging.debug_database", false)) {
                this.plugin.getLogger().fine("Added prerequisite " + prerequisiteId + " for trade: " + tradeId);
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to add trade prerequisite: " + e.getMessage());
        }
    }

    public void removeTradePrerequisite(String prerequisiteId) {
        this.prerequisiteCache.remove(prerequisiteId);
        try {
            this.databaseService.executeUpdate("DELETE FROM trade_prerequisites WHERE prerequisite_id = ?", pstmt -> pstmt.setString(1, prerequisiteId), "removeTradePrerequisite");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to remove trade prerequisite: " + e.getMessage());
        }
    }

    public List<TradePrerequisite> getTradePrerequisites(String tradeId) {
        ArrayList<TradePrerequisite> prerequisites = new ArrayList<TradePrerequisite>();
        try {
            this.databaseService.executeQuery("SELECT * FROM trade_prerequisites WHERE trade_id = ?", rs -> {
                while (rs.next()) {
                    String prerequisiteId = rs.getString("prerequisite_id");
                    String typeStr = rs.getString("prerequisite_type");
                    String requiredValue = rs.getString("required_value");
                    String description = rs.getString("description");
                    try {
                        PrerequisiteType type = PrerequisiteType.valueOf(typeStr);
                        TradePrerequisite prereq = new TradePrerequisite(prerequisiteId, tradeId, type, requiredValue, description);
                        prerequisites.add(prereq);
                        this.prerequisiteCache.put(prerequisiteId, prereq);
                    }
                    catch (IllegalArgumentException e) {
                        this.plugin.getLogger().warning("Invalid prerequisite type: " + typeStr);
                    }
                }
                return null;
            }, tradeId);
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to load trade prerequisites: " + e.getMessage());
        }
        return prerequisites;
    }

    public boolean arePrerequisitesMet(String tradeId, Player player, Villager villager, Map<String, Object> context) {
        if (!this.prerequisitesEnabled) {
            return true;
        }
        List<TradePrerequisite> prerequisites = this.getTradePrerequisites(tradeId);
        if (prerequisites.isEmpty()) {
            return true;
        }
        for (TradePrerequisite prereq : prerequisites) {
            if (prereq.isMet(player, villager, context, this)) continue;
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Prerequisite not met for trade " + tradeId + ": " + prereq.getDescription());
            }
            return false;
        }
        return true;
    }

    public List<String> getUnmetPrerequisites(String tradeId, Player player, Villager villager, Map<String, Object> context) {
        ArrayList<String> unmet = new ArrayList<String>();
        if (!this.prerequisitesEnabled) {
            return unmet;
        }
        List<TradePrerequisite> prerequisites = this.getTradePrerequisites(tradeId);
        for (TradePrerequisite prereq : prerequisites) {
            if (prereq.isMet(player, villager, context, this)) continue;
            unmet.add(prereq.getDescription());
        }
        return unmet;
    }

    private String generatePrerequisiteId(String tradeId, PrerequisiteType type) {
        return tradeId + "_" + type.name().toLowerCase() + "_" + System.currentTimeMillis();
    }

    public boolean isTradeRandomlyAvailable(String tradeId, Player player, Villager villager) {
        boolean available;
        if (!this.randomizationEnabled) {
            return true;
        }
        double baseChance = 1.0 - this.tradeAvailabilityRandomness;
        if (player != null) {
            int reputation = this.reputationService != null ? this.reputationService.getPlayerReputation(player.getUniqueId()) : 0;
            double reputationModifier = Math.min((double)reputation / 1000.0, 0.1);
            baseChance += reputationModifier;
        }
        if (villager != null) {
            double levelModifier = (double)villager.getVillagerLevel() * 0.02;
            baseChance += levelModifier;
        }
        double worldModifier = this.getWorldAvailabilityModifier(player != null ? player.getLocation().getWorld() : null, player != null ? player.getLocation().getBlock().getBiome() : null);
        double random = ThreadLocalRandom.current().nextDouble();
        boolean bl = available = random < Math.min(baseChance *= worldModifier, 1.0);
        if (this.plugin.getConfig().getBoolean("logging.debug_trades", false) && !available) {
            this.plugin.getLogger().fine("Trade " + tradeId + " randomly unavailable (chance: " + String.format("%.2f", baseChance) + ")");
        }
        return available;
    }

    public int applyPriceRandomization(int basePrice, String tradeId, Player player, Villager villager) {
        if (!this.randomizationEnabled || basePrice <= 0) {
            return basePrice;
        }
        double variance = this.priceRandomnessFactor;
        double minMultiplier = 1.0 - variance;
        double maxMultiplier = 1.0 + variance;
        if (player != null && this.reputationService != null) {
            int reputation = this.reputationService.getPlayerReputation(player.getUniqueId());
            double reputationDiscount = Math.min((double)reputation / 2000.0, 0.2);
            minMultiplier = 1.0 - (variance *= 1.0 - reputationDiscount);
            maxMultiplier = 1.0 + variance;
        }
        double multiplier = minMultiplier + ThreadLocalRandom.current().nextDouble() * (maxMultiplier - minMultiplier);
        double worldModifier = this.getWorldPriceModifier(player != null ? player.getLocation().getWorld() : null, player != null ? player.getLocation().getBlock().getBiome() : null);
        int randomizedPrice = (int)Math.max(1L, Math.round((double)basePrice * (multiplier *= worldModifier)));
        if (this.plugin.getConfig().getBoolean("logging.debug_trades", false) && (double)Math.abs(randomizedPrice - basePrice) > (double)basePrice * 0.1) {
            this.plugin.getLogger().fine("Trade " + tradeId + " price randomized: " + basePrice + " -> " + randomizedPrice + " (multiplier: " + String.format("%.2f", multiplier) + ")");
        }
        return randomizedPrice;
    }

    public List<String> generateRandomTrades(String profession, int villagerLevel, Player player) {
        ArrayList<String> randomTrades = new ArrayList<String>();
        if (!this.randomizationEnabled || this.maxRandomTradesPerVillager <= 0) {
            return randomTrades;
        }
        List<String> availableTrades = TradeRegistry.getTradesForProfession(profession);
        if (availableTrades == null || availableTrades.isEmpty()) {
            return randomTrades;
        }
        if ((availableTrades = availableTrades.stream().filter(tradeId -> {
            int tradeTier = this.extractTradeTierFromId((String)tradeId);
            return tradeTier <= villagerLevel + 1;
        }).collect(Collectors.toList())).size() <= 3) {
            return randomTrades;
        }
        int numRandomTrades = Math.min(this.maxRandomTradesPerVillager, Math.max(1, availableTrades.size() / 4));
        Collections.shuffle(availableTrades);
        for (int i = 0; i < numRandomTrades && i < availableTrades.size(); ++i) {
            if (!(ThreadLocalRandom.current().nextDouble() < 0.3)) continue;
            randomTrades.add(availableTrades.get(i));
        }
        if (this.plugin.getConfig().getBoolean("logging.debug_trades", false) && !randomTrades.isEmpty()) {
            this.plugin.getLogger().fine("Generated " + randomTrades.size() + " random trades for " + profession + " level " + villagerLevel);
        }
        return randomTrades;
    }

    private double getWorldAvailabilityModifier(World world, Biome biome) {
        double modifier = 1.0;
        if (biome != null) {
            modifier *= this.biomeTradeModifiers.getOrDefault(biome.getKey().asString(), 1.0).doubleValue();
        }
        if (world != null) {
            String dimensionKey = this.getDimensionKey(world);
            modifier *= this.dimensionTradeModifiers.getOrDefault(dimensionKey, 1.0).doubleValue();
        }
        return modifier;
    }

    private double getWorldPriceModifier(World world, Biome biome) {
        double modifier = 1.0;
        if (biome != null) {
            modifier *= this.biomeTradeModifiers.getOrDefault(biome.getKey().asString(), 1.0).doubleValue();
        }
        if (world != null) {
            String dimensionKey = this.getDimensionKey(world);
            modifier *= this.dimensionTradeModifiers.getOrDefault(dimensionKey, 1.0).doubleValue();
        }
        return modifier;
    }

    private String getDimensionKey(World world) {
        if (world == null) {
            return "NORMAL";
        }
        String worldName = world.getName().toLowerCase();
        if (worldName.contains("nether")) {
            return "NETHER";
        }
        if (worldName.contains("end") || worldName.contains("the_end")) {
            return "THE_END";
        }
        return "NORMAL";
    }

    private int extractTradeTierFromId(String tradeId) {
        if (tradeId == null) {
            return 1;
        }
        if (tradeId.contains("_l5") || tradeId.contains("master")) {
            return 5;
        }
        if (tradeId.contains("_l4") || tradeId.contains("expert")) {
            return 4;
        }
        if (tradeId.contains("_l3") || tradeId.contains("journeyman")) {
            return 3;
        }
        if (tradeId.contains("_l2") || tradeId.contains("apprentice")) {
            return 2;
        }
        return 1;
    }

    public void recordTradeDiscovery(Player player, String tradeId) {
        if (!this.discoveryEnabled || player == null) {
            return;
        }
        UUID playerId = player.getUniqueId();
        String cacheKey = String.valueOf(playerId) + "_" + tradeId;
        try {
            boolean isFullyDiscovered = this.databaseService.executeQuery("SELECT fully_discovered FROM player_trade_discoveries WHERE player_uuid = '" + playerId.toString() + "' AND trade_id = '" + tradeId + "'", rs -> {
                if (rs.next()) {
                    return rs.getBoolean("fully_discovered");
                }
                return false;
            }, "checkFullyDiscovered");
            if (isFullyDiscovered) {
                return;
            }
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO player_trade_discoveries (player_uuid, trade_id, discovery_level, hints_received, last_hint_time, fully_discovered, discovered_at) VALUES (?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, playerId.toString());
                pstmt.setString(2, tradeId);
                pstmt.setInt(3, this.discoveryHintsPerTrade);
                pstmt.setInt(4, this.discoveryHintsPerTrade);
                pstmt.setLong(5, System.currentTimeMillis());
                pstmt.setBoolean(6, true);
                pstmt.setLong(7, System.currentTimeMillis());
            }, "recordTradeDiscovery");
            List<String> discoveredTrades = this.playerDiscoveryCache.get(playerId.toString());
            if (discoveredTrades == null) {
                discoveredTrades = new ArrayList<String>();
            }
            if (!discoveredTrades.contains(tradeId)) {
                discoveredTrades = new ArrayList<String>(discoveredTrades);
                discoveredTrades.add(tradeId);
                this.playerDiscoveryCache.put(playerId.toString(), discoveredTrades);
            }
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Player " + player.getName() + " fully discovered trade: " + tradeId);
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to record trade discovery: " + e.getMessage());
        }
    }

    public Optional<String> provideTradeHint(Player player, String tradeId) {
        if (!this.discoveryEnabled || player == null) {
            return Optional.empty();
        }
        UUID playerId = player.getUniqueId();
        try {
            Map discoveryData = this.databaseService.executeQuery("SELECT discovery_level, hints_received, last_hint_time FROM player_trade_discoveries WHERE player_uuid = '" + playerId.toString() + "' AND trade_id = '" + tradeId + "'", rs -> {
                if (rs.next()) {
                    HashMap<String, Number> data = new HashMap<String, Number>();
                    data.put("discovery_level", rs.getInt("discovery_level"));
                    data.put("hints_received", rs.getInt("hints_received"));
                    data.put("last_hint_time", rs.getLong("last_hint_time"));
                    return data;
                }
                return null;
            }, "checkDiscoveryStatus");
            if (discoveryData != null) {
                long lastHintTime = (Long)discoveryData.get("last_hint_time");
                long timeSinceLastHint = System.currentTimeMillis() - lastHintTime;
                if (timeSinceLastHint < this.hintCooldownMinutes * 60L * 1000L) {
                    return Optional.empty();
                }
            }
            if (ThreadLocalRandom.current().nextDouble() < this.discoveryRevealChance) {
                String hint = this.generateTradeHint(tradeId, discoveryData);
                int newHintsReceived = discoveryData != null ? (Integer)discoveryData.get("hints_received") + 1 : 1;
                this.databaseService.executeUpdate("INSERT OR REPLACE INTO player_trade_discoveries (player_uuid, trade_id, discovery_level, hints_received, last_hint_time, fully_discovered, discovered_at) VALUES (?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                    pstmt.setString(1, playerId.toString());
                    pstmt.setString(2, tradeId);
                    pstmt.setInt(3, Math.min(newHintsReceived, this.discoveryHintsPerTrade));
                    pstmt.setInt(4, newHintsReceived);
                    pstmt.setLong(5, System.currentTimeMillis());
                    pstmt.setBoolean(6, newHintsReceived >= this.discoveryHintsPerTrade);
                    pstmt.setLong(7, discoveryData != null ? (Long)discoveryData.get("discovered_at") : System.currentTimeMillis());
                }, "provideTradeHint");
                if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                    this.plugin.getLogger().fine("Provided hint for trade " + tradeId + " to player " + player.getName());
                }
                return Optional.of(hint);
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to provide trade hint: " + e.getMessage());
        }
        return Optional.empty();
    }

    public List<String> getDiscoveredTrades(Player player) {
        if (!this.discoveryEnabled || player == null) {
            return new ArrayList<String>();
        }
        UUID playerId = player.getUniqueId();
        String cacheKey = playerId.toString();
        List<String> cached = this.playerDiscoveryCache.get(cacheKey);
        if (cached != null) {
            return new ArrayList<String>(cached);
        }
        ArrayList<String> discoveredTrades = new ArrayList<String>();
        try {
            this.databaseService.executeQuery("SELECT trade_id FROM player_trade_discoveries WHERE player_uuid = ? AND fully_discovered = 1", rs -> {
                while (rs.next()) {
                    discoveredTrades.add(rs.getString("trade_id"));
                }
                return null;
            }, playerId.toString());
            this.playerDiscoveryCache.put(cacheKey, new ArrayList<String>(discoveredTrades));
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to load discovered trades: " + e.getMessage());
        }
        return discoveredTrades;
    }

    public double getDiscoveryProgress(Player player, String tradeId) {
        if (!this.discoveryEnabled || player == null) {
            return 0.0;
        }
        try {
            Integer hintsReceived = this.databaseService.executeQuery("SELECT hints_received FROM player_trade_discoveries WHERE player_uuid = '" + player.getUniqueId().toString() + "' AND trade_id = '" + tradeId + "'", rs -> {
                if (rs.next()) {
                    return rs.getInt("hints_received");
                }
                return 0;
            }, "getDiscoveryProgress");
            return Math.min((double)hintsReceived.intValue() / (double)this.discoveryHintsPerTrade, 1.0);
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get discovery progress: " + e.getMessage());
            return 0.0;
        }
    }

    private String generateTradeHint(String tradeId, Map<String, Object> discoveryData) {
        int hintLevel = discoveryData != null ? (Integer)discoveryData.get("hints_received") + 1 : 1;
        TradeRegistry.TradeDefinition tradeDefinition = TradeRegistry.getTradeDefinition(tradeId);
        if (tradeDefinition == null) {
            return "\u00a7e[Discovery] \u00a7fA mysterious trade awaits...";
        }
        switch (hintLevel) {
            case 1: {
                return "\u00a7e[Discovery] \u00a7fYou sense that " + tradeDefinition.getProfession().toLowerCase() + "s might offer something related to " + this.getItemCategoryHint(tradeDefinition) + "...";
            }
            case 2: {
                return "\u00a7e[Discovery] \u00a7fThe " + tradeDefinition.getProfession().toLowerCase() + " seems to be preparing a trade involving " + this.getItemTypeHint(tradeDefinition) + "...";
            }
        }
        return "\u00a7e[Discovery] \u00a7fYou've almost uncovered the " + tradeDefinition.getProfession().toLowerCase() + "'s " + TradeRegistry.getTradeDescription(tradeId).toLowerCase() + " trade!";
    }

    private String getItemCategoryHint(TradeRegistry.TradeDefinition definition) {
        String outputMaterial = definition.getOutputMaterial().name().toLowerCase();
        if (outputMaterial.contains("sword") || outputMaterial.contains("axe") || outputMaterial.contains("pickaxe") || outputMaterial.contains("shovel")) {
            return "tools";
        }
        if (outputMaterial.contains("helmet") || outputMaterial.contains("chestplate") || outputMaterial.contains("leggings") || outputMaterial.contains("boots")) {
            return "armor";
        }
        if (outputMaterial.contains("wool") || outputMaterial.contains("carpet") || outputMaterial.contains("bed") || outputMaterial.contains("banner")) {
            return "decorative items";
        }
        if (outputMaterial.contains("potion") || outputMaterial.contains("book")) {
            return "magical items";
        }
        return "useful items";
    }

    private String getItemTypeHint(TradeRegistry.TradeDefinition definition) {
        Material material = definition.getOutputMaterial();
        String name = material.name().toLowerCase().replace("_", " ");
        if (name.contains("diamond")) {
            return "something shiny";
        }
        if (name.contains("iron")) {
            return "something metallic";
        }
        if (name.contains("gold")) {
            return "something golden";
        }
        if (name.contains("wood") || name.contains("plank")) {
            return "something wooden";
        }
        if (name.contains("stone")) {
            return "something stony";
        }
        return "something special";
    }

    public void createTradeChain(String parentTradeId, String childTradeId, int requiredCompletions) {
        if (!this.tradeChainsEnabled) {
            return;
        }
        String chainId = this.generateChainId(parentTradeId, childTradeId);
        int unlockOrder = this.getNextUnlockOrder(parentTradeId);
        TradeChain chain = new TradeChain(chainId, parentTradeId, childTradeId, unlockOrder, requiredCompletions);
        try {
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO trade_chains (chain_id, parent_trade_id, child_trade_id, unlock_order, required_completions, created_at) VALUES (?, ?, ?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, chainId);
                pstmt.setString(2, parentTradeId);
                pstmt.setString(3, childTradeId);
                pstmt.setInt(4, unlockOrder);
                pstmt.setInt(5, requiredCompletions);
                pstmt.setLong(6, chain.getCreatedAt());
            }, "createTradeChain");
            this.tradeChainCache.put(chainId, chain);
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Created trade chain: " + parentTradeId + " -> " + childTradeId + " (order: " + unlockOrder + ", completions: " + requiredCompletions + ")");
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to create trade chain: " + e.getMessage());
        }
    }

    public List<String> getChainUnlocks(String parentTradeId, Player player) {
        if (!this.tradeChainsEnabled) {
            return new ArrayList<String>();
        }
        ArrayList<String> unlockedTrades = new ArrayList<String>();
        try {
            List chains = this.databaseService.executeQuery("SELECT * FROM trade_chains WHERE parent_trade_id = '" + parentTradeId + "' ORDER BY unlock_order", rs -> {
                ArrayList<TradeChain> result = new ArrayList<TradeChain>();
                while (rs.next()) {
                    String chainId = rs.getString("chain_id");
                    String childTradeId = rs.getString("child_trade_id");
                    int unlockOrder = rs.getInt("unlock_order");
                    int requiredCompletions = rs.getInt("required_completions");
                    TradeChain chain = new TradeChain(chainId, parentTradeId, childTradeId, unlockOrder, requiredCompletions);
                    result.add(chain);
                    this.tradeChainCache.put(chainId, chain);
                }
                return result;
            }, "getChainUnlocks");
            for (TradeChain chain : chains) {
                if (!this.isChainCompleted(chain, player)) continue;
                unlockedTrades.add(chain.getChildTradeId());
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get chain unlocks: " + e.getMessage());
        }
        return unlockedTrades;
    }

    public boolean isChainCompleted(TradeChain chain, Player player) {
        if (player == null) {
            return false;
        }
        try {
            Integer completions = this.databaseService.executeQuery("SELECT COUNT(*) as completions FROM player_trade_discoveries WHERE player_uuid = '" + player.getUniqueId().toString() + "' AND trade_id = '" + chain.getParentTradeId() + "' AND fully_discovered = 1", rs -> rs.next() ? rs.getInt("completions") : 0, "checkChainCompletion");
            return completions >= chain.getRequiredCompletions();
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to check chain completion: " + e.getMessage());
            return false;
        }
    }

    public double getChainProgress(TradeChain chain, Player player) {
        if (player == null) {
            return 0.0;
        }
        try {
            Integer currentCompletions = this.databaseService.executeQuery("SELECT COUNT(*) as completions FROM player_trade_discoveries WHERE player_uuid = '" + player.getUniqueId().toString() + "' AND trade_id = '" + chain.getParentTradeId() + "' AND fully_discovered = 1", rs -> rs.next() ? rs.getInt("completions") : 0, "getChainProgress");
            return Math.min((double)currentCompletions.intValue() / (double)chain.getRequiredCompletions(), 1.0);
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get chain progress: " + e.getMessage());
            return 0.0;
        }
    }

    public List<TradeChain> getAllTradeChains() {
        ArrayList<TradeChain> chains = new ArrayList<TradeChain>();
        try {
            this.databaseService.executeQuery("SELECT * FROM trade_chains ORDER BY parent_trade_id, unlock_order", rs -> {
                while (rs.next()) {
                    String chainId = rs.getString("chain_id");
                    String parentTradeId = rs.getString("parent_trade_id");
                    String childTradeId = rs.getString("child_trade_id");
                    int unlockOrder = rs.getInt("unlock_order");
                    int requiredCompletions = rs.getInt("required_completions");
                    TradeChain chain = new TradeChain(chainId, parentTradeId, childTradeId, unlockOrder, requiredCompletions);
                    chains.add(chain);
                }
                return null;
            }, "getAllTradeChains");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get all trade chains: " + e.getMessage());
        }
        return chains;
    }

    public void removeTradeChain(String chainId) {
        this.tradeChainCache.remove(chainId);
        try {
            this.databaseService.executeUpdate("DELETE FROM trade_chains WHERE chain_id = ?", pstmt -> pstmt.setString(1, chainId), "removeTradeChain");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to remove trade chain: " + e.getMessage());
        }
    }

    private String generateChainId(String parentTradeId, String childTradeId) {
        return parentTradeId + "_to_" + childTradeId + "_" + System.currentTimeMillis();
    }

    private int getNextUnlockOrder(String parentTradeId) {
        try {
            Integer maxOrder = this.databaseService.executeQuery("SELECT MAX(unlock_order) as max_order FROM trade_chains WHERE parent_trade_id = '" + parentTradeId + "'", rs -> rs.next() ? rs.getInt("max_order") : 0, "getNextUnlockOrder");
            return maxOrder + 1;
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get next unlock order: " + e.getMessage());
            return 1;
        }
    }

    public MerchantRecipe applyWorldVariations(MerchantRecipe originalRecipe, String tradeId, Player player, Villager villager) {
        Biome biome;
        World world;
        if (!this.worldVariationsEnabled || originalRecipe == null) {
            return originalRecipe;
        }
        double priceModifier = 1.0;
        double availabilityModifier = 1.0;
        List<Object> specialTrades = new ArrayList();
        Object object = player != null ? player.getWorld() : (world = villager != null ? villager.getWorld() : null);
        Biome biome2 = player != null ? player.getLocation().getBlock().getBiome() : (biome = villager != null ? villager.getLocation().getBlock().getBiome() : null);
        if (biome != null) {
            priceModifier *= this.biomeTradeModifiers.getOrDefault(biome.getKey().asString(), 1.0).doubleValue();
        }
        if (world != null) {
            String dimensionKey = this.getDimensionKey(world);
            priceModifier *= this.dimensionTradeModifiers.getOrDefault(dimensionKey, 1.0).doubleValue();
        }
        if (world != null && biome != null) {
            specialTrades = this.getSpecialTradesForLocation(world.getName(), biome.getKey().asString());
        }
        if (Math.abs(priceModifier - 1.0) > 0.05) {
            int basePrice = this.calculateBasePrice(originalRecipe);
            int modifiedPrice = (int)Math.max(1L, Math.round((double)basePrice * priceModifier));
            MerchantRecipe modifiedRecipe = this.createModifiedRecipe(originalRecipe, modifiedPrice);
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Applied world variation to trade " + tradeId + ": price " + basePrice + " -> " + modifiedPrice + " (modifier: " + String.format("%.2f", priceModifier) + ")");
            }
            return modifiedRecipe;
        }
        return originalRecipe;
    }

    public List<String> getSpecialTradesForLocation(String worldName, String biomeName) {
        ArrayList<String> specialTrades = new ArrayList<String>();
        try {
            this.databaseService.executeQuery("SELECT special_trades FROM world_trade_variations WHERE (world_name = '" + worldName + "' OR world_name IS NULL)  AND (biome_name = '" + biomeName + "' OR biome_name IS NULL) AND special_trades IS NOT NULL", rs -> {
                while (rs.next()) {
                    String[] trades;
                    String tradesStr = rs.getString("special_trades");
                    if (tradesStr == null || tradesStr.trim().isEmpty()) continue;
                    for (String trade : trades = tradesStr.split(",")) {
                        String trimmed = trade.trim();
                        if (trimmed.isEmpty() || specialTrades.contains(trimmed)) continue;
                        specialTrades.add(trimmed);
                    }
                }
                return null;
            }, "getSpecialTradesForLocation");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get special trades for location: " + e.getMessage());
        }
        return specialTrades;
    }

    public void addWorldVariation(String worldName, String biomeName, String dimensionType, double tradeModifier, double availabilityModifier, List<String> specialTrades) {
        if (!this.worldVariationsEnabled) {
            return;
        }
        String variationId = this.generateVariationId(worldName, biomeName, dimensionType);
        try {
            String specialTradesStr = specialTrades != null && !specialTrades.isEmpty() ? String.join((CharSequence)",", specialTrades) : null;
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO world_trade_variations (variation_id, world_name, biome_name, dimension_type, trade_modifier, availability_modifier, special_trades, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, variationId);
                pstmt.setString(2, worldName);
                pstmt.setString(3, biomeName);
                pstmt.setString(4, dimensionType);
                pstmt.setDouble(5, tradeModifier);
                pstmt.setDouble(6, availabilityModifier);
                pstmt.setString(7, specialTradesStr);
                pstmt.setLong(8, System.currentTimeMillis());
            }, "addWorldVariation");
            if (biomeName != null) {
                this.biomeTradeModifiers.put(biomeName, tradeModifier);
            }
            if (dimensionType != null) {
                this.dimensionTradeModifiers.put(dimensionType, tradeModifier);
            }
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Added world variation: " + variationId + " (trade_mod: " + tradeModifier + ", avail_mod: " + availabilityModifier + ")");
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to add world variation: " + e.getMessage());
        }
    }

    public Map<String, Object> getWorldVariationStats() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("biomeModifiers", this.biomeTradeModifiers.size());
        stats.put("dimensionModifiers", this.dimensionTradeModifiers.size());
        try {
            Integer totalVariations = this.databaseService.executeQuery("SELECT COUNT(*) as count FROM world_trade_variations", rs -> rs.next() ? rs.getInt("count") : 0, "getWorldVariationStats");
            stats.put("totalVariations", totalVariations);
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get world variation stats: " + e.getMessage());
            stats.put("totalVariations", 0);
        }
        return stats;
    }

    private String generateVariationId(String worldName, String biomeName, String dimensionType) {
        StringBuilder id = new StringBuilder("variation_");
        if (worldName != null) {
            id.append(worldName).append("_");
        }
        if (biomeName != null) {
            id.append(biomeName).append("_");
        }
        if (dimensionType != null) {
            id.append(dimensionType).append("_");
        }
        id.append(System.currentTimeMillis());
        return id.toString();
    }

    private MerchantRecipe createModifiedRecipe(MerchantRecipe originalRecipe, int newPrice) {
        MerchantRecipe modifiedRecipe = this.createMerchantRecipe(originalRecipe.getResult(), this.getRecipeUses(originalRecipe), this.getRecipeMaxUses(originalRecipe), this.getRecipeHasExperienceReward(originalRecipe), this.getRecipeVillagerExperience(originalRecipe), 0.0f);
        ArrayList<ItemStack> ingredients = new ArrayList<ItemStack>(originalRecipe.getIngredients());
        for (int i = 0; i < ingredients.size(); ++i) {
            ItemStack ingredient = (ItemStack)ingredients.get(i);
            if (ingredient.getType() != Material.EMERALD) continue;
            ingredients.set(i, new ItemStack(Material.EMERALD, Math.min(newPrice, 64)));
            break;
        }
        modifiedRecipe.setIngredients(ingredients);
        this.setRecipeDemand(modifiedRecipe, 0);
        this.setRecipeSpecialPrice(modifiedRecipe, 0);
        this.setRecipePriceMultiplier(modifiedRecipe, 0.0f);
        return modifiedRecipe;
    }

    private int getRecipeUses(MerchantRecipe recipe) {
        try {
            return recipe.getUses();
        }
        catch (NoSuchMethodError e) {
            return 0;
        }
    }

    private int getRecipeMaxUses(MerchantRecipe recipe) {
        try {
            return recipe.getMaxUses();
        }
        catch (NoSuchMethodError e) {
            return 99999;
        }
    }

    private boolean getRecipeHasExperienceReward(MerchantRecipe recipe) {
        try {
            return recipe.hasExperienceReward();
        }
        catch (NoSuchMethodError e) {
            return false;
        }
    }

    private int getRecipeVillagerExperience(MerchantRecipe recipe) {
        try {
            return recipe.getVillagerExperience();
        }
        catch (NoSuchMethodError e) {
            return 0;
        }
    }

    private void setRecipeDemand(MerchantRecipe recipe, int demand) {
        try {
            recipe.setDemand(demand);
        }
        catch (NoSuchMethodError noSuchMethodError) {
            // empty catch block
        }
    }

    private void setRecipeSpecialPrice(MerchantRecipe recipe, int specialPrice) {
        try {
            recipe.setSpecialPrice(specialPrice);
        }
        catch (NoSuchMethodError noSuchMethodError) {
            // empty catch block
        }
    }

    private void setRecipePriceMultiplier(MerchantRecipe recipe, float multiplier) {
        try {
            recipe.setPriceMultiplier(multiplier);
        }
        catch (NoSuchMethodError noSuchMethodError) {
            // empty catch block
        }
    }

    private MerchantRecipe createMerchantRecipe(ItemStack result, int uses, int maxUses, boolean experienceReward, int villagerExperience, float priceMultiplier) {
        try {
            return new MerchantRecipe(result, uses, maxUses, experienceReward, villagerExperience, priceMultiplier);
        }
        catch (NoSuchMethodError e) {
            try {
                return new MerchantRecipe(result, uses, maxUses, experienceReward, villagerExperience, priceMultiplier);
            }
            catch (Exception ex) {
                return new MerchantRecipe(result, maxUses);
            }
        }
    }

    private int calculateBasePrice(MerchantRecipe recipe) {
        List ingredients = recipe.getIngredients();
        int totalEmeralds = 0;
        for (ItemStack ingredient : ingredients) {
            if (ingredient.getType() != Material.EMERALD) continue;
            totalEmeralds += ingredient.getAmount();
        }
        return Math.max(1, totalEmeralds);
    }

    public boolean isTradeUnlockedForPlayer(String tradeId, Player player, Villager villager) {
        if (!this.progressiveUnlockingEnabled) {
            return true;
        }
        if (player == null) {
            return true;
        }
        if (!this.checkExperienceUnlock(tradeId, player)) {
            return false;
        }
        if (!this.checkAchievementUnlock(tradeId, player)) {
            return false;
        }
        return villager == null || this.checkVillagerRelationshipUnlock(tradeId, player, villager);
    }

    public void recordUnlockProgress(String tradeId, Player player, String progressType, int progressAmount) {
        if (!this.progressiveUnlockingEnabled || player == null) {
            return;
        }
        try {
            String progressKey = String.valueOf(player.getUniqueId()) + "_" + tradeId + "_" + progressType;
            Integer currentProgress = this.databaseService.executeQuery("SELECT progress_value FROM trade_unlock_progress WHERE player_uuid = '" + player.getUniqueId().toString() + "' AND trade_id = '" + tradeId + "' AND progress_type = '" + progressType + "'", rs -> rs.next() ? rs.getInt("progress_value") : 0, "getCurrentProgress");
            int newProgress = currentProgress + progressAmount;
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO trade_unlock_progress (player_uuid, trade_id, progress_type, progress_value, last_updated) VALUES (?, ?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, player.getUniqueId().toString());
                pstmt.setString(2, tradeId);
                pstmt.setString(3, progressType);
                pstmt.setInt(4, newProgress);
                pstmt.setLong(5, System.currentTimeMillis());
            }, "recordUnlockProgress");
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Recorded unlock progress for " + player.getName() + ": " + tradeId + " (" + progressType + ": " + newProgress + ")");
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to record unlock progress: " + e.getMessage());
        }
    }

    public Map<String, Integer> getUnlockProgress(String tradeId, Player player) {
        HashMap<String, Integer> progress = new HashMap<String, Integer>();
        if (!this.progressiveUnlockingEnabled || player == null) {
            return progress;
        }
        try {
            this.databaseService.executeQuery("SELECT progress_type, progress_value FROM trade_unlock_progress WHERE player_uuid = '" + player.getUniqueId().toString() + "' AND trade_id = '" + tradeId + "'", rs -> {
                while (rs.next()) {
                    String type = rs.getString("progress_type");
                    int value = rs.getInt("progress_value");
                    progress.put(type, value);
                }
                return null;
            }, "getUnlockProgress");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get unlock progress: " + e.getMessage());
        }
        return progress;
    }

    public List<String> getUnlockedTradesForPlayer(Player player) {
        ArrayList<String> unlockedTrades = new ArrayList<String>();
        if (!this.progressiveUnlockingEnabled || player == null) {
            return unlockedTrades;
        }
        try {
            this.databaseService.executeQuery("SELECT DISTINCT trade_id FROM trade_unlock_progress WHERE player_uuid = '" + player.getUniqueId().toString() + "'", rs -> {
                while (rs.next()) {
                    unlockedTrades.add(rs.getString("trade_id"));
                }
                return null;
            }, "getUnlockedTradesForPlayer");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to get unlocked trades: " + e.getMessage());
        }
        return unlockedTrades;
    }

    public void setUnlockCriteria(String tradeId, String criteriaType, String criteriaValue) {
        if (!this.progressiveUnlockingEnabled) {
            return;
        }
        try {
            this.databaseService.executeUpdate("INSERT OR REPLACE INTO trade_unlock_criteria (trade_id, criteria_type, criteria_value, created_at) VALUES (?, ?, ?, ?)", pstmt -> {
                pstmt.setString(1, tradeId);
                pstmt.setString(2, criteriaType);
                pstmt.setString(3, criteriaValue);
                pstmt.setLong(4, System.currentTimeMillis());
            }, "setUnlockCriteria");
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to set unlock criteria: " + e.getMessage());
        }
    }

    private boolean checkExperienceUnlock(String tradeId, Player player) {
        try {
            Integer requiredLevels = this.databaseService.executeQuery("SELECT criteria_value FROM trade_unlock_criteria WHERE trade_id = '" + tradeId + "' AND criteria_type = 'experience_levels'", rs -> rs.next() ? Integer.valueOf(Integer.parseInt(rs.getString("criteria_value"))) : null, "checkExperienceUnlock");
            if (requiredLevels != null) {
                return player.getLevel() >= requiredLevels;
            }
        }
        catch (DatabaseException | NumberFormatException e) {
            this.plugin.getLogger().warning("Failed to check experience unlock: " + e.getMessage());
        }
        return true;
    }

    private boolean checkAchievementUnlock(String tradeId, Player player) {
        try {
            String requiredAchievement = this.databaseService.executeQuery("SELECT criteria_value FROM trade_unlock_criteria WHERE trade_id = '" + tradeId + "' AND criteria_type = 'achievement'", rs -> rs.next() ? rs.getString("criteria_value") : null, "checkAchievementUnlock");
            if (requiredAchievement != null) {
                try {
                    Statistic statistic = Statistic.valueOf((String)requiredAchievement.toUpperCase());
                    return player.getStatistic(statistic) > 0;
                }
                catch (IllegalArgumentException e) {
                    return false;
                }
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().warning("Failed to check achievement unlock: " + e.getMessage());
        }
        return true;
    }

    private boolean checkVillagerRelationshipUnlock(String tradeId, Player player, Villager villager) {
        try {
            Integer requiredReputation = this.databaseService.executeQuery("SELECT criteria_value FROM trade_unlock_criteria WHERE trade_id = '" + tradeId + "' AND criteria_type = 'villager_reputation'", rs -> rs.next() ? Integer.valueOf(Integer.parseInt(rs.getString("criteria_value"))) : null, "checkVillagerRelationshipUnlock");
            if (requiredReputation != null && this.reputationService != null) {
                return this.reputationService.getPlayerReputation(player.getUniqueId()) >= requiredReputation;
            }
        }
        catch (DatabaseException | NumberFormatException e) {
            this.plugin.getLogger().warning("Failed to check villager relationship unlock: " + e.getMessage());
        }
        return true;
    }

    public MerchantRecipe applyAllEnhancements(MerchantRecipe originalRecipe, String tradeId, Player player, Villager villager, Map<String, Object> context) {
        int basePrice;
        int randomizedPrice;
        if (!this.enhancementsEnabled) {
            return originalRecipe;
        }
        MerchantRecipe enhancedRecipe = originalRecipe;
        if (this.worldVariationsEnabled) {
            enhancedRecipe = this.applyWorldVariations(enhancedRecipe, tradeId, player, villager);
        }
        if (this.randomizationEnabled && enhancedRecipe == originalRecipe && (randomizedPrice = this.applyPriceRandomization(basePrice = this.calculateBasePrice(originalRecipe), tradeId, player, villager)) != basePrice) {
            enhancedRecipe = this.createModifiedRecipe(originalRecipe, randomizedPrice);
        }
        return enhancedRecipe;
    }

    public boolean isTradeAvailable(String tradeId, Player player, Villager villager, Map<String, Object> context) {
        if (!this.enhancementsEnabled) {
            return true;
        }
        if (this.prerequisitesEnabled && !this.arePrerequisitesMet(tradeId, player, villager, context)) {
            return false;
        }
        if (this.progressiveUnlockingEnabled && !this.isTradeUnlockedForPlayer(tradeId, player, villager)) {
            return false;
        }
        return !this.randomizationEnabled || this.isTradeRandomlyAvailable(tradeId, player, villager);
    }

    public void onTradeCompleted(String tradeId, Player player, Villager villager) {
        if (!this.enhancementsEnabled) {
            return;
        }
        if (this.discoveryEnabled) {
            this.recordTradeDiscovery(player, tradeId);
        }
        if (this.progressiveUnlockingEnabled) {
            this.recordUnlockProgress(tradeId, player, "trades_completed", 1);
        }
        if (this.tradeChainsEnabled) {
            List<String> newUnlocks = this.getChainUnlocks(tradeId, player);
            for (String newTradeId : newUnlocks) {
                if (player == null) continue;
                player.sendMessage("\u00a7a[Trade Chain] \u00a7fYou've unlocked a new trade: " + TradeRegistry.getTradeDescription(newTradeId));
            }
        }
    }

    public static enum PrerequisiteType {
        TIME_BASED("Time-based requirements (play time)"),
        REPUTATION_BASED("Reputation-based requirements"),
        LEVEL_BASED("Villager level requirements"),
        QUEST_BASED("Quest completion requirements"),
        WORLD_BASED("World/dimension requirements"),
        ACHIEVEMENT_BASED("Minecraft achievement/statistics requirements");

        private final String description;

        private PrerequisiteType(String description) {
            this.description = description;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static class TradePrerequisite {
        private final String prerequisiteId;
        private final String tradeId;
        private final PrerequisiteType type;
        private final String requiredValue;
        private final String description;
        private final long createdAt;

        public TradePrerequisite(String prerequisiteId, String tradeId, PrerequisiteType type, String requiredValue, String description) {
            this.prerequisiteId = prerequisiteId;
            this.tradeId = tradeId;
            this.type = type;
            this.requiredValue = requiredValue;
            this.description = description;
            this.createdAt = System.currentTimeMillis();
        }

        public boolean isMet(Player player, Villager villager, Map<String, Object> context, TradeEnhancementService service) {
            switch (this.type.ordinal()) {
                case 0: {
                    return this.checkTimeBasedPrerequisite(player);
                }
                case 1: {
                    return this.checkReputationBasedPrerequisite(player, service);
                }
                case 2: {
                    return this.checkLevelBasedPrerequisite(villager);
                }
                case 3: {
                    return this.checkQuestBasedPrerequisite(player, context);
                }
                case 4: {
                    return this.checkWorldBasedPrerequisite(player);
                }
                case 5: {
                    return this.checkAchievementBasedPrerequisite(player);
                }
            }
            return true;
        }

        private boolean checkTimeBasedPrerequisite(Player player) {
            try {
                long requiredHours = Long.parseLong(this.requiredValue);
                long playerPlayTime = player.getStatistic(Statistic.PLAY_ONE_MINUTE) / 72000;
                return playerPlayTime >= requiredHours;
            }
            catch (NumberFormatException e) {
                return false;
            }
        }

        private boolean checkReputationBasedPrerequisite(Player player, TradeEnhancementService service) {
            if (service.reputationService == null) {
                return true;
            }
            try {
                int requiredReputation = Integer.parseInt(this.requiredValue);
                return service.reputationService.getPlayerReputation(player.getUniqueId()) >= requiredReputation;
            }
            catch (NumberFormatException e) {
                return false;
            }
        }

        private boolean checkLevelBasedPrerequisite(Villager villager) {
            try {
                int requiredLevel = Integer.parseInt(this.requiredValue);
                return villager.getVillagerLevel() >= requiredLevel;
            }
            catch (NumberFormatException e) {
                return false;
            }
        }

        private boolean checkQuestBasedPrerequisite(Player player, Map<String, Object> context) {
            Object completedQuests = context.get("completed_quests");
            if (completedQuests instanceof Collection) {
                return ((Collection)completedQuests).contains(this.requiredValue);
            }
            return false;
        }

        private boolean checkWorldBasedPrerequisite(Player player) {
            return player.getWorld().getName().equals(this.requiredValue);
        }

        private boolean checkAchievementBasedPrerequisite(Player player) {
            try {
                Statistic statistic = Statistic.valueOf((String)this.requiredValue.toUpperCase());
                return player.getStatistic(statistic) > 0;
            }
            catch (IllegalArgumentException e) {
                return false;
            }
        }

        public String getPrerequisiteId() {
            return this.prerequisiteId;
        }

        public String getTradeId() {
            return this.tradeId;
        }

        public PrerequisiteType getType() {
            return this.type;
        }

        public String getRequiredValue() {
            return this.requiredValue;
        }

        public String getDescription() {
            return this.description;
        }

        public long getCreatedAt() {
            return this.createdAt;
        }
    }

    public static class TradeChain {
        private final String chainId;
        private final String parentTradeId;
        private final String childTradeId;
        private final int unlockOrder;
        private final int requiredCompletions;
        private final long createdAt;

        public TradeChain(String chainId, String parentTradeId, String childTradeId, int unlockOrder, int requiredCompletions) {
            this.chainId = chainId;
            this.parentTradeId = parentTradeId;
            this.childTradeId = childTradeId;
            this.unlockOrder = unlockOrder;
            this.requiredCompletions = requiredCompletions;
            this.createdAt = System.currentTimeMillis();
        }

        public String getChainId() {
            return this.chainId;
        }

        public String getParentTradeId() {
            return this.parentTradeId;
        }

        public String getChildTradeId() {
            return this.childTradeId;
        }

        public int getUnlockOrder() {
            return this.unlockOrder;
        }

        public int getRequiredCompletions() {
            return this.requiredCompletions;
        }

        public long getCreatedAt() {
            return this.createdAt;
        }
    }
}

