/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import com.alan.VillagerTradeManager.scheduler.TaskScheduler;
import com.alan.VillagerTradeManager.services.DatabaseService;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.entity.Villager;

public class VillagerLevelService {
    private final VillagerTradeManager plugin;
    private final DatabaseService databaseService;
    private final TaskScheduler taskScheduler;
    private static final int[] LEVEL_XP_REQUIREMENTS = new int[]{0, 10, 70, 150, 250};
    private static final Map<Integer, Integer> TRADE_XP_REWARDS = new HashMap<Integer, Integer>();
    private final Map<UUID, VillagerProgression> progressionCache = new ConcurrentHashMap<UUID, VillagerProgression>();
    private static final long CACHE_TIMEOUT_MS = 300000L;
    private int[] xpRequirements;
    private int[] xpRewards;
    private boolean enabled;
    private boolean hideUnavailableTrades;
    private double xpMultiplier;
    private final Map<UUID, VillagerProgression> cache;

    public VillagerLevelService(VillagerTradeManager plugin, DatabaseService databaseService, TaskScheduler taskScheduler) {
        this.plugin = plugin;
        this.databaseService = databaseService;
        this.taskScheduler = taskScheduler;
        this.cache = new ConcurrentHashMap<UUID, VillagerProgression>();
        this.loadConfiguration();
        this.startCacheCleanupTask();
    }

    private void loadConfiguration() {
        this.xpRequirements = new int[6];
        this.xpRequirements[1] = this.plugin.getConfig().getInt("level_progression.xp_requirements.level_1", 0);
        this.xpRequirements[2] = this.plugin.getConfig().getInt("level_progression.xp_requirements.level_2", 100);
        this.xpRequirements[3] = this.plugin.getConfig().getInt("level_progression.xp_requirements.level_3", 250);
        this.xpRequirements[4] = this.plugin.getConfig().getInt("level_progression.xp_requirements.level_4", 500);
        this.xpRequirements[5] = this.plugin.getConfig().getInt("level_progression.xp_requirements.level_5", 1000);
        this.xpRewards = new int[6];
        this.xpRewards[1] = this.plugin.getConfig().getInt("level_progression.xp_rewards.tier_1", 5);
        this.xpRewards[2] = this.plugin.getConfig().getInt("level_progression.xp_rewards.tier_2", 10);
        this.xpRewards[3] = this.plugin.getConfig().getInt("level_progression.xp_rewards.tier_3", 15);
        this.xpRewards[4] = this.plugin.getConfig().getInt("level_progression.xp_rewards.tier_4", 20);
        this.xpRewards[5] = this.plugin.getConfig().getInt("level_progression.xp_rewards.tier_5", 25);
        this.enabled = this.plugin.getConfig().getBoolean("level_progression.enabled", true);
        this.hideUnavailableTrades = this.plugin.getConfig().getBoolean("level_progression.trade_availability.hide_unavailable_trades", true);
        this.xpMultiplier = this.plugin.getConfig().getDouble("level_progression.bonuses.xp_multiplier", 1.0);
    }

    private void startCacheCleanupTask() {
        this.taskScheduler.runAsyncRepeating(() -> {
            long currentTime = System.currentTimeMillis();
            this.cache.entrySet().removeIf(entry -> currentTime - ((VillagerProgression)entry.getValue()).getLastUpdated() > 300000L);
        }, 6000L, 6000L);
    }

    public VillagerProgression getVillagerProgression(Villager villager) throws DatabaseException {
        String currentProfession;
        UUID villagerId = villager.getUniqueId();
        VillagerProgression cached = this.progressionCache.get(villagerId);
        if (cached != null && !cached.isExpired()) {
            return cached;
        }
        VillagerProgression progression = this.loadVillagerProgression(villagerId);
        if (progression == null) {
            String profession = this.getVillagerProfession(villager);
            long now = System.currentTimeMillis();
            progression = new VillagerProgression(villagerId, 0, 1, 0, profession, now);
            this.saveVillagerProgression(progression);
        }
        if (!(currentProfession = this.getVillagerProfession(villager)).equals(progression.getProfession())) {
            progression.setProfession(currentProfession);
            this.saveVillagerProgression(progression);
        }
        this.cache.put(villagerId, progression);
        return progression;
    }

    public void awardXpForTrade(Villager villager, int tradeTier) throws DatabaseException {
        if (!this.enabled) {
            return;
        }
        VillagerProgression progression = this.getVillagerProgression(villager);
        int baseXpReward = this.xpRewards[Math.min(tradeTier, this.xpRewards.length - 1)];
        int xpReward = (int)Math.round((double)baseXpReward * this.xpMultiplier);
        progression.setCurrentXp(progression.getCurrentXp() + xpReward);
        progression.setTotalXpEarned(progression.getTotalXpEarned() + xpReward);
        progression.setLastUpdated(System.currentTimeMillis());
        this.checkLevelAdvancement(progression);
        this.saveVillagerProgression(progression);
        this.cache.put(villager.getUniqueId(), progression);
        if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
            this.plugin.getLogger().fine("Awarded " + xpReward + " XP to villager " + String.valueOf(villager.getUniqueId()) + " (Level: " + progression.getCurrentLevel() + ", XP: " + progression.getCurrentXp() + ")");
        }
    }

    private void checkLevelAdvancement(VillagerProgression progression) {
        int oldLevel = progression.getCurrentLevel();
        while (progression.getCurrentLevel() < this.xpRequirements.length - 1 && progression.getCurrentXp() >= this.xpRequirements[progression.getCurrentLevel()]) {
            progression.setCurrentLevel(progression.getCurrentLevel() + 1);
            if (!this.plugin.getConfig().getBoolean("logging.debug_trades", false)) continue;
            this.plugin.getLogger().fine("Villager " + String.valueOf(progression.getVillagerId()) + " advanced to level " + progression.getCurrentLevel());
        }
        if (progression.getCurrentLevel() > oldLevel) {
            try {
                Villager villager;
                Villager villager2 = villager = this.plugin.getServer().getEntity(progression.getVillagerId()) instanceof Villager ? (Villager)this.plugin.getServer().getEntity(progression.getVillagerId()) : null;
                if (villager != null) {
                    if (this.plugin.getCustomEventListener() != null) {
                        this.plugin.getCustomEventListener().fireVillagerLevelUpEvent(villager, oldLevel, progression.getCurrentLevel(), progression.getCurrentXp() - this.xpRequirements[oldLevel], progression.getTotalXpEarned());
                    }
                    if (this.plugin.getTradeHookManager() != null) {
                        this.plugin.getTradeHookManager().fireLevelUpEvent(villager, oldLevel, progression.getCurrentLevel(), progression.getCurrentXp() - this.xpRequirements[oldLevel], progression.getTotalXpEarned());
                    }
                }
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Failed to fire level up event/hook: " + e.getMessage());
            }
        }
    }

    public boolean isTradeTierAvailable(Villager villager, int tradeTier) throws DatabaseException {
        if (!this.isLevelProgressionEnabled()) {
            return true;
        }
        VillagerProgression progression = this.getVillagerProgression(villager);
        return tradeTier <= progression.getCurrentLevel();
    }

    public int getMaxTradeTier(Villager villager) throws DatabaseException {
        if (!this.isLevelProgressionEnabled()) {
            return 5;
        }
        VillagerProgression progression = this.getVillagerProgression(villager);
        return Math.min(progression.getCurrentLevel(), 5);
    }

    public void setVillagerLevel(Villager villager, int level) throws DatabaseException {
        if (level < 1 || level > 5) {
            throw new IllegalArgumentException("Level must be between 1 and 5");
        }
        VillagerProgression progression = this.getVillagerProgression(villager);
        progression.setCurrentLevel(level);
        progression.setCurrentXp(this.xpRequirements[level]);
        progression.setLastUpdated(System.currentTimeMillis());
        this.saveVillagerProgression(progression);
        this.cache.put(villager.getUniqueId(), progression);
    }

    public void addVillagerXp(Villager villager, int xpAmount) throws DatabaseException {
        if (xpAmount < 0) {
            throw new IllegalArgumentException("XP amount must be positive");
        }
        VillagerProgression progression = this.getVillagerProgression(villager);
        progression.setCurrentXp(progression.getCurrentXp() + xpAmount);
        progression.setTotalXpEarned(progression.getTotalXpEarned() + xpAmount);
        progression.setLastUpdated(System.currentTimeMillis());
        this.checkLevelAdvancement(progression);
        this.saveVillagerProgression(progression);
        this.progressionCache.put(villager.getUniqueId(), progression);
    }

    public void resetVillagerProgression(Villager villager) throws DatabaseException {
        UUID villagerId = villager.getUniqueId();
        String profession = this.getVillagerProfession(villager);
        long now = System.currentTimeMillis();
        VillagerProgression progression = new VillagerProgression(villagerId, 0, 1, 0, profession, now);
        this.saveVillagerProgression(progression);
        this.progressionCache.put(villagerId, progression);
    }

    private VillagerProgression loadVillagerProgression(UUID villagerId) throws DatabaseException {
        return this.databaseService.executeQuery("SELECT current_xp, current_level, total_xp_earned, profession, last_updated, created_at FROM villager_progression WHERE villager_uuid = '" + villagerId.toString() + "'", rs -> {
            if (rs.next()) {
                return new VillagerProgression(villagerId, rs.getInt("current_xp"), rs.getInt("current_level"), rs.getInt("total_xp_earned"), rs.getString("profession"), rs.getLong("last_updated"));
            }
            return null;
        }, "loadVillagerProgression");
    }

    private void saveVillagerProgression(VillagerProgression progression) throws DatabaseException {
        long now = System.currentTimeMillis();
        this.databaseService.executeUpdate("INSERT OR REPLACE INTO villager_progression (villager_uuid, current_xp, current_level, total_xp_earned, profession, last_updated, created_at) VALUES (?, ?, ?, ?, ?, ?, ?)", pstmt -> {
            pstmt.setString(1, progression.getVillagerId().toString());
            pstmt.setInt(2, progression.getCurrentXp());
            pstmt.setInt(3, progression.getCurrentLevel());
            pstmt.setInt(4, progression.getTotalXpEarned());
            pstmt.setString(5, progression.getProfession());
            pstmt.setLong(6, now);
            pstmt.setLong(7, progression.getLastUpdated() == 0L ? now : progression.getLastUpdated());
        }, "saveVillagerProgression");
    }

    private String getVillagerProfession(Villager villager) {
        if (villager.getProfession() == null) {
            return "none";
        }
        try {
            return villager.getProfession().getKey().getKey().toLowerCase();
        }
        catch (Exception e) {
            try {
                return villager.getProfession().toString().toLowerCase();
            }
            catch (Exception ex) {
                return "none";
            }
        }
    }

    private boolean isLevelProgressionEnabled() {
        return this.plugin.getConfig().getBoolean("level_progression.enabled", true);
    }

    public int[] getLevelXpRequirements() {
        return (int[])this.xpRequirements.clone();
    }

    public int getXpRewardForTradeTier(int tier) {
        return this.xpRewards[Math.min(tier, this.xpRewards.length - 1)];
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public void cleanupCache() {
        this.cache.entrySet().removeIf(entry -> ((VillagerProgression)entry.getValue()).isExpired());
    }

    public Map<String, Object> getCacheStatistics() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("progressionCacheSize", this.cache.size());
        stats.put("progressionCacheMaxAge", 300000L);
        return stats;
    }

    static {
        TRADE_XP_REWARDS.put(1, 1);
        TRADE_XP_REWARDS.put(2, 2);
        TRADE_XP_REWARDS.put(3, 3);
        TRADE_XP_REWARDS.put(4, 5);
        TRADE_XP_REWARDS.put(5, 8);
    }

    public static class VillagerProgression {
        private final UUID villagerId;
        private int currentXp;
        private int currentLevel;
        private int totalXpEarned;
        private String profession;
        private long lastUpdated;

        public VillagerProgression(UUID villagerId, int currentXp, int currentLevel, int totalXpEarned, String profession, long lastUpdated) {
            this.villagerId = villagerId;
            this.currentXp = currentXp;
            this.currentLevel = currentLevel;
            this.totalXpEarned = totalXpEarned;
            this.profession = profession;
            this.lastUpdated = lastUpdated;
        }

        public UUID getVillagerId() {
            return this.villagerId;
        }

        public int getCurrentXp() {
            return this.currentXp;
        }

        public int getCurrentLevel() {
            return this.currentLevel;
        }

        public int getTotalXpEarned() {
            return this.totalXpEarned;
        }

        public String getProfession() {
            return this.profession;
        }

        public long getLastUpdated() {
            return this.lastUpdated;
        }

        public void setCurrentXp(int currentXp) {
            this.currentXp = currentXp;
        }

        public void setCurrentLevel(int currentLevel) {
            this.currentLevel = currentLevel;
        }

        public void setTotalXpEarned(int totalXpEarned) {
            this.totalXpEarned = totalXpEarned;
        }

        public void setProfession(String profession) {
            this.profession = profession;
        }

        public void setLastUpdated(long lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        public boolean isExpired() {
            return System.currentTimeMillis() - this.lastUpdated > 300000L;
        }

        public int getXpForNextLevel() {
            return Integer.MAX_VALUE;
        }

        public int getTotalXpForNextLevel() {
            return Integer.MAX_VALUE;
        }
    }
}

