/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.services;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.exception.DatabaseException;
import com.alan.VillagerTradeManager.model.CustomTradeDefinition;
import com.alan.VillagerTradeManager.scheduler.TaskScheduler;
import com.alan.VillagerTradeManager.services.DatabaseService;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Material;
import org.bukkit.entity.Entity;
import org.bukkit.entity.WanderingTrader;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.MerchantRecipe;

public class WanderingTraderService {
    private final VillagerTradeManager plugin;
    private final DatabaseService databaseService;
    private final TaskScheduler taskScheduler;
    private final Map<String, CustomTradeDefinition> customTrades = new ConcurrentHashMap<String, CustomTradeDefinition>();
    private boolean enabled = true;
    private boolean replaceVanillaTrades = false;
    private int maxCustomTrades = 6;

    public WanderingTraderService(VillagerTradeManager plugin, DatabaseService databaseService, TaskScheduler taskScheduler) {
        this.plugin = plugin;
        this.databaseService = databaseService;
        this.taskScheduler = taskScheduler;
        this.loadConfiguration();
        this.loadCustomTrades();
    }

    private void loadConfiguration() {
        this.enabled = this.plugin.getConfig().getBoolean("wandering_traders.enable_custom_trades", true);
        this.replaceVanillaTrades = this.plugin.getConfig().getBoolean("wandering_traders.replace_vanilla_trades", false);
        this.maxCustomTrades = this.plugin.getConfig().getInt("wandering_traders.max_custom_trades", 6);
    }

    private void loadCustomTrades() {
        try {
            this.databaseService.executeQuery("SELECT * FROM wandering_trader_trades", rs -> {
                this.customTrades.clear();
                while (rs.next()) {
                    String itemType = rs.getString("item_type");
                    int quantity = rs.getInt("item_quantity");
                    int emeraldCost = rs.getInt("emerald_cost");
                    long createdAt = rs.getLong("created_at");
                    CustomTradeDefinition trade = new CustomTradeDefinition(itemType, quantity, emeraldCost, createdAt);
                    this.customTrades.put(trade.getTradeId(), trade);
                }
                return null;
            }, "loadCustomTrades");
            if (this.plugin.getConfig().getBoolean("logging.startup_verbose", false)) {
                this.plugin.getLogger().info("Loaded " + this.customTrades.size() + " wandering trader custom trades");
            }
        }
        catch (DatabaseException e) {
            this.plugin.getLogger().severe("Failed to load wandering trader custom trades: " + e.getMessage());
        }
    }

    public boolean addTrade(String itemType, int quantity, int emeraldCost) throws DatabaseException {
        Material material = Material.matchMaterial((String)itemType);
        if (material == null) {
            throw new IllegalArgumentException("Invalid item type: " + itemType);
        }
        if (quantity <= 0 || quantity > 64) {
            throw new IllegalArgumentException("Quantity must be between 1 and 64");
        }
        if (emeraldCost <= 0 || emeraldCost > 64) {
            throw new IllegalArgumentException("Emerald cost must be between 1 and 64");
        }
        CustomTradeDefinition trade = new CustomTradeDefinition(material.name(), quantity, emeraldCost);
        this.databaseService.executeUpdate("INSERT OR REPLACE INTO wandering_trader_trades (trade_id, item_type, item_quantity, emerald_cost, created_at) VALUES (?, ?, ?, ?, ?)", pstmt -> {
            pstmt.setString(1, trade.getTradeId());
            pstmt.setString(2, trade.getItemType());
            pstmt.setInt(3, trade.getQuantity());
            pstmt.setInt(4, trade.getEmeraldCost());
            pstmt.setLong(5, trade.getCreatedAt());
        }, "addWanderingTraderTrade");
        this.customTrades.put(trade.getTradeId(), trade);
        return true;
    }

    public boolean removeTrade(String itemType) throws DatabaseException {
        ArrayList<String> toRemove = new ArrayList<String>();
        for (Map.Entry<String, CustomTradeDefinition> entry : this.customTrades.entrySet()) {
            if (!entry.getValue().getItemType().equalsIgnoreCase(itemType)) continue;
            toRemove.add(entry.getKey());
        }
        if (toRemove.isEmpty()) {
            return false;
        }
        for (String tradeId : toRemove) {
            this.databaseService.executeUpdate("DELETE FROM wandering_trader_trades WHERE trade_id = ?", pstmt -> pstmt.setString(1, tradeId), "removeWanderingTraderTrade");
            this.customTrades.remove(tradeId);
        }
        return true;
    }

    public int clearAllTrades() throws DatabaseException {
        int count = this.customTrades.size();
        this.databaseService.executeUpdate("DELETE FROM wandering_trader_trades", null, "clearWanderingTraderTrades");
        this.customTrades.clear();
        return count;
    }

    public Collection<CustomTradeDefinition> getAllTrades() {
        return new ArrayList<CustomTradeDefinition>(this.customTrades.values());
    }

    public void applyCustomTrades(WanderingTrader trader) {
        if (!this.enabled || this.customTrades.isEmpty()) {
            return;
        }
        this.taskScheduler.runSyncForEntity((Entity)trader, () -> {
            ArrayList<MerchantRecipe> recipes = new ArrayList<MerchantRecipe>();
            if (!this.replaceVanillaTrades) {
                recipes.addAll(trader.getRecipes());
            }
            int added = 0;
            for (CustomTradeDefinition trade : this.customTrades.values()) {
                if (added >= this.maxCustomTrades) break;
                try {
                    MerchantRecipe recipe = this.createRecipe(trade);
                    recipes.add(recipe);
                    ++added;
                }
                catch (Exception e) {
                    this.plugin.getLogger().warning("Failed to create wandering trader recipe for " + trade.getItemType() + ": " + e.getMessage());
                }
            }
            trader.setRecipes(recipes);
            if (this.plugin.getConfig().getBoolean("logging.debug_trades", false)) {
                this.plugin.getLogger().fine("Applied " + added + " custom trades to wandering trader");
            }
        });
    }

    private MerchantRecipe createRecipe(CustomTradeDefinition trade) {
        Material material = Material.valueOf((String)trade.getItemType());
        ItemStack result = new ItemStack(material, trade.getQuantity());
        MerchantRecipe recipe = new MerchantRecipe(result, 999);
        recipe.addIngredient(new ItemStack(Material.EMERALD, trade.getEmeraldCost()));
        return recipe;
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public boolean isReplaceVanillaTrades() {
        return this.replaceVanillaTrades;
    }

    public void setReplaceVanillaTrades(boolean replaceVanillaTrades) {
        this.replaceVanillaTrades = replaceVanillaTrades;
    }

    public int getMaxCustomTrades() {
        return this.maxCustomTrades;
    }

    public void setMaxCustomTrades(int maxCustomTrades) {
        this.maxCustomTrades = maxCustomTrades;
    }
}

