/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.validation;

import com.alan.VillagerTradeManager.services.SimpleTradeMapper;
import com.alan.VillagerTradeManager.services.TradeRegistry;
import com.alan.VillagerTradeManager.validation.ValidationResult;
import java.util.List;
import java.util.regex.Pattern;

public class InputValidator {
    public static final int MIN_EMERALD_COST = 1;
    public static final int MAX_EMERALD_COST = 64;
    public static final int MIN_ENCHANTMENT_LEVEL = 1;
    public static final int MAX_ENCHANTMENT_LEVEL = 5;
    public static final int MIN_RESTOCK_MINUTES = 1;
    public static final int MAX_RESTOCK_MINUTES = 1440;
    public static final int MIN_RESTOCK_LIMIT = 1;
    public static final int MAX_RESTOCK_LIMIT = 1000;
    private static final Pattern ALPHANUMERIC_PATTERN = Pattern.compile("^[a-zA-Z0-9_]+$");
    private static final Pattern PROFESSION_PATTERN = Pattern.compile("^[a-zA-Z]+$");
    private static final Pattern TRADE_ID_PATTERN = Pattern.compile("^[a-zA-Z0-9_]+$");

    public static ValidationResult<String> validateProfession(String profession) {
        if (profession == null || profession.trim().isEmpty()) {
            return ValidationResult.failure("Profession cannot be null or empty");
        }
        String sanitized = InputValidator.sanitizeProfession(profession);
        if (!PROFESSION_PATTERN.matcher(sanitized).matches()) {
            return ValidationResult.failure("Profession contains invalid characters. Only letters allowed.");
        }
        if (!TradeRegistry.isValidProfession(sanitized)) {
            String validProfessions = String.join((CharSequence)", ", TradeRegistry.getAllProfessions().stream().map(TradeRegistry::getProfessionDisplayName).toList());
            return ValidationResult.failure("Invalid profession '" + sanitized + "'. Valid professions: " + validProfessions);
        }
        return ValidationResult.success(sanitized);
    }

    public static ValidationResult<String> validateTradeId(String tradeId) {
        if (tradeId == null || tradeId.trim().isEmpty()) {
            return ValidationResult.failure("Trade ID cannot be null or empty");
        }
        String sanitized = InputValidator.sanitizeTradeId(tradeId);
        if (!TRADE_ID_PATTERN.matcher(sanitized).matches()) {
            return ValidationResult.failure("Trade ID contains invalid characters. Only alphanumeric characters and underscores allowed.");
        }
        if (TradeRegistry.getTradeDefinition(sanitized) == null) {
            return ValidationResult.failure("Invalid trade ID '" + sanitized + "'. Use /villagertrade listtrades <profession> to see available trades.");
        }
        return ValidationResult.success(sanitized);
    }

    public static ValidationResult<String> validateTradeIdForProfession(String tradeId, String profession) {
        ValidationResult<String> professionResult = InputValidator.validateProfession(profession);
        if (professionResult.isInvalid()) {
            return ValidationResult.failure(professionResult.getErrors());
        }
        ValidationResult<String> tradeIdResult = InputValidator.validateTradeId(tradeId);
        if (tradeIdResult.isInvalid()) {
            return ValidationResult.failure(tradeIdResult.getErrors());
        }
        String validProfession = professionResult.getValue();
        String validTradeId = tradeIdResult.getValue();
        List<String> professionTrades = TradeRegistry.getTradesForProfession(validProfession);
        if (!professionTrades.contains(validTradeId)) {
            return ValidationResult.failure("Trade ID '" + validTradeId + "' does not belong to profession '" + validProfession + "'");
        }
        return ValidationResult.success(validTradeId);
    }

    public static ValidationResult<String> validateSimpleTradeName(String simpleName, String profession) {
        List<String> professionTrades;
        boolean hasSimpleMapper;
        String[] parts;
        ValidationResult<String> professionResult = InputValidator.validateProfession(profession);
        if (professionResult.isInvalid()) {
            return ValidationResult.failure(professionResult.getErrors());
        }
        if (simpleName == null || simpleName.trim().isEmpty()) {
            return ValidationResult.failure("Trade name cannot be null or empty");
        }
        String sanitized = InputValidator.sanitizeSimpleName(simpleName);
        String validProfession = professionResult.getValue();
        if ("librarian".equals(validProfession) && sanitized.contains("_") && (parts = sanitized.split("_", 2)).length == 2) {
            String enchantName = parts[0].toLowerCase();
            String levelPart = parts[1];
            if (InputValidator.isValidEnchantmentName(enchantName)) {
                ValidationResult<Integer> levelResult = InputValidator.validateEnchantmentLevel(levelPart);
                if (levelResult.isValid()) {
                    return ValidationResult.success(sanitized);
                }
                return ValidationResult.failure("Invalid enchantment level '" + levelPart + "' for '" + enchantName + "'. Level must be between 1 and 5.");
            }
        }
        if ((hasSimpleMapper = SimpleTradeMapper.hasProfession(validProfession)) && SimpleTradeMapper.hasSimpleName(validProfession, sanitized)) {
            return ValidationResult.success(sanitized);
        }
        if (TradeRegistry.getTradeDefinition(sanitized) != null && (professionTrades = TradeRegistry.getTradesForProfession(validProfession)).contains(sanitized)) {
            return ValidationResult.success(sanitized);
        }
        return ValidationResult.failure("Invalid trade '" + sanitized + "' for profession '" + validProfession + "'. Use /villagertrade listtrades " + validProfession + " to see available trades.");
    }

    public static ValidationResult<Integer> validateEmeraldCost(String costString) {
        ValidationResult<Integer> numberResult = InputValidator.validatePositiveInteger(costString, "emerald cost");
        if (numberResult.isInvalid()) {
            return numberResult;
        }
        int cost = numberResult.getValue();
        if (cost < 1) {
            return ValidationResult.failure("Emerald cost must be at least 1");
        }
        if (cost > 64) {
            return ValidationResult.failure("Emerald cost cannot exceed 64 (Minecraft stack limit)");
        }
        return ValidationResult.success(cost);
    }

    public static ValidationResult<Integer> validateEnchantmentLevel(String levelString) {
        ValidationResult<Integer> numberResult = InputValidator.validatePositiveInteger(levelString, "enchantment level");
        if (numberResult.isInvalid()) {
            return numberResult;
        }
        int level = numberResult.getValue();
        if (level < 1) {
            return ValidationResult.failure("Enchantment level must be at least 1");
        }
        if (level > 5) {
            return ValidationResult.failure("Enchantment level cannot exceed 5");
        }
        return ValidationResult.success(level);
    }

    public static ValidationResult<Integer> validateRestockMinutes(String minutesString) {
        ValidationResult<Integer> numberResult = InputValidator.validatePositiveInteger(minutesString, "restock interval");
        if (numberResult.isInvalid()) {
            return numberResult;
        }
        int minutes = numberResult.getValue();
        if (minutes < 1) {
            return ValidationResult.failure("Restock interval must be at least 1 minute(s)");
        }
        if (minutes > 1440) {
            return ValidationResult.failure("Restock interval cannot exceed 1440 minutes (24 hours)");
        }
        return ValidationResult.success(minutes);
    }

    public static ValidationResult<Integer> validateRestockLimit(String limitString) {
        ValidationResult<Integer> numberResult = InputValidator.validatePositiveInteger(limitString, "restock limit");
        if (numberResult.isInvalid()) {
            return numberResult;
        }
        int limit = numberResult.getValue();
        if (limit < 1) {
            return ValidationResult.failure("Restock limit must be at least 1");
        }
        if (limit > 1000) {
            return ValidationResult.failure("Restock limit cannot exceed 1000");
        }
        return ValidationResult.success(limit);
    }

    public static ValidationResult<Integer> validatePositiveInteger(String input, String fieldName) {
        if (input == null || input.trim().isEmpty()) {
            return ValidationResult.failure(fieldName + " cannot be null or empty");
        }
        String sanitized = input.trim();
        try {
            int value = Integer.parseInt(sanitized);
            if (value <= 0) {
                return ValidationResult.failure(fieldName + " must be a positive number");
            }
            return ValidationResult.success(value);
        }
        catch (NumberFormatException e) {
            return ValidationResult.failure("Invalid " + fieldName + " '" + sanitized + "'. Please enter a valid number.");
        }
    }

    public static ValidationResult<Boolean> validateBoolean(String input, String fieldName) {
        String sanitized;
        if (input == null || input.trim().isEmpty()) {
            return ValidationResult.failure(fieldName + " cannot be null or empty");
        }
        switch (sanitized = input.trim().toLowerCase()) {
            case "on": 
            case "true": 
            case "1": 
            case "yes": 
            case "enable": 
            case "enabled": {
                return ValidationResult.success(true);
            }
            case "off": 
            case "false": 
            case "0": 
            case "no": 
            case "disable": 
            case "disabled": {
                return ValidationResult.success(false);
            }
        }
        return ValidationResult.failure("Invalid " + fieldName + " '" + input + "'. Use 'on' or 'off'.");
    }

    public static ValidationResult<String[]> validateArgumentCount(String[] args, int minArgs, int maxArgs, String commandName) {
        if (args.length < minArgs) {
            return ValidationResult.failure("Insufficient arguments for " + commandName + ". Expected at least " + minArgs + ", got " + args.length);
        }
        if (args.length > maxArgs) {
            return ValidationResult.failure("Too many arguments for " + commandName + ". Expected at most " + maxArgs + ", got " + args.length);
        }
        return ValidationResult.success(args);
    }

    private static String sanitizeProfession(String profession) {
        return profession.trim().toLowerCase();
    }

    private static String sanitizeTradeId(String tradeId) {
        return tradeId.trim();
    }

    private static String sanitizeSimpleName(String simpleName) {
        return simpleName.trim().toLowerCase().replace(" ", "_");
    }

    private static boolean isValidEnchantmentName(String enchantName) {
        return enchantName.matches("^(efficiency|fortune|silk_touch|unbreaking|sharpness|smite|bane_of_arthropods|knockback|fire_aspect|looting|sweeping|protection|fire_protection|feather_falling|blast_protection|projectile_protection|respiration|aqua_affinity|thorns|depth_strider|frost_walker|mending|curse_of_vanishing|curse_of_binding|power|punch|flame|infinity|mending|loyalty|impaling|riptide|channeling|multishot|piercing|quick_charge|soul_speed|swift_sneak)$");
    }

    public static ValidationResult<SetPriceArgs> validateSetPriceArgs(String[] args) {
        if (args.length < 3 || args.length > 5) {
            return ValidationResult.failure("Invalid number of arguments. Use: /villagertrade setprice <profession> <trade> <cost> [price]");
        }
        ValidationResult<String> professionResult = InputValidator.validateProfession(args[0]);
        if (professionResult.isInvalid()) {
            return ValidationResult.failure(professionResult.getErrors());
        }
        ValidationResult<String> tradeResult = InputValidator.validateSimpleTradeName(args[1], professionResult.getValue());
        if (tradeResult.isInvalid()) {
            return ValidationResult.failure(tradeResult.getErrors());
        }
        ValidationResult<Integer> costResult = InputValidator.validateEmeraldCost(args[2]);
        if (costResult.isInvalid()) {
            return ValidationResult.failure(costResult.getErrors());
        }
        Integer price = null;
        if (args.length >= 4) {
            ValidationResult<Integer> priceResult = InputValidator.validateEmeraldCost(args[3]);
            if (priceResult.isInvalid()) {
                return ValidationResult.failure(priceResult.getErrors());
            }
            price = priceResult.getValue();
        }
        return ValidationResult.success(new SetPriceArgs(professionResult.getValue(), tradeResult.getValue(), costResult.getValue(), price));
    }

    public static class SetPriceArgs {
        public final String profession;
        public final String tradeId;
        public final int emeraldCost;
        public final Integer price;

        public SetPriceArgs(String profession, String tradeId, int emeraldCost, Integer price) {
            this.profession = profession;
            this.tradeId = tradeId;
            this.emeraldCost = emeraldCost;
            this.price = price;
        }
    }
}

