/*
 * Decompiled with CFR 0.152.
 */
package com.alan.VillagerTradeManager.database;

import com.alan.VillagerTradeManager.VillagerTradeManager;
import com.alan.VillagerTradeManager.database.DatabaseConfig;
import com.alan.VillagerTradeManager.database.DatabaseException;
import com.alan.VillagerTradeManager.database.DatabaseManager;
import com.alan.VillagerTradeManager.database.DatabaseProvider;
import com.alan.VillagerTradeManager.database.DatabaseType;
import com.alan.VillagerTradeManager.database.MigrationOptions;
import com.alan.VillagerTradeManager.database.MigrationProgress;
import com.alan.VillagerTradeManager.database.MigrationResult;
import com.alan.VillagerTradeManager.database.MigrationValidator;
import com.alan.VillagerTradeManager.database.SQLiteDatabaseProvider;
import com.alan.VillagerTradeManager.database.SupabaseDatabaseProvider;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

public class DatabaseMigrationManager {
    private final VillagerTradeManager plugin;
    private final DatabaseManager databaseManager;
    private volatile boolean migrationInProgress = false;
    private volatile MigrationProgress currentProgress;
    private static final List<String> MIGRATION_TABLES = Arrays.asList("settings", "custom_prices", "villager_restock_data");

    public DatabaseMigrationManager(VillagerTradeManager plugin, DatabaseManager databaseManager) {
        this.plugin = plugin;
        this.databaseManager = databaseManager;
    }

    public CompletableFuture<MigrationResult> migrateDatabase(DatabaseType sourceType, DatabaseType targetType, MigrationOptions options) {
        if (this.migrationInProgress) {
            return CompletableFuture.completedFuture(MigrationResult.failure("Migration already in progress"));
        }
        if (sourceType == targetType) {
            return CompletableFuture.completedFuture(MigrationResult.failure("Source and target databases must be different"));
        }
        this.migrationInProgress = true;
        this.currentProgress = new MigrationProgress();
        return CompletableFuture.supplyAsync(() -> {
            try {
                MigrationResult result;
                this.plugin.getLogger().info("Starting database migration: " + String.valueOf((Object)sourceType) + " -> " + String.valueOf((Object)targetType));
                if (!this.validateConnections(sourceType, targetType)) {
                    MigrationResult migrationResult = MigrationResult.failure("Database connection validation failed");
                    return migrationResult;
                }
                if (options.createBackup) {
                    this.createBackup(sourceType);
                }
                if ((result = this.executeMigration(sourceType, targetType, options)).isSuccess()) {
                    this.plugin.getLogger().info("Database migration completed successfully");
                } else {
                    this.plugin.getLogger().severe("Database migration failed: " + result.getErrorMessage());
                    if (options.rollbackOnFailure) {
                        this.rollbackMigration(sourceType, targetType);
                    }
                }
                MigrationResult migrationResult = result;
                return migrationResult;
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Migration failed with exception: " + e.getMessage());
                e.printStackTrace();
                MigrationResult migrationResult = MigrationResult.failure("Migration failed: " + e.getMessage());
                return migrationResult;
            }
            finally {
                this.migrationInProgress = false;
                this.currentProgress = null;
            }
        });
    }

    private boolean validateConnections(DatabaseType sourceType, DatabaseType targetType) {
        try {
            DatabaseProvider sourceProvider = this.createProvider(sourceType);
            DatabaseProvider targetProvider = this.createProvider(targetType);
            boolean sourceValid = sourceProvider.testConnection().get();
            boolean targetValid = targetProvider.testConnection().get();
            sourceProvider.close();
            targetProvider.close();
            if (!sourceValid) {
                this.plugin.getLogger().severe("Source database connection test failed");
                return false;
            }
            if (!targetValid) {
                this.plugin.getLogger().severe("Target database connection test failed");
                return false;
            }
            return true;
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Connection validation failed: " + e.getMessage());
            return false;
        }
    }

    private void createBackup(DatabaseType sourceType) {
        this.plugin.getLogger().info("Creating backup of source database: " + String.valueOf((Object)sourceType));
        if (sourceType == DatabaseType.SQLITE) {
            this.createSQLiteBackup();
        } else if (sourceType == DatabaseType.SUPABASE) {
            this.createSupabaseBackup();
        }
    }

    private void createSQLiteBackup() {
        try {
            File dbFile = new File(this.plugin.getDataFolder(), "villager_data.db");
            if (!dbFile.exists()) {
                this.plugin.getLogger().warning("SQLite database file not found, skipping backup");
                return;
            }
            File backupDir = new File(this.plugin.getDataFolder(), "backups");
            if (!backupDir.exists()) {
                backupDir.mkdirs();
            }
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            File backupFile = new File(backupDir, "villager_data_backup_" + timestamp + ".db");
            Files.copy(dbFile.toPath(), backupFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            this.plugin.getLogger().info("SQLite backup created: " + backupFile.getName());
            this.cleanupOldBackups(backupDir, 5);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to create SQLite backup: " + e.getMessage());
        }
    }

    private void createSupabaseBackup() {
        this.plugin.getLogger().info("Supabase backup requested - implement data export to backup tables if needed");
    }

    private void cleanupOldBackups(File backupDir, int keepCount) {
        try {
            File[] backupFiles = backupDir.listFiles((dir, name) -> name.startsWith("villager_data_backup_") && name.endsWith(".db"));
            if (backupFiles == null || backupFiles.length <= keepCount) {
                return;
            }
            Arrays.sort(backupFiles, Comparator.comparingLong(File::lastModified).reversed());
            for (int i = keepCount; i < backupFiles.length; ++i) {
                if (!backupFiles[i].delete()) continue;
                this.plugin.getLogger().fine("Deleted old backup: " + backupFiles[i].getName());
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to cleanup old backups: " + e.getMessage());
        }
    }

    private MigrationResult executeMigration(DatabaseType sourceType, DatabaseType targetType, MigrationOptions options) {
        this.currentProgress.setTotalTables(MIGRATION_TABLES.size());
        this.currentProgress.setStatus("Starting migration...");
        try {
            MigrationValidator validator = new MigrationValidator(this.plugin);
            if (!validator.validateSourceData(sourceType)) {
                return MigrationResult.failure("Source data validation failed");
            }
            for (String tableName : MIGRATION_TABLES) {
                this.currentProgress.setCurrentTable(tableName);
                this.currentProgress.setStatus("Migrating table: " + tableName);
                MigrationResult tableResult = this.migrateTable(tableName, sourceType, targetType, options);
                if (!tableResult.isSuccess()) {
                    return MigrationResult.failure("Failed to migrate table '" + tableName + "': " + tableResult.getErrorMessage());
                }
                this.currentProgress.incrementCompletedTables();
            }
            if (!validator.validateTargetData(targetType)) {
                return MigrationResult.failure("Target data validation failed");
            }
            this.currentProgress.setStatus("Migration completed successfully");
            return MigrationResult.success((int)this.currentProgress.getTotalRecordsMigrated());
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Migration execution failed: " + e.getMessage());
            return MigrationResult.failure("Migration execution failed: " + e.getMessage());
        }
    }

    private MigrationResult migrateTable(String tableName, DatabaseType sourceType, DatabaseType targetType, MigrationOptions options) {
        try {
            this.plugin.getLogger().info("Migrating table: " + tableName);
            List<Map<String, Object>> sourceData = this.getTableData(tableName, sourceType);
            this.currentProgress.setTotalRecords(sourceData.size());
            if (sourceData.isEmpty()) {
                this.plugin.getLogger().info("Table '" + tableName + "' is empty, skipping");
                return MigrationResult.success(0);
            }
            if (options.clearTargetTables) {
                this.clearTargetTable(tableName, targetType);
            }
            int batchSize = options.batchSize;
            int migrated = 0;
            for (int i = 0; i < sourceData.size(); i += batchSize) {
                int endIndex = Math.min(i + batchSize, sourceData.size());
                List<Map<String, Object>> batch = sourceData.subList(i, endIndex);
                MigrationResult batchResult = this.insertBatch(tableName, batch, targetType);
                if (!batchResult.isSuccess()) {
                    return MigrationResult.failure("Failed to insert batch: " + batchResult.getErrorMessage());
                }
                this.currentProgress.setCurrentRecords(migrated += batch.size());
                double progress = (double)migrated / (double)sourceData.size() * 100.0;
                this.currentProgress.setStatus(String.format("Migrating %s: %.1f%% complete (%d/%d records)", tableName, progress, migrated, sourceData.size()));
            }
            this.plugin.getLogger().info("Successfully migrated " + migrated + " records for table: " + tableName);
            return MigrationResult.success(migrated);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to migrate table '" + tableName + "': " + e.getMessage());
            return MigrationResult.failure("Table migration failed: " + e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<Map<String, Object>> getTableData(String tableName, DatabaseType dbType) {
        ArrayList<Map<String, Object>> data = new ArrayList<Map<String, Object>>();
        try (DatabaseProvider provider = this.createProvider(dbType);){
            if (provider instanceof SQLiteDatabaseProvider) {
                List<Map<String, Object>> list = this.getSQLiteTableData(tableName, provider);
                return list;
            }
            if (!(provider instanceof SupabaseDatabaseProvider)) return data;
            List<Map<String, Object>> list = this.getSupabaseTableData(tableName, (SupabaseDatabaseProvider)provider);
            return list;
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to get table data for '" + tableName + "': " + e.getMessage());
        }
        return data;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<Map<String, Object>> getSQLiteTableData(String tableName, DatabaseProvider provider) {
        ArrayList<Map<String, Object>> data = new ArrayList<Map<String, Object>>();
        Connection connection = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            connection = provider.getConnection().get();
            stmt = connection.createStatement();
            rs = stmt.executeQuery("SELECT * FROM " + tableName);
            while (rs.next()) {
                HashMap<String, Object> row = new HashMap<String, Object>();
                ResultSetMetaData metaData = rs.getMetaData();
                int columnCount = metaData.getColumnCount();
                for (int i = 1; i <= columnCount; ++i) {
                    String columnName = metaData.getColumnName(i);
                    Object value = rs.getObject(i);
                    row.put(columnName, value);
                }
                data.add(row);
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to read SQLite data: " + e.getMessage());
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Unexpected error reading SQLite data: " + e.getMessage());
        }
        finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (stmt != null) {
                    stmt.close();
                }
                if (connection != null) {
                    connection.close();
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to close SQLite resources: " + e.getMessage());
            }
        }
        return data;
    }

    private List<Map<String, Object>> getSupabaseTableData(String tableName, SupabaseDatabaseProvider provider) {
        ArrayList<Map<String, Object>> data = new ArrayList<Map<String, Object>>();
        try {
            ((CompletableFuture)provider.executeQuery(tableName, null).thenAccept(resultSet -> {
                try {
                    while (resultSet.next()) {
                        HashMap<String, String> row = new HashMap<String, String>();
                        row.put("key", resultSet.getString("key"));
                        row.put("value", resultSet.getString("value"));
                        data.add(row);
                    }
                }
                catch (Exception e) {
                    this.plugin.getLogger().severe("Failed to read Supabase data: " + e.getMessage());
                }
            })).get();
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to get Supabase data: " + e.getMessage());
        }
        return data;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clearTargetTable(String tableName, DatabaseType targetType) {
        try (DatabaseProvider provider = this.createProvider(targetType);){
            if (provider instanceof SQLiteDatabaseProvider) {
                provider.getConnection().thenAccept(connection -> {
                    try (Statement stmt = connection.createStatement();){
                        stmt.execute("DELETE FROM " + tableName);
                    }
                    catch (Exception e) {
                        this.plugin.getLogger().warning("Failed to clear SQLite table: " + e.getMessage());
                    }
                });
            } else if (provider instanceof SupabaseDatabaseProvider) {
                this.plugin.getLogger().info("Clearing Supabase table: " + tableName);
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to clear target table: " + e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MigrationResult insertBatch(String tableName, List<Map<String, Object>> batch, DatabaseType targetType) {
        try (DatabaseProvider provider = null;){
            provider = this.createProvider(targetType);
            if (provider instanceof SQLiteDatabaseProvider) {
                MigrationResult migrationResult = this.insertSQLiteBatch(tableName, batch, provider);
                return migrationResult;
            }
            if (provider instanceof SupabaseDatabaseProvider) {
                MigrationResult migrationResult = this.insertSupabaseBatch(tableName, batch, (SupabaseDatabaseProvider)provider);
                return migrationResult;
            }
            MigrationResult migrationResult = MigrationResult.failure("Unsupported target database type");
            return migrationResult;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MigrationResult insertSQLiteBatch(String tableName, List<Map<String, Object>> batch, DatabaseProvider provider) {
        Connection connection = null;
        Statement pstmt = null;
        try {
            connection = provider.getConnection().get();
            String sql = this.buildInsertSQL(tableName, batch.get(0).keySet());
            pstmt = connection.prepareStatement(sql);
            for (Map<String, Object> row : batch) {
                int paramIndex = 1;
                for (String column : row.keySet()) {
                    pstmt.setObject(paramIndex++, row.get(column));
                }
                pstmt.executeUpdate();
            }
            MigrationResult migrationResult = MigrationResult.success(batch.size());
            return migrationResult;
        }
        catch (Exception e) {
            MigrationResult migrationResult = MigrationResult.failure("SQLite batch insert failed: " + e.getMessage());
            return migrationResult;
        }
        finally {
            try {
                if (pstmt != null) {
                    pstmt.close();
                }
                if (connection != null) {
                    connection.close();
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to close SQLite resources: " + e.getMessage());
            }
        }
    }

    private MigrationResult insertSupabaseBatch(String tableName, List<Map<String, Object>> batch, SupabaseDatabaseProvider provider) {
        try {
            for (Map<String, Object> row : batch) {
                provider.executeInsert(tableName, row).get();
            }
            return MigrationResult.success(batch.size());
        }
        catch (Exception e) {
            return MigrationResult.failure("Supabase batch insert failed: " + e.getMessage());
        }
    }

    private String buildInsertSQL(String tableName, Set<String> columns) {
        StringBuilder sql = new StringBuilder("INSERT INTO ");
        sql.append(tableName).append(" (");
        String columnList = String.join((CharSequence)", ", columns);
        sql.append(columnList).append(") VALUES (");
        String placeholders = String.join((CharSequence)", ", Collections.nCopies(columns.size(), "?"));
        sql.append(placeholders).append(")");
        return sql.toString();
    }

    private void rollbackMigration(DatabaseType sourceType, DatabaseType targetType) {
        this.plugin.getLogger().info("Rolling back migration...");
        try {
            if (sourceType == DatabaseType.SQLITE) {
                this.rollbackSQLiteMigration();
            } else if (sourceType == DatabaseType.SUPABASE) {
                this.rollbackSupabaseMigration();
            }
            this.plugin.getLogger().info("Migration rollback completed");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Migration rollback failed: " + e.getMessage());
            this.plugin.getLogger().severe("Manual intervention may be required to restore database consistency");
        }
    }

    private void rollbackSQLiteMigration() {
        try {
            File dbFile = new File(this.plugin.getDataFolder(), "villager_data.db");
            File backupDir = new File(this.plugin.getDataFolder(), "backups");
            if (!backupDir.exists()) {
                this.plugin.getLogger().warning("No backup directory found for rollback");
                return;
            }
            File[] backupFiles = backupDir.listFiles((dir, name) -> name.startsWith("villager_data_backup_") && name.endsWith(".db"));
            if (backupFiles == null || backupFiles.length == 0) {
                this.plugin.getLogger().warning("No backup files found for rollback");
                return;
            }
            Arrays.sort(backupFiles, Comparator.comparingLong(File::lastModified).reversed());
            File latestBackup = backupFiles[0];
            this.plugin.getLogger().info("Restoring from backup: " + latestBackup.getName());
            if (this.databaseManager != null && this.databaseManager.getProvider() instanceof SQLiteDatabaseProvider) {
                this.databaseManager.getProvider().close();
            }
            Files.copy(latestBackup.toPath(), dbFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            this.plugin.getLogger().info("SQLite database restored from backup successfully");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to rollback SQLite migration: " + e.getMessage());
            throw new RuntimeException("SQLite rollback failed", e);
        }
    }

    private void rollbackSupabaseMigration() {
        this.plugin.getLogger().info("Supabase rollback requested - implement data restoration from backup tables if needed");
        this.plugin.getLogger().warning("Supabase rollback is not fully implemented - manual intervention may be required");
    }

    public MigrationProgress getCurrentProgress() {
        return this.currentProgress;
    }

    public boolean isMigrationInProgress() {
        return this.migrationInProgress;
    }

    public boolean cancelMigration() {
        if (!this.migrationInProgress) {
            return false;
        }
        this.plugin.getLogger().info("Cancelling migration...");
        this.migrationInProgress = false;
        this.currentProgress.setStatus("Migration cancelled by user");
        return true;
    }

    private DatabaseProvider createProvider(DatabaseType dbType) throws DatabaseException {
        switch (dbType) {
            case SQLITE: {
                String dbFile = String.valueOf(this.plugin.getDataFolder()) + "/villager_data.db";
                this.plugin.getLogger().info("Creating SQLite provider for migration: " + dbFile);
                return new SQLiteDatabaseProvider(dbFile);
            }
            case SUPABASE: {
                if (this.plugin.getConfig().getConfigurationSection("database.supabase") != null) {
                    DatabaseConfig.SupabaseConfig supabaseConfig = new DatabaseConfig(this.plugin.getConfig().getConfigurationSection("database")).getSupabaseConfig();
                    this.plugin.getLogger().info("Creating Supabase provider for migration");
                    return new SupabaseDatabaseProvider(supabaseConfig);
                }
                throw new DatabaseException("Supabase configuration not found in config.yml");
            }
        }
        throw new DatabaseException("Unsupported database type for migration: " + String.valueOf((Object)dbType));
    }
}

