/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.downloadbypass.util;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.texboobcat.downloadbypass.Downloadbypass;
import org.texboobcat.downloadbypass.config.ModDependency;

public class SkippedModsTracker {
    private static final String SKIPPED_FILE_NAME = "skipped-mods.json";
    private static final long DEFAULT_SKIP_DURATION_MS = 604800000L;
    private final Path configDir;
    private final long skipDurationMs;
    private final Map<String, SkipEntry> skippedMods;
    private final Gson gson;

    public SkippedModsTracker(Path configDir) {
        this(configDir, 604800000L);
    }

    public SkippedModsTracker(Path configDir, long skipDurationMs) {
        this.configDir = configDir;
        this.skipDurationMs = skipDurationMs;
        this.skippedMods = new ConcurrentHashMap<String, SkipEntry>();
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.loadSkipped();
    }

    public boolean shouldSkip(ModDependency dep) {
        if (dep == null || dep.getName() == null) {
            return false;
        }
        String key = this.buildKey(dep);
        SkipEntry entry = this.skippedMods.get(key);
        if (entry == null) {
            return false;
        }
        long now = System.currentTimeMillis();
        long elapsed = now - entry.timestamp;
        if (elapsed > this.skipDurationMs) {
            this.skippedMods.remove(key);
            this.saveSkipped();
            Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Skip expired for {}, will prompt again", (Object)dep.getName());
            return false;
        }
        long remainingDays = (this.skipDurationMs - elapsed) / 86400000L;
        Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Skipping prompt for {} (declined {} days ago, {} days remaining)", (Object)dep.getName(), (Object)(elapsed / 86400000L), (Object)remainingDays);
        return true;
    }

    public void recordSkip(ModDependency dep) {
        if (dep == null || dep.getName() == null) {
            return;
        }
        String key = this.buildKey(dep);
        SkipEntry entry = new SkipEntry();
        entry.modName = dep.getName();
        entry.timestamp = System.currentTimeMillis();
        entry.version = dep.getVersion();
        this.skippedMods.put(key, entry);
        this.saveSkipped();
        Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Recorded skip for {} (will not prompt for {} days)", (Object)dep.getName(), (Object)(this.skipDurationMs / 86400000L));
    }

    public void clearSkip(ModDependency dep) {
        if (dep == null || dep.getName() == null) {
            return;
        }
        String key = this.buildKey(dep);
        if (this.skippedMods.remove(key) != null) {
            this.saveSkipped();
            Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Cleared skip record for {}", (Object)dep.getName());
        }
    }

    public void clearAll() {
        this.skippedMods.clear();
        this.saveSkipped();
        Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Cleared all skip records");
    }

    public void cleanup() {
        long now = System.currentTimeMillis();
        int removed = 0;
        for (Map.Entry<String, SkipEntry> entry : this.skippedMods.entrySet()) {
            long elapsed = now - entry.getValue().timestamp;
            if (elapsed <= this.skipDurationMs) continue;
            this.skippedMods.remove(entry.getKey());
            ++removed;
        }
        if (removed > 0) {
            this.saveSkipped();
            Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Cleaned up {} expired skip records", (Object)removed);
        }
    }

    public SkipStats getStats() {
        int total = this.skippedMods.size();
        long now = System.currentTimeMillis();
        int expired = 0;
        for (SkipEntry entry : this.skippedMods.values()) {
            long elapsed = now - entry.timestamp;
            if (elapsed <= this.skipDurationMs) continue;
            ++expired;
        }
        return new SkipStats(total, total - expired, expired);
    }

    private String buildKey(ModDependency dep) {
        String version = dep.getVersion() != null ? dep.getVersion() : "unknown";
        return dep.getName().toLowerCase() + ":" + version;
    }

    private void loadSkipped() {
        try {
            Path skipFile;
            if (!Files.exists(this.configDir, new LinkOption[0])) {
                Files.createDirectories(this.configDir, new FileAttribute[0]);
            }
            if (!Files.exists(skipFile = this.configDir.resolve(SKIPPED_FILE_NAME), new LinkOption[0])) {
                return;
            }
            String json = Files.readString(skipFile);
            Map loaded = (Map)this.gson.fromJson(json, new TypeToken<Map<String, SkipEntry>>(){}.getType());
            if (loaded != null) {
                this.skippedMods.putAll(loaded);
                Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Loaded {} skipped mod records from disk", (Object)loaded.size());
            }
        }
        catch (IOException e) {
            Downloadbypass.LOGGER.warn("[EXTERNAL-MODS] Failed to load skipped mods: {}", (Object)e.getMessage());
        }
    }

    private void saveSkipped() {
        try {
            if (!Files.exists(this.configDir, new LinkOption[0])) {
                Files.createDirectories(this.configDir, new FileAttribute[0]);
            }
            Path skipFile = this.configDir.resolve(SKIPPED_FILE_NAME);
            String json = this.gson.toJson(this.skippedMods);
            Files.writeString(skipFile, (CharSequence)json, new OpenOption[0]);
            Downloadbypass.LOGGER.debug("[EXTERNAL-MODS] Saved {} skipped mod records to disk", (Object)this.skippedMods.size());
        }
        catch (IOException e) {
            Downloadbypass.LOGGER.warn("[EXTERNAL-MODS] Failed to save skipped mods: {}", (Object)e.getMessage());
        }
    }

    private static class SkipEntry {
        String modName;
        long timestamp;
        String version;

        private SkipEntry() {
        }
    }

    public static class SkipStats {
        public final int total;
        public final int active;
        public final int expired;

        public SkipStats(int total, int active, int expired) {
            this.total = total;
            this.active = active;
            this.expired = expired;
        }

        public String toString() {
            return String.format("Skip Stats: %d total, %d active, %d expired", this.total, this.active, this.expired);
        }
    }
}

